/* $Id: base.c,v 1.5 2001/05/24 14:53:58 malekith Exp $ */

#include <ytk/ytk.h>
#include <ytk/int/object.h>
#include <yw/util.h>
#include <yw/hash.h>

#include <string.h>

const char *ytk_class_of(YtkObject *o)
{
	return o->klass ? o->klass->name : NULL;
}

/* Next/prev/active stuff */
YtkObject *ytk_object_next(YtkObject *o)
{
	return o->next;
}

YtkObject *ytk_object_children(YtkObject *o)
{
	return o->children;
}

YtkObject *ytk_object_focused(YtkObject *o)
{
	return o->active;
}

int ytk_object_has_focus(YtkObject *o)
{
	if (o->parent == NULL)
		return 1;
	else
		return o->parent->active == o && 
		       ytk_object_has_focus(o->parent);
}

void ytk_object_link_child(YtkObject *o, YtkObject *child)
{
	if (o->last_child) {
		o->last_child->next = child;
		o->last_child = child;
	} else {
		o->last_child = o->children = child;
	}

	child->parent = o;
	ytk_inc_ref(child);

	ytk_send_void_msg_va(child, "p" "new_parent", o, YW_END);

	if (o->active == NULL)
		ytk_object_focus(o, child);
}

YtkObject *ytk_object_parent(YtkObject *o)
{
	return o->parent;
}

YwWindow *ytk_object_window(YtkObject *obj)
{
	while (obj) {
		if (obj->window)
			return obj->window;
		obj = obj->parent;
	}

	return NULL;
}

void ytk_bind_object_to_window(YtkObject *obj, YwWindow *window)
{
	obj->window = window;
	if (obj->context == NULL)
		ytk_bind_to_context(NULL, obj);
}

static YwHash *classes;

/* this is ready to export, but I guess there won't be
 * need to do this
 */
static YtkClass *ytk_find_class(const char *name)
{
	if (classes == NULL)
		return NULL;
	return yw_hash_find(classes, name);
}

YtkObject *ytk_new_object(const char *class_name, 
			  YtkCallback callback,
			  YtkFree finalizer,
			  void *data)
{
	YtkObject *r;
	YtkClass *c = NULL;

	if (class_name && 
	    (c = ytk_find_class(class_name)) == NULL) {
		ytk_fatal("ytk_new_object: %s: class not registred", 
			  class_name);
		return NULL;
	}

	r = YW_NEW_0(YtkObject);
	r->data = data;
	r->refcnt = 1;
	r->klass = c;
	r->callback = callback;
	r->finalizer = finalizer;

	if (c)
		c->refcnt++;

	return r;
}

void ytk_inc_ref(YtkObject *obj)
{
	yw_assert(obj->refcnt);
	obj->refcnt++;
}

static void destroy_object(YtkObject *obj)
{
	YtkObject *child;
	
	/* first unlink children... */
	while (obj->children) {
		child = obj->children;
		obj->children = obj->children->next;

		child->parent = NULL;
		ytk_send_void_msg_va(child, "k", "lost_parent", YW_END);
		ytk_dec_ref(child);
	}
	
	if (obj->finalizer)
		obj->finalizer(obj->data);
	
	if (obj->klass)
		obj->klass->refcnt--;
	
	if (obj->responder)
		ytk_dec_ref(obj->responder);

	obj->refcnt = 0;
	
	if (obj->context)
		/* when it's called on refcnt==0 obj,
		 * it unbinds it. */
		ytk_bind_to_context(NULL, obj);
		
	/* be as destructive as we can... */
	memset(obj, 0, sizeof(YtkObject));
	yw_free(obj);
}

void ytk_dec_ref(YtkObject *obj)
{
	yw_assert(obj->refcnt);
	if (obj->refcnt == 1)
		destroy_object(obj);
	else
		obj->refcnt--;
}

void ytk_unregister_class(const char *name)
{
	YtkClass *c;
	
	c = yw_hash_unlink(classes, name);
	yw_assert(c && c->refcnt == 0);
	yw_free(c->name);
	yw_free(c);
}

void ytk_register_class(const char *name, 
			YtkCreator creator, void *creator_data)
{
	YtkClass *c;
	
	if (classes == NULL)
		classes = yw_hash_new();
		
	/* cannot register the same class twice */
	yw_assert(yw_hash_find(classes, name) == NULL);
	
	c = YW_NEW_0(YtkClass);
	c->name = yw_strdup(name);
	c->creator = creator;
	c->creator_data = creator_data;

	yw_hash_add(classes, c);
}

YtkObject *ytk_new(const char *name, YwPacket *tags)
{
	YtkClass *c;

	c = ytk_find_class(name);

	if (c == NULL) {
		ytk_fatal("ytk_new: %s: class not found", name);
		return NULL;
	}
	
	return c->creator(c->creator_data, tags);
}

YtkObject *ytk_new_ap(const char *name, va_list ap)
{
	YtkObject *o;
	YwPacket *tags;
	
	tags = yw_packet_make_v(yw_taglist_packet, ap);
	
	o = ytk_new(name, tags);

	yw_packet_free(tags);

	return o;
}

YtkObject *ytk_new_va(const char *name, ...)
{
	YtkObject *o;
	va_list ap;

	va_start(ap, name);
	o = ytk_new_ap(name, ap);
	va_end(ap);

	return o;
}
