#include <ypp/Packet.h>
#include <ypp/TagList.h>

namespace Ypp {

BinData::BinData(const void *ptr, int len)
	: ptr(NULL)
{
	copy(ptr, len);
}

BinData::BinData(const BinData &r)
	: ptr(NULL) 
{
	copy(r.ptr, r.len); 
}

const BinData &BinData::operator =(const BinData &r)
{
	if (this != &r)
		copy(r.ptr, r.len);
	return *this;
}

void BinData::copy(const void *sptr, int slen)
{
	yw_free(ptr);
	ptr = yw_malloc_0(slen + 1);
	memcpy(ptr, sptr, len = slen);
}

void Packet::append(const Word &w)
{
	yw_packet_append_word(data, w.ref());
}

Word Packet::word(int n) const
{
	const YwWord *w;

	w = bare_word(n);
	if (w == NULL)
		throw WordNotFoundException();
	
	return Word(w);
}

Packet::Packet(const YwPacket *pkt)
{
	if (pkt)
		data = yw_packet_copy_of(pkt);
	else
		data = yw_packet_new(yw_taglist_packet);
}

Packet::Packet(const YwWord *w, int len, YwPacketType type)
	: data(yw_packet_new(type))
{
	yw_packet_append_words(data, w, len);
}

Packet::Packet()
	: data(yw_packet_new(yw_taglist_packet))
{
}

Packet::Packet(YwPacketType t, ...)
{
	va_list ap;
	
	va_start(ap, t);
	data = yw_packet_make_v(t, ap);
	va_end(ap);
}

Packet::Packet(const Packet &r)
	: data(yw_packet_copy_of(r.ref()))
{
}

const Packet &Packet::operator =(const Packet &r)
{
	if (&r != this) {
		yw_packet_free(data);
		data = yw_packet_copy_of(r.ref());
	}

	return *this;
}

const String &String::operator =(const String &r)
{
	if (&r != this) {
		yw_string_free(&data);
		yw_string_assign_string(&data, r.ref());
	}

	return *this;
}

AsciiString String::get_cstring(const char *enc, int *len)
{
	char *p = NULL;
	
	yw_string_get_cstring(&data, enc, &p, len);

	if (p == NULL)
		throw ConversionException();
	
	return AsciiString(0, p);
}

TagList::TagList(const Packet &pkt, int skip)
	: copy(pkt.bare_word(skip))
{
	head = cur = copy.bare_word(0);
}

bool TagList::eot()
{
	return (cur == NULL);
}

void TagList::next()
{
	if (eot())
		throw NoNextTagException();
	
	cur = yw_word_next(yw_word_next((YwWord*)cur));
}

void TagList::reset()
{
	cur = head;
}

AsciiString TagList::name()
{
	if (eot())
		throw NoNextTagException();
	return Word(cur).keyword();
}

Word TagList::value()
{
	if (eot())
		throw NoNextTagException();
		
	return Word(yw_word_next((YwWord *)cur));
}

Word TagList::find(const AsciiString &name) const
{
	const YwWord *w;

	w = yw_tag_find((YwWord *)head, name.ref());
	
	if (w == NULL)
		throw TagNotFoundException();
	
	return Word(w);
}

Word TagList::find(const AsciiString &name, const Word &def) const
{
	const YwWord *w;

	w = yw_tag_find((YwWord *)head, name.ref());
	
	if (w == NULL)
		return def;
	
	return Word(w);
}

bool TagList::present(const AsciiString &name) const
{
	return yw_tag_find((YwWord *)head, name.ref()) != NULL;
}


int32_t Word::integer() const
{
	int res;

	if (yw_word_fetch_int(data, &res))
		throw WordTypeException();
	
	return (int32_t)res;
}

AsciiString Word::keyword() const
{
	const char *res;

	if (yw_word_fetch_keyword(data, &res))
		throw WordTypeException();
	
	return res;
}

String Word::string() const
{
	YwString *res;
	
	if (yw_word_fetch_string(data, &res))
		throw WordTypeException();
	
	return res;
}

void *Word::ptr() const
{
	void *res;

	if (yw_word_fetch_ptr(data, &res))
		throw WordTypeException();
	
	return res;
}

BinData Word::bindata() const
{
	void *ptr;
	int len;
	
	if (yw_word_fetch_bindata(data, &ptr, &len))
		throw WordTypeException();
	
	return BinData(ptr, len);
}

int Word::mapped_keyword(const YwMapEntry e[], YwMapCache **c) const
{
	int res = 0;

	if (yw_word_fetch_mapped_keyword(data, e, c, &res))
		throw WordTypeException();
	
	return res;
}

const Word &Word::operator =(const Word &r)
{
	if (&r != this) {
		yw_word_free(data);
		data = yw_word_copy_of(r.ref());
	}

	return *this;
}

uint32_t String::operator[] (int p) const
{
	if (p < 0 || p >= length())
		throw StringIdxException();
	
	return ref()->chars[p];
}

uint32_t &String::operator[] (int p)
{
	if (p < 0 || p >= length())
		throw StringIdxException();
	
	return data.chars[p];
}

} // namespace Ypp
