#include <ypp/Widget.h>

namespace Ypp {

// not real infinity (INT_MAX), so we won't overflow.
const int Size::infinity = 100000;

bool Widget::no_buf()
{
	YwWindow *w;
	
	w = get_self().window();
	if (w == NULL)
		return true;
	
	if ((buf.buf = yw_window_buffer(w)) == NULL)
		return true; 
	
	return false;
}

void Widget::handle_event(const AsciiString &name, const Word &val)
{
	get_self().send_event_up(name, val);
}

void Widget::redraw()
{
	if (no_buf())
		return;
	
	draw();

	buf.sync();
}

namespace {
	enum {
		op_x,
		op_y,
		op_width,
		op_height,
		op_max_width,
		op_max_height,
		op_min_width,
		op_min_height,
		op_oops
	};
	YwMapEntry op_e[] = {
		// {class: YppWidget}
		// {super: YppClass}
		// {abstract}
		// An abstract representation of some widget. It includes
		// several properties present in all other widgets.
		// {int} X cooridanate within upper-level (parent) widget.
		{ "x", op_x },
		// {int} Y cooridanate within parent widget.
		{ "y", op_y },
		// {int} Width of widget. When max_width or min_width
		// is also set, this is trated more like some kind of
		// hint.
		{ "width", op_width },
		// {int} Height of widget.
		{ "height", op_height },
		// {int} Maximal width, widget can take.
		{ "max_width", op_max_width },
		// {int} Maximal height, widget can take.
		{ "max_height", op_max_height },
		// {int} Minimal width, widget can take.
		{ "min_width", op_min_width },
		// {int} Minimal height, widget can take.
		{ "min_height", op_min_height },
		// {class: end}
		{ NULL, op_oops },
	};
	YwMapCache *op_cache;
}

Word Widget::get_op(const AsciiString &op)
{
	switch (op.map(op_e, &op_cache)) {
	case op_x:
		return pos_x;
	case op_y:
		return pos_y;
	case op_width:
		return width;
	case op_height:
		return height;
	case op_max_width:
		return preset.max.w == -1 ? width : preset.max.w;
	case op_max_height:
		return preset.max.h == -1 ? height : preset.max.h;
	case op_min_width:
		return preset.min.w == -1 ? width : preset.min.w;
	case op_min_height:
		return preset.min.h == -1 ? height : preset.min.h;
	default:
		throw OpException();
	}
}

void Widget::set_op(const AsciiString &op, const Word &v)
{
	switch (op.map(op_e, &op_cache)) {
	case op_width:
		preset.mid.w = v.integer();
		break;
	case op_height:
		preset.mid.h = v.integer();
		break;
	case op_max_width:
		preset.max.w = v.integer();
		break;
	case op_max_height:
		preset.max.h = v.integer();
		break;
	case op_min_width:
		preset.min.w = v.integer();
		break;
	case op_min_height:
		preset.min.h = v.integer();
		break;
	default:
		throw OpException();
	}

	//resize();
}

void Widget::draw()
{
}

void Widget::lost_parent()
{
	buf.buf = NULL;
}

void Widget::new_parent()
{
}

VarSize Widget::get_sizes()
{
	return preset;
}

void Widget::set_position(const TagList &t)
{
	pos_x = t.find("pos_x", pos_x).integer();
	pos_y = t.find("pos_y", pos_y).integer();
	width = t.find("width", width).integer();
	height = t.find("height", height).integer();
}

bool Widget::can_get_focus()
{
	return false;
}

Packet Widget::dispatch(const Packet &pkt)
{
	enum {
		do_get_op,
		do_set_op,
		do_draw,
		do_lost_parent,
		do_new_parent,
		do_get_sizes,
		do_set_position,
		do_can_get_focus,
		do_kbd,
		do_event,
		do_oops
	};
	YwMapEntry e[] = {
		{ "get_op", do_get_op },
		{ "set_op", do_set_op },
		{ "draw", do_draw },
		{ "lost_parent", do_lost_parent },
		{ "new_parent", do_new_parent },
		{ "get_sizes", do_get_sizes },
		{ "set_position", do_set_position },
		{ "can_get_focus", do_can_get_focus },
		{ "kbd", do_kbd },
		{ "event", do_event },
		{ NULL, do_oops }
	};
	static YwMapCache *cache;
	Packet ret(yw_reply_packet, YW_END);

	switch (pkt.word(0).mapped_keyword(e, &cache)) {
	case do_get_op:
		ret.append_keyword("op_value");
		ret.append(get_op(pkt.word(1).keyword()));
		break;
	case do_set_op:
		set_op(pkt.word(1).keyword(), pkt.word(2));
		ret.append_keyword("ok");
		break;
	case do_draw:
		draw();
		break;
	case do_lost_parent:
		lost_parent();
		break;
	case do_new_parent:
		new_parent();
		break;
	case do_get_sizes:
	{
		VarSize v;
		v = get_sizes();
		
		if (v.max.w == -1)
			v.max.w = v.mid.w;
		if (v.max.h == -1)
			v.max.h = v.mid.h;
		if (v.min.w == -1)
			v.min.w = v.mid.w;
		if (v.min.h == -1)
			v.min.h = v.mid.h;
			
		ret = Packet(yw_reply_packet,
				"k", "sizes",
				"i" "width", v.mid.w,
				"i" "height", v.mid.h,
				"i" "max_width", v.max.w,
				"i" "max_height", v.max.h,
				"i" "min_width", v.min.w,
				"i" "min_height", v.min.h,
				YW_END);
		break;
	}
	case do_set_position:
		set_position(TagList(pkt, 1));
		break;
	case do_can_get_focus:
		ret.append_keyword("r_can_get_focus");
		ret.append_keyword(can_get_focus() ? "yes" : "no");
		break;
	case do_kbd:
		ret.append_keyword("kbd_handled");
		ret.append_keyword(handle_key(pkt) ? "yes" : "no");
		break;
	case do_event:
		handle_event(pkt.word(1).keyword(), pkt.word(2));
		ret.append_keyword("ok");
		break;
	default:
		// as Class doesn't understand anything
		// we'll get MsgNotUnderstoodException here.
		return Class::dispatch(pkt);
	}

	return ret;
}

bool Widget::handle_norm_key(const String &, int)
{
	return false;
}

bool Widget::handle_spec_key(int, int)
{
	return false;
}

bool Widget::handle_key(const Packet &pkt)
{
	YwString *str;
	int flags, key;
	
	if (yw_key_normal((YwPacket*)pkt.ref(), &str, &flags) == 0)
		return handle_norm_key(String(str), flags);
	else if (yw_key_special((YwPacket*)pkt.ref(), &key, &flags) == 0)
		return handle_spec_key(key, flags);
	
	return false;
}

void Widget::set_named_ops(const char **p, const TagList &t)
{
	while (*p) {
		if (t.present(*p))
			set_op(*p, t.find(*p));
		p++;
	}
}

void Widget::init(const TagList &t)
{
	YtkObject *resp = (YtkObject*)t.find("responder", Word((void*)0)).ptr();
	if (resp) {
		get_self().set_responder(resp);
		ytk_dec_ref(resp);
	}
	
	preset.mid.w = t.find("width", preset.mid.w).integer();
	preset.mid.h = t.find("height", preset.mid.h).integer();
	
	preset.min.w = t.find("min_width", preset.mid.w).integer();
	preset.min.h = t.find("min_height", preset.mid.h).integer();

	// when no min_* default to fixed size, otherwise,
	// default to no-upper-limit
	preset.max.w = t.find("max_width", t.present("min_width") ? 
		Size::infinity : preset.mid.w).integer();
	preset.max.h = t.find("max_height", t.present("min_height") ?
		Size::infinity : preset.mid.h).integer();

	pos_x = pos_y = 0;
	width = preset.mid.w;
	height = preset.mid.h;
}

Widget::Widget() : preset(Size(-1, -1))
{
	pos_x = pos_y = 0;
	width = preset.mid.w;
	height = preset.mid.h;
}

}
