/* $Id: Group.cc,v 1.3 2001/06/02 11:18:29 malekith Exp $ */

#include <ypp/Group.h>

// Register Group

namespace Ypp {

namespace {
	enum {
		op_layout,
		op_spacing,
		op_oops
	};
	YwMapEntry op_e[] = {
		// {class: YppGroup}
		// {super: YppWidget}
		// This class represent a bit abstract grouping
		// widget. It isn't abstract in a formal way, though.
		// Instances can be created.
		// Additonaly YppGroup class, and its subclasses
		// support initialzation with "child" attribute.
		// It's argument is pointer to child, wich we should
		// link.
		// {keyword} Way of placing elements in the group.
		// Can be "horizontal" or "vertical".
		{ "layout", op_layout },
		// {int} Number of blank characters deviding
		// elements inside group.
		{ "spacing", op_spacing },
		// {class: end}
		{ NULL, op_oops }
	};
	YwMapCache *op_cache;
}

Word Group::get_op(const AsciiString &name)
{
	switch (name.map(op_e, &op_cache)) {
	case op_layout:
		switch (layout) {
		case horizontal:
			return "horizontal";
		case vertical:
			return "vertical";
		}
		break;
	case op_spacing:
		return spacing;
	default:
		return Widget::get_op(name);
	}

	throw LogicException();
}

void Group::set_op(const AsciiString &name, const Word &w)
{
	switch (name.map(op_e, &op_cache)) {
	case op_layout:
		layout = w.keyword() == "vertical" ?
			vertical : horizontal;
		break;
	case op_spacing:
		spacing = w.integer();
		break;
	default:
		Widget::set_op(name, w);
		break;
	}
}

Group::~Group()
{
	clear_cache();
}

void Group::init(const TagList &tt)
{
	static const char *ops[] = {
		"layout",
		"spacing",
		NULL
	};
	TagList t = tt;
	
	Widget::init(t);
	set_named_ops(ops, t);
	
	// now we need to link all children
	while (!t.eot()) {
		if (t.name() == "child") {
			YtkObject *o;

			o = (YtkObject*)t.value().ptr();
			get_self().link_child(Object(o));
			ytk_dec_ref(o);
		}
		t.next();
	}
}

Group::Group()
{
	layout = horizontal;
	spacing = 0;
	cache = NULL;
}

void Group::broadcast(const Packet &pkt)
{
	Object s = get_self();
	
	if (!s.has_children())
		return;
		
	Object o = s.children();
	for (;;) {
		o.send_msg(pkt);
		if (o.has_next())
			o = o.next();
		else
			break;
	}
}

void Group::draw()
{
	if (no_buf())
		return;
	
	Packet pkt(yw_void_call_packet, "k", "draw", YW_END);
	broadcast(pkt);

	if (get_self().has_focused())
		get_self().focused().send_msg(pkt);
}

void Group::lost_parent()
{
	broadcast(Packet(yw_void_call_packet, "k", "lost_parent", YW_END));

	Widget::lost_parent();
}

void Group::new_parent()
{
	broadcast(Packet(yw_void_call_packet, 
			"p" "new_parent", get_self().ref(), 
			YW_END));

	Widget::new_parent();
}

VarSize Group::process_sizes(VarSize &vs, const TagList &t, int spc)
{
	VarSize s(Size(0,0));

	s.mid.w = t.find("width", 0).integer();
	s.mid.h = t.find("height", 0).integer();
	s.min.w = t.find("min_width", s.mid.w).integer();
	s.min.h = t.find("min_height", s.mid.h).integer();
	s.max.w = t.find("max_width", Size::infinity).integer();
	s.max.h = t.find("max_height", Size::infinity).integer();

	if (s.mid.w > s.max.w)
		s.mid.w = s.max.w;
	if (s.mid.h > s.max.h)
		s.mid.h = s.max.h;
	if (s.mid.w < s.min.w)
		s.mid.w = s.min.w;
	if (s.mid.h < s.min.h)
		s.mid.h = s.min.h;
	
	if (layout == horizontal) {
		vs.min.w += s.min.w + spc;
		vs.mid.w += s.mid.w + spc;
		if (vs.max.w < Size::infinity)
			vs.max.w += s.max.w + spc;

		if (s.min.h > vs.min.h)
			vs.min.h = s.min.h;
		if (s.max.h < vs.max.h)
			vs.max.h = s.max.h;
	} else {
		vs.min.h += s.min.h + spc;
		vs.mid.h += s.mid.h + spc;
		if (vs.max.h < Size::infinity)
			vs.max.h += s.max.h + spc;

		if (s.min.w > vs.min.w)
			vs.min.w = s.min.w;
		if (s.max.w < vs.max.w)
			vs.max.w = s.max.w;
	}

	return s;
}

void Group::clear_cache()
{
	CacheEntry *c;
	
	while (cache) {
		c = cache;
		cache = cache->next;
		delete c;
	}
}

VarSize Group::get_sizes()
{
	Object s = get_self();
	VarSize vs(Size(0, 0));
	
	if (!s.has_children())
		return Widget::get_sizes();
		
	Object o = s.children();
	
	clear_cache();
	
	for (;;) {
		Packet pkt = o.send_msg(Packet(yw_call_packet, 
					"k", "get_sizes",
					YW_END));
		VarSize s(Size(0, 0));
		
		if (AsciiString(pkt.word(0).keyword()) == "sizes")
			s = process_sizes(vs, TagList(pkt, 1), 
					  o.has_next() ? spacing : 0);
		
		cache = new CacheEntry(s, cache);
		
		if (o.has_next())
			o = o.next();
		else
			break;
	}

	// restrict to preset
	if (preset.min.w != -1 && vs.min.w < preset.min.w)
		vs.min.w = preset.min.w;
	if (preset.min.h != -1 && vs.min.h < preset.min.h)
		vs.min.h = preset.min.h;
		
	if (vs.max.w > preset.max.w && preset.max.w > vs.min.w)
		vs.max.w = preset.max.w;
	if (vs.max.h > preset.max.h && preset.max.h > vs.min.h)
		vs.max.h = preset.max.h;
	
	if (vs.max.w < vs.min.w)
		vs.max.w = vs.min.w;
	if (vs.max.h < vs.min.h)
		vs.max.h = vs.min.h;

	// don't excess infinity :^)
	if (vs.max.w > Size::infinity)
		vs.max.w = Size::infinity;
	if (vs.max.h > Size::infinity)
		vs.max.h = Size::infinity;
	
	// and put proposed between min and max
	if (vs.mid.w < vs.min.w)
		vs.mid.w = vs.min.w;
	if (vs.mid.w > vs.max.w)
		vs.mid.w = vs.max.w;
	if (vs.mid.h < vs.min.h)
		vs.mid.h = vs.min.h;
	if (vs.mid.h > vs.max.h)
		vs.mid.h = vs.max.h;
		
	// relink cache in the right order
	{
		CacheEntry *p = NULL, *tmp;
		while (cache) {
			tmp = cache;
			cache = cache->next;
			tmp->next = p;
			p = tmp;
		}
		cache = p;
	}
	
	// finally out...
	return vs;
}

// adjust elements in *cache, proportianally, based on
// proposed dimension (but don't excess [min,max]).
void Group::adjust_positions(int sum)
{
	int dim, elem, last_dim;
	CacheEntry *p;
	
	if (layout == horizontal)
		dim = width;
	else
		dim = height;
		
	if (dim == sum)
		return;
		
	// need to adjust
	for (p = cache; p; p = p->next) {
		elem = layout == horizontal ? 
			p->size.mid.w : p->size.mid.h;
		if (elem == 0)
			elem = 1;
		else
			elem = elem * dim / sum;
		if (layout == horizontal) {
			if (elem < p->size.min.w)
				elem = p->size.min.w;
			if (elem > p->size.max.w)
				elem = p->size.max.w;
			p->size.mid.w = elem;
		} else {
			if (elem < p->size.min.h)
				elem = p->size.min.h;
			if (elem > p->size.max.h)
				elem = p->size.max.h;
			p->size.mid.h = elem;
		}
			
		dim -= elem;
	}
	
	// fine tune...
	last_dim = 0;
	while (dim != last_dim) {
		last_dim = dim;
		for (p = cache; p && dim; p = p->next)
			if (dim > 0) {
				if (layout == horizontal) { 
					if (p->size.mid.w < p->size.max.w) {
						p->size.mid.w++;
						dim--;
					}
				} else {
					if (p->size.mid.h < p->size.max.h) {
						p->size.mid.h++;
						dim--;
					}
				}
			} else {
				if (layout == horizontal) {
					if (p->size.mid.w > p->size.min.w) {
						p->size.mid.w--;
						dim++;
					}
				} else {
					if (p->size.mid.h > p->size.min.h) {
						p->size.mid.h--;
						dim++;
					}
				}
			}
	}
}

void Group::set_position(const TagList &t)
{
	CacheEntry *p;
	int sum = 0;
	int spc = 0;
	
	Widget::set_position(t);

	// in first round cut down one dimension,
	// and sum the second.
	for (p = cache; p; p = p->next) {
		if (layout == horizontal) {
			p->size.mid.h = height;
			sum += p->size.mid.w;
		} else {
			p->size.mid.w = width;
			sum += p->size.mid.h;
		}
		if (p->next)
			spc += spacing;
	}
	
	// then adjust summed.
	adjust_positions(sum - spc);

	// finally we should have got right dimensions -- broadcast
	// message to children
	
	Object s = get_self();
	
	if (s.has_children()) {
		Object o = s.children();
		p = cache;
		
		int x = pos_x;
		int y = pos_y;
		
		for (;;) {
			o.send_msg(Packet(yw_void_call_packet,
					"k", "set_position",
					"i" "pos_x", x,
					"i" "pos_y", y,
					"i" "width", p->size.mid.w,
					"i" "height", p->size.mid.h,
					YW_END));
			
			if (layout == horizontal)
				x += p->size.mid.w + spacing;
			else
				y += p->size.mid.h + spacing;
			
			p = p->next;
			if (p && o.has_next())
				o = o.next();
			else
				break;
		}
	}
	
	clear_cache();
}

namespace {

void redraw_win(Object o)
{
	if (o.has_parent())
		redraw_win(o.parent());
	else
		o.send_msg(Packet(yw_void_call_packet, 
					"k", "draw", 
				  YW_END));
}

}

// FIXME: we cannot just redraw self here (Widget::redraw()),
// becouse focuse could have left us...
bool Group::handle_norm_key(const String &s, int flags)
{
	if (s == "\t" || s == "\n") {
		if (flags & (yw_key_meta | yw_key_ctrl))
			get_self().focus_prev();
		else
			get_self().focus_next();
		redraw_win(get_self());
		return true;
	}

	return false;
}

bool Group::handle_spec_key(int key, int /*flags*/)
{
	switch (key) {
	case yw_key_up:
	case yw_key_left:
		get_self().focus_prev();
		redraw_win(get_self());
		return true;
	case yw_key_down:
	case yw_key_right:
		get_self().focus_next();
		redraw_win(get_self());
		return true;
	}

	return false;
}

bool Group::can_get_focus()
{
	return get_self().has_focused();
}

} // namespace Ypp
