/* $Id: zlib.c,v 1.4 2001/04/26 15:00:08 malekith Exp $ */

#include <yw/util.h>
#include <yw/int/conn.h>
#include <yw/int/calls.h>

#include <stdio.h>

#ifdef YW_HAS_ZLIB

#include <zlib.h>

struct zlib_state {
	z_stream read_s;
	void *read_buf;
	int read_buf_size;

	z_stream write_s;
	void *write_buf;
	int write_buf_size;

	int (*read)(YwConnection *conn, void *buf, int n);
	int (*can_read)(YwConnection *conn, int ms);
	int (*write)(YwConnection *conn, const void *buf, int n);
};

static voidpf zalloc(voidpf opaque, uInt items, uInt size)
{
	(void)opaque;
	return yw_malloc(items * size);
}

static void zfree(voidpf opaque, voidpf address)
{
	(void)opaque;
	yw_free(address);
}

static int zlib_can_read(YwConnection *conn, int ms)
{
	struct zlib_state *ctx = conn->zlib_state;

	return (ctx->read_s.avail_in == 0) ? 
		ctx->can_read(conn, ms) : ctx->read_s.avail_in;
}

static int zlib_read(YwConnection *conn, void *buf, int size)
{
	int r;
	struct zlib_state *ctx;

	if (size == 0)
		return 0;
		
	ctx = conn->zlib_state;

	if (ctx->read_s.avail_in == 0) {
		if (size > ctx->read_buf_size)
			ctx->read_buf = yw_realloc(ctx->read_buf, 
				ctx->read_buf_size = 
					size > 1024 ? size * 2 : 2048);
		r = ctx->read(conn, ctx->read_buf, ctx->read_buf_size);
		if (r <= 0)
			return r;
		ctx->read_s.avail_in = r;
		ctx->read_s.next_in = ctx->read_buf;
	}

	ctx->read_s.avail_out = size;
	ctx->read_s.next_out = buf;

	r = inflate(&ctx->read_s, Z_SYNC_FLUSH);

	if (r != Z_OK) 
		yw_int_return_err(conn, YW_ERR_ZLIB);

	/* note that this can legally return 0, even not at EOF. */
	return (size - ctx->read_s.avail_out);
}

static int zlib_write(YwConnection *conn, const void *buf, int size)
{
	int r;
	struct zlib_state *ctx;

	if (size == 0)
		return 0;
		
	ctx = conn->zlib_state;
	
	if (size + size / 10 + 16 > ctx->write_buf_size)
		ctx->write_buf = yw_realloc(ctx->write_buf, 
				ctx->write_buf_size = size * 2 + 16);

	ctx->write_s.next_in = (void*)buf;
	ctx->write_s.avail_in = size;
	ctx->write_s.next_out = ctx->write_buf;
	ctx->write_s.avail_out = ctx->write_buf_size;

	r = deflate(&ctx->write_s, Z_SYNC_FLUSH);

	yw_assert(r == Z_OK);
	yw_assert(ctx->write_s.avail_in == 0);
	
	r = ctx->write(conn, ctx->write_buf, 
		       ctx->write_buf_size - ctx->write_s.avail_out);

	return r;
}

void yw_int_conn_zlib_free(YwConnection *conn)
{
	struct zlib_state *ctx;
	
	if ((ctx = conn->zlib_state)) {
		yw_free(ctx->read_buf);
		yw_free(ctx->write_buf);
		deflateEnd(&ctx->write_s);
		inflateEnd(&ctx->read_s);
		yw_free(ctx);
	}
}

void yw_int_enable_zlib(YwConnection *conn)
{
	struct zlib_state *ctx;
	int r;

	ctx = YW_NEW_0(struct zlib_state);

	ctx->read = conn->read;
	ctx->write = conn->write;
	ctx->can_read = conn->can_read;

	ctx->read_s.zalloc = zalloc;
	ctx->read_s.zfree = zfree;
	ctx->write_s.zalloc = zalloc;
	ctx->write_s.zfree = zfree;

	r = deflateInit(&ctx->write_s, 3);
	yw_assert(r == Z_OK);
	r = inflateInit(&ctx->read_s);
	yw_assert(r == Z_OK);

	conn->read = zlib_read;
	conn->write = zlib_write;
	conn->can_read = zlib_can_read;
	conn->zlib_state = ctx;
}

#endif /* YW_HAS_ZLIB */
