/* $Id: word.c,v 1.3 2001/05/05 11:36:18 dobrek Exp $ */

#include <yw/int/packet.h>
#include <yw/map.h>
#include <yw/util.h>
#include <string.h>

/**
 * {simple: yw/packet.h: fetch pointer from given word}
 * {this} fetches pointer from word <a>w</a>, and
 * stores it at *<a>res</a>.
 * {retval} <literal>0</literal> is returned in case of sucess, 
 * or <literal>-1</literal> when there is no such word, or it has
 * wrong type, *<a>res</a> is not touched in the later case.
 * {see: yw_packet_word(3)}
 */
int yw_word_fetch_ptr(YwWord *w, void **res)
{
	if (w == NULL || w->type != yw_ptr_word)
		return -1;
	*res = w->val.ptr;
	return 0;
}

/**
 * {simple: yw/packet.h: fetch void data from given word}
 * {this} checks whatever given word is of void type.
 * {retval} <literal>0</literal> is returned in case of sucess, 
 * or <literal>-1</literal> when there is no such word, or it has
 * wrong type.
 * {see: yw_word_word(3)}
 */
int yw_word_fetch_void(YwWord *w)
{
	if (w == NULL || w->type != yw_void_word)
		return -1;
	return 0;
}

/**
 * {simple: yw/packet.h: fetch binary data from given word}
 * {this} fetches binary data from word <a>w</a>, and stores
 * pointer to it at *<a>data</a>, and length at *<a>len</a>
 * if it's not NULL. These data is only valid until calling
 * yw_word_free(3) on <a>pkt</a>, though you are not expected
 * to free nor change this data.
 * {retval} <literal>0</literal> is returned in case of sucess,
 * or <literal>-1</literal> when there is no such word, or it
 * has wrong type, *<a>data</a> and *<a>len</a> are not touched
 * in the later case.
 * {see: yw_word_word(3)}
 */
int yw_word_fetch_bindata(YwWord *w, void **data, int *len)
{
	if (w == NULL || w->type != yw_bindata_word)
		return -1;
	*data = w->val.bin.data;
	if (len)
		*len = w->val.bin.len;
	return 0;
}

/**
 * {simple: yw/packet.h: fetch integer from given word}
 * {this} fetches integer from word <a>w</a>, and
 * stores it at *<a>res</a>.
 * {retval} <literal>0</literal> is returned in case of sucess, 
 * or <literal>-1</literal> when there is no such word, or it has
 * wrong type, *<a>res</a> is not touched in the later case.
 * {see: yw_word_word(3)}
 */
int yw_word_fetch_int(YwWord *w, int *res)
{
	if (w == NULL || w->type != yw_int_word)
		return -1;
	*res = w->val.i;
	return 0;
}

/**
 * {simple: yw/packet.h: fetch keyword from given word}
 * {this} fetches keyword from word <a>w</a>, and
 * stores it at *<a>res</a>.
 * These data is only valid until calling yw_word_free(3)
 * on <a>pkt</a>, though you are not expected to free nor change this data.
 * Returned keyword is '\0' terminated.
 * {retval} <literal>0</literal> is returned in case of sucess, 
 * or <literal>-1</literal> when there is no such word, or it has
 * wrong type, *<a>res</a> is not touched in the later case.
 * {see: yw_word_fetch_mapped_keyword(3), yw_packet_word(3)}
 */
int yw_word_fetch_keyword(YwWord *w, const char **res)
{
	if (w == NULL || w->type != yw_key_word)
		return -1;
	*res = w->val.ascii;
	return 0;
}

/**
 * {simple: yw/packet.h: fetch unicode string from given word}
 * {this} fetches unicode string from word <a>w</a>, and
 * stores pointer to it at *<a>res</a>.
 * These data is only valid until calling yw_word_free(3)
 * on <a>pkt</a>, though you are not expected to free nor change this data.
 * {retval} <literal>0</literal> is returned in case of sucess, 
 * or <literal>-1</literal> when there is no such word, or it has
 * wrong type, *<a>res</a> is not touched in the later case.
 * {see: yw_word_fetch_keyword(3), yw_packet_word(3)}
 */
int yw_word_fetch_string(YwWord *w, YwString **res)
{
	if (w == NULL || w->type != yw_string_word)
		return -1;
	*res = &w->val.string;
	return 0;
}

/**
 * {simple: yw/packet.h: fetch mapped keyword from given word}
 * {this} fetches keyword from word <a>w</a>, treates it with
 * yw_map(3) and stores result at *<a>res</a>.
 * {retval} <literal>0</literal> is returned in case of sucess, 
 * or <literal>-1</literal> when there is no such word, or it has
 * wrong type, *<a>res</a> is not touched in the later case.
 * {see: yw_word_fetch_keyword(3), yw_packet_word(3)}
 */
int yw_word_fetch_mapped_keyword(YwWord *w, const YwMapEntry e[],
				 YwMapCache **cache, int *res)
{
	const char *kw;

	if (yw_word_fetch_keyword(w, &kw))
		return -1;

	*res = yw_map(kw, e, cache);

	return 0;
}

/**
 * {simple: yw/packet.h: find named tag}
 * {this} searched taglist pointed to by <a>tags</a> for
 * tag named <a>name</a>. When taglist is invalid -- it bombs.
 * Same for yw_tag_fetch_* functions.
 * {retval} Pointer to value of tag named <a>name</a>, or NULL
 * if it's not found.
 * {see: yw_tag_validate(3)}
 */
YwWord *yw_tag_find(YwWord *tags, const char *name)
{
	YwWord *p, *n;

	for (p = tags; p; p = n->next) {
		yw_assert(p->type == yw_key_word);
		n = p->next;
		yw_assert(n);
		if (strcmp(name, p->val.ascii) == 0)
			return n;
	}

	return NULL;
}

/** 
 * {simple: yw/packet.h: check whatever taglist is valid}
 * {this} walks through taglist pointed to by <a>tags</a>
 * and checks its structure. When this function suceeds
 * yw_tag_find(3) won't bomb.
 * {retval} <literal>0</literal> when taglist is valid, 
 * <literal>-1</literal> otherwise.
 * {see: yw_tag_find(3)}
 */
int yw_tag_validate(YwWord *tags)
{
	YwWord *p, *n;

	for (p = tags; p; p = n->next) {
		if (p->type != yw_key_word)
			return -1;
		n = p->next;
		if (n == NULL)
			return -1;
	}

	return 0;
}

/**
 * {simple: yw/packet.h: retrive type of word}
 * {this} checks type of word <a>w</a>.
 * {retval} Type of word, or <symbol>yw_bad_word</symbol> when 
 * <a>w</a> is NULL.
 * {see: yw_word_next(3)}
 */
YwWordType yw_word_type(YwWord *w)
{
	return w == NULL ? yw_bad_word : w->type;
}

/**
 * {simple: yw/packet.h: retrive next word from list}
 * {this} is used to walk through list of words (e.g. in packet).
 * {retval} Pointer to next word, or NULL when there is no next word
 * or <a>w</a> is NULL.
 * {see: yw_word_type(3), yw_packet_word(3)}
 */
YwWord *yw_word_next(YwWord *w)
{
	return w == NULL ? NULL : w->next;
}

YwWord *yw_word_new_bindata(const void *ptr, int len)
{
	YwWord *w;

	w = YW_NEW_0(YwWord);
	w->type = yw_bindata_word;
	w->val.bin.data = yw_malloc_0(len + 1);
	w->val.bin.len = len;
	memcpy(w->val.bin.data, ptr, len);

	return w;
}

YwWord *yw_word_new_keyword(const char *keyword)
{
	YwWord *w;

	w = YW_NEW_0(YwWord);
	w->type = yw_key_word;
	w->val.ascii = yw_strdup(keyword);

	return w;
}

YwWord *yw_word_new_int(int32_t i)
{
	YwWord *w;

	w = YW_NEW_0(YwWord);
	w->type = yw_int_word;
	w->val.i = i;

	return w;
}

YwWord *yw_word_new_void()
{
	YwWord *w;

	w = YW_NEW_0(YwWord);
	w->type = yw_void_word;

	return w;
}

YwWord *yw_word_new_bad()
{
	YwWord *w;

	w = YW_NEW_0(YwWord);
	w->type = yw_bad_word;

	return w;
}

YwWord *yw_word_new_ptr(void *ptr)
{
	YwWord *w;

	w = YW_NEW_0(YwWord);
	w->type = yw_ptr_word;
	w->val.ptr = ptr;

	return w;
}

YwWord *yw_word_new_string(const YwString *str)
{
	YwWord *w;

	w = YW_NEW_0(YwWord);
	w->type = yw_string_word;
	yw_string_assign_string(&w->val.string, str);

	return w;
}

YwWord *yw_word_copy_of(const YwWord *src)
{
	YwWord *w;

	w = YW_NEW_0(YwWord);
	
	switch (w->type = src->type) {
	case yw_string_word:
		yw_string_assign_string(&w->val.string, 
					&src->val.string);
		break;
	case yw_key_word:
		w->val.ascii = yw_strdup(src->val.ascii);
		break;
	case yw_int_word:
		w->val.i = src->val.i;
		break;
	case yw_bindata_word:
		w->val.bin.data = yw_malloc_0(
				w->val.bin.len = src->val.bin.len);
		memcpy(w->val.bin.data, src->val.bin.data,
			w->val.bin.len);
		break;
	case yw_bad_word:
	case yw_void_word:
		break;
	case yw_ptr_word:
		w->val.ptr = src->val.ptr;
		break;
	default:
		yw_halt();
	}

	return w;
}

void yw_word_free(YwWord *w)
{
	if (w == NULL)
		return;

	switch (w->type) {
	case yw_key_word:
		yw_free(w->val.ascii);
		break;
	case yw_string_word:
		yw_string_free(&w->val.string);
		break;
	case yw_bindata_word:
		yw_free(&w->val.bin.data);
		break;
	case yw_int_word:
	case yw_void_word:
	case yw_ptr_word:
		break;
	default:
		yw_halt();
	}
	yw_free(w);
}
