/* $Id: window.c,v 1.1 2001/05/04 11:04:23 malekith Exp $ */

#include <yw/window.h>
#include <yw/util.h>

struct YwWindow_struct {
	int id;
	YwConnection *conn;
	YwBuffer *buf;

	int w, h;
};

static int char_width(void *userdata, uint32_t ch)
{
	YwWindow *w = userdata;

	return yw_conn_char_width(w->conn, ch);
}

static void cache_widths(void *userdata, const YwString *s)
{
	YwWindow *w = userdata;

	yw_conn_cache_widths(w->conn, s);
}

static void remote_goto(void *userdata, int x, int y)
{
	YwWindow *w = userdata;

	yw_conn_queue_and_free(w->conn, yw_packet_make(yw_void_call_packet,
			"k", "goto",
			"i", w->id,
			"i", x,
			"i", y,
			YW_END));
}

static void remote_puts(void *userdata, const YwString *str)
{
	YwWindow *w = userdata;

	yw_conn_queue_and_free(w->conn, yw_packet_make(yw_void_call_packet,
			"k", "print",
			"s", str,
			YW_END));
}

static void remote_attr(void *userdata, YwBufAttr a, int v)
{
	YwWindow *w = userdata;

	yw_conn_queue_and_free(w->conn, yw_packet_make(yw_void_call_packet,
			"k", "attr",
			"k", yw_bufattr_name(a),
			"i", v,
			YW_END));
}

static void remote_flush(void *userdata)
{
	YwWindow *w = userdata;

	yw_conn_queue_and_free(w->conn, yw_packet_make(yw_void_call_packet,
			"k", "flush",
			YW_END));
	yw_conn_queue_flush(w->conn);
}

static YwBufferCallback bufcb;

/**
 * {simple: yw/window.h: create new window}
 * {this} creates new window on server side (by sending open_window
 * command to the server) and YwWindow helper structure on client
 * side. YwWindow supports all primitive window operations,
 * like drawing, scrolling, etc. Window is connected to server 
 * through <a>conn</a>. <a>taglist</a> contains parameters
 * of created window. {this} fetches for own purpouses 
 * "width", "virtual_width", "height" and "virtual_height".
 * However entire packet is passed to server (and then to WM) untouched.
 * {retval} Pointer to YwWindow structure, or NULL in case of error.
 * Note: in current implementation, this function bombs when server
 * refuses to create window, so it never returns NULL.
 * {see: yw_window_new_and_free(3), yw_window_close(3), yw_window_buffer(3)}
 */
YwWindow *yw_window_new(YwConnection *conn, YwPacket *taglist)
{
	YwWindow *w;
	YwPacket *pkt;
	const char *kw;
	YwWord *tags;

	w = YW_NEW_0(YwWindow);
	
	w->conn = conn;
	w->w = 40;
	w->h = 10;
	tags = yw_packet_word(taglist, 0);
	yw_word_fetch_int(yw_tag_find(tags, "width"), &w->w);
	yw_word_fetch_int(yw_tag_find(tags, "virtual_width"), &w->w);
	yw_word_fetch_int(yw_tag_find(tags, "height"), &w->h);
	yw_word_fetch_int(yw_tag_find(tags, "virtual_height"), &w->h);

	if (bufcb.char_width == NULL) {
		bufcb.char_width = char_width;
		bufcb.cache_widths = cache_widths;
		bufcb.remote_goto = remote_goto;
		bufcb.remote_puts = remote_puts;
		bufcb.remote_attr = remote_attr;
		bufcb.remote_flush = remote_flush;
	}

	pkt = yw_packet_make(yw_call_packet, "k", "open_window", YW_END);
	yw_packet_append_words(pkt, tags, -1);
	
	yw_conn_send_and_free(conn, pkt);
	pkt = yw_conn_recv_reply(conn, -1);
	
	if (pkt == NULL || 
	    yw_word_fetch_keyword(yw_packet_word(pkt, 0), &kw) ||
	    strcmp(kw, "win_id") != 0 ||
	    yw_word_fetch_int(yw_packet_word(pkt, 1), &w->id))
	    	yw_halt();
	
	w->buf = yw_buffer_new(&bufcb, w, w->w, w->h, 1);

	return w;
}

/**
 * {simple: yw/window.h: create new window and free taglist}
 * {this} acts the same as yw_window_new(3), and then it
 * yw_packet_free(3) <a>tags</a> argument.
 * {see: yw_window_new(3), yw_packet_free(3)}
 */
YwWindow *yw_window_new_and_free(YwConnection *conn, YwPacket *tags)
{
	YwWindow *w;

	w = yw_window_new(conn, tags);
	yw_packet_free(tags);

	return w;
}

/**
 * {simple: yw/window.h: close window and release YwWindow}
 * {this} sends "close_window" command to the server, in order
 * to close the window. Then it releases any resources associated
 * with <a>win</a> parameter.
 * {see: yw_window_new(3)}
 */
void yw_window_close(YwWindow *win)
{
	yw_conn_send_and_free(win->conn, 
		yw_packet_make(yw_void_call_packet,
			"k", "close_window",
			"i", win->id, 
			YW_END));
	yw_buffer_free(win->buf);
	yw_free(win);
}

/**
 * {simple: yw/window.h: get buffer of window}
 * {this} reveals YwBuffer structure of window <a>win</a>.
 * You can draw into this buffer using yw_buffer_goto(3),
 * yw_buffer_puts(3) etc. In order to get updated content
 * actually displayed, you need to call yw_buffer_sync(3)
 * on this buffer.
 * {retval} Pointer to buffer associated with window.
 * {see: yw_window_new(3), yw_window_id(3), yw_buffer_new(3)}
 */
YwBuffer *yw_window_buffer(YwWindow *win)
{
	return win->buf;
}

/**
 * {simple: yw/window.h: get win_id of window}
 * {this} returns integer, that identifies window <a>win</a>
 * for server. It can be used to call server directly (and not
 * with yw_window_* functions).
 * {retval} Win_id of window.
 * {see: yw_window_new(3), yw_window_buffer(3)}
 */
int yw_window_id(YwWindow *win)
{
	return win->id;
}

/**
 * {simple: yw/window.h: set origin of virtual window}
 * Some windows are scrollable. This is the case when virtual
 * width and/or height is bigger then actuall window width
 * or height. {this} can be used to scroll such a window.
 * It sets cooridantes of visible rectangle in virtual
 * window to (<a>x</a>, <a>y</a>) for <a>win</a>.
 * {see: yw_window_buffer(3), yw_window_new(3)}
 */
void yw_window_set_origin(YwWindow *win, int x, int y)
{
	yw_conn_send_and_free(win->conn, 
		yw_packet_make(yw_void_call_packet,
			"k", "set_origin",
			"i", win->id,
			"i", x,
			"i", y, 
			YW_END));
}
