/* $Id: util.c,v 1.4 2001/05/11 10:38:05 malekith Exp $ */

/* needed for usleep() */
#define _XOPEN_SOURCE 500

#include <yw/util.h>

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

/**
 * {simple: yw/util.h: wait a bit}
 * Sleep <a>ms</a> miliseconds.
 * {see: usleep, sleep}
 */
void yw_sleep(int ms)
{
	usleep(ms * 1000);
}

/**
 * {simple: yw/util.h: bomb horribly}
 * Write <a>msg</a> to stderr and exit.
 * {see: exit}
 */
void yw_die(const char *msg)
{
	fprintf(stderr, "\n\nFATAL ERROR: %s\n\n", msg);
	exit(2);
}

/**
 * {simple: yw/util.h: concatenate 8bit strings}
 * Concat all 8bit strings, last argument has to be NULL.
 * {retval} yw_malloc()ed string.
 * {see: yw_strdup, yw_free}
 */
char *yw_concat(const char *a1, ...)
{
	va_list ap;
	const char *p;
	char *ret, *dst;
	int len = 0;

	va_start(ap, a1);
	for (p = a1; p; p = va_arg(ap, const char *))
		len += strlen(p);
	va_end(ap);
	
	len++;

	ret = dst = yw_malloc(len);
	
	va_start(ap, a1);
	for (p = a1; p; p = va_arg(ap, const char *)) {
		strcpy(dst, p);
		dst = yw_strchr(dst, 0);
	}
	va_end(ap);

	return ret;
}

/**
 * {simple: yw/util.h: grow allocated memory}
 * {this} is same as realloc(3), but it bombs
 * in case of error.
 * {see: realloc, yw_malloc, yw_free}
 */
void *yw_realloc(void *ptr, int size)
{
	if (ptr == 0)
		ptr = malloc(size);
	else
		ptr = realloc(ptr, size);
		
	if (ptr == NULL)
		yw_die("yw_realloc: out of memory.");

	return ptr;
}

/**
 * {simple: yw/util.h: allocate memory}
 * {this} is same as malloc(3), but it bombs
 * in case of error.
 * {see: malloc, yw_malloc_0, yw_free}
 */
void *yw_malloc(int size)
{
	void *ptr;

	ptr = malloc(size);
	if (ptr == NULL)
		yw_die("yw_malloc: out of memory.");

	return ptr;
}

/**
 * {simple: yw/util.h: allocate and zero memory}
 * {this} is same as malloc(3), but it bombs
 * in case of error. It additionaly initialize allocated
 * memory to 0.
 * {see: yw_malloc, yw_free}
 */
void *yw_malloc_0(int size)
{
	void *ptr;

	ptr = malloc(size);
	if (ptr == NULL)
		yw_die("yw_malloc_0: out of memory.");

	memset(ptr, 0, size);
	
	return ptr;
}

/**
 * {simple: yw/util.h: free allocated memory}
 * {this} is same as free(3), but it does
 * nothing when passed NULL pointer.
 * {see: yw_malloc, yw_realloc, yw_malloc_0, free}
 */
void yw_free(void *ptr)
{
	if (ptr)
		free(ptr);
}

/**
 * {simple: yw/util.h: copy string into allocated buffer}
 * {this} allocated strlen(<a>str</a>) + 1 bytes of memory
 * and copies <a>str</a> there.
 * {retval} Pointer to allocated data. Caller is responsible for releasing
 * it using yw_free(3) when no longer needed.
 * {see: yw_setstr, yw_free}
 */
char *yw_strdup(const char *str)
{
	char *ptr;
	int len;

	len = strlen(str) + 1;
	ptr = malloc(len);
	if (ptr == NULL)
		yw_die("yw_strdup: out of memory.");
		
	memcpy(ptr, str, len);

	return ptr;
}

/**
 * {simple: yw/util.h: copy string into allocated buffer releasing previous content}
 * {this} allocated strlen(<a>str</a>) + 1 bytes of memory
 * and copies <a>str</a> to *<a>dst</a>. If *<a>dst</a> wasn't NULL it
 * is freed.
 * {retval} Pointer to allocated data (i.e. *<a>dst</a>). Caller is responsible 
 * for releasing it using yw_free(3) when no longer needed.
 * {see: yw_strdup, yw_free}
 */
char *yw_setstr(char **dst, const char *str)
{
	int len;
	char *tmp;


	tmp = *dst;
	
	len = strlen(str) + 1;
	*dst = malloc(len);
	if (*dst == NULL)
		yw_die("yw_setstr: out of memory.");
		
	memcpy(*dst, str, len);

	yw_free(tmp);
	
	return *dst;
}

/**
 * {simple: yw/util.h: find one string in another}
 * {this} seraches in \0 terminated 8 bit string <a>heystack</a>
 * for <a>needle</a>. It's the same as BSD strstr(3) function.
 * Implementation is schamelessly stolen from Linux kernel.
 * {retval} When <a>needle</a> is found, pointer to it in <a>heystack</a>
 * and otherwise NULL is returned.
 * {see: strstr}
 */
char *yw_strstr(const char *heystack, const char *needle)
{
	int l1, l2;

	l2 = strlen(needle);
	if (!l2)
		return (char *) heystack;
	l1 = strlen(heystack);
	while (l1 >= l2) {
		l1--;
		if (!memcmp(heystack,needle,l2))
			return (char *) heystack;
		heystack++;
	}
	return NULL;
}

/**
 * {simple: yw/util.h: locate character in string}
 * {this} returns pointer to the first occurence of character <a>c</a> 
 * in string <a>s</a>. It's placed here, as it's non-ANSI.
 * {retval} Pointer to character found or NULL.
 */
char *yw_strchr(const char *s, int c)
{
	for (; *s != (char)c; s++)
		if (*s == 0)
			return NULL;
	return (char*)s;
}

/* stolen from linux kernel */
void *yw_memmove(void *dest, const void *src, unsigned count)
{
	char *tmp, *s;

	if (dest <= src) {
		tmp = (char *) dest;
		s = (char *) src;
		while (count--)
			*tmp++ = *s++;
		}
	else {
		tmp = (char *) dest + count;
		s = (char *) src + count;
		while (count--)
			*--tmp = *--s;
		}

	return dest;
}
