/* $Id: sysconn.c,v 1.11 2001/05/11 10:38:05 malekith Exp $ */

/* for getaddrinfo() */
#define _XOPEN_SOURCE 500

#include <yw/util.h>
#include <yw/random.h>
#include <yw/int/conn.h>
#include <yw/int/calls.h>
#include <yw/int/packet.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netdb.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>

int yw_conn_fd(YwConnection *conn)
{
	return conn->sock_fd;
}

static int simple_can_read(YwConnection *conn, int ms)
{
	int r;
	fd_set rd;
	struct timeval tv;

	FD_ZERO(&rd);
	FD_SET(conn->sock_fd, &rd);
	tv.tv_sec = ms / 1000;
	tv.tv_usec = (ms % 1000) * 1000;
	
	do
		r = select(conn->sock_fd + 1, &rd, NULL, NULL, &tv);
	while (r == -1 && errno == EINTR);

	if (r == -1)
		yw_int_return_err(conn, YW_ERR_READ);
	
	return r;
}

static int simple_read(YwConnection *conn, void *buf, int len)
{
	int r;
	
	if (conn->state)
		return conn->state;
	
	do	
		r = recv(conn->sock_fd, buf, len, 0);
	while (r == -1 && errno == EINTR);

	if (r == 0)
		yw_int_return_err(conn, YW_ERR_EOF);
	else if (r < 0)
		yw_int_return_err(conn, YW_ERR_READ);

	return r;
}

static int simple_write(YwConnection *conn, const void *buf, int len)
{
	int r, p;
	
	if (conn->state)
		return conn->state;

	for (p = 0; p < len; p += r) {
		do
			r = send(conn->sock_fd, (uint8_t*)buf + p, len - p, 0);
		while (r == -1 && errno == EINTR);
		if (r < 0)
			yw_int_return_err(conn, YW_ERR_WRITE);
	}

	return len;
}

static int try_connect(YwConnection *conn)
{
	char *target;
	struct addrinfo ai, *a = NULL, *p;
	int err;

	memset(&ai, 0, sizeof(ai));
	ai.ai_family = PF_UNSPEC;
	ai.ai_socktype = SOCK_STREAM;

	target = yw_strdup(conn->display);
	*yw_strchr(target, '%') = 0;

	err = getaddrinfo(target, yw_strchr(target, 0) + 1, &ai, &a);

	if (err)
		yw_int_return_err(conn, YW_ERR_BAD_AUTH_FILE);

	conn->sock_fd = -1;
	
	for (p = a; p; p = p->ai_next) {
		conn->sock_fd = socket(p->ai_family, SOCK_STREAM, 
				       p->ai_protocol);
		if (conn->sock_fd == -1)
			continue;
		if (connect(conn->sock_fd, p->ai_addr, p->ai_addrlen)) { 
			close(conn->sock_fd);
			conn->sock_fd = -1;
		} else
			break;
	}

	if (conn->sock_fd == -1)
		yw_int_return_err(conn, YW_ERR_SOCKET);

	return 0;
}

static void kill_socket(YwConnection *conn)
{
	if (conn->sock_fd != -1)
		close(conn->sock_fd);
}

/**
 * {simple: yw/sock.h: allocate and initialize memory for new connection}
 * {this} allocates new connection.
 * {retval} Opaque pointer.
 * {see: yw_conn_open, yw_conn_free}
 */
YwConnection *yw_conn_new()
{
	YwConnection *conn;

	conn = YW_NEW_0(YwConnection);

#define env(member, NAME) \
	conn->member = getenv(NAME) ? yw_strdup(getenv(NAME)) : 0

	env(authority_file, "YAUTHORITY");
	env(display, "YDISPLAY");
	env(home, "HOME");
	env(random_file, "RANDFILE");
	if (conn->random_file == 0)
		conn->random_file = yw_strdup("/dev/urandom");
	env(randseed, "YRANDSEED");
	
	conn->crypt = YW_MAYBE;
	conn->zlib = YW_MAYBE;

	conn->read = simple_read;
	conn->write = simple_write;
	conn->can_read = simple_can_read;

	conn->state = YW_ERR_NOT_CONNECTED;

	conn->sock_fd = -1;

	return conn;
}

static int read_authority(YwConnection *conn)
{
	char line[300] = "";
	FILE *f;
	int n;

	f = fopen(conn->authority_file, "rb");
	if (f == 0)
		yw_int_return_err(conn, YW_ERR_NO_AUTH_FILE);
	fgets(line, sizeof(line), f);
	n = strlen(line);
	if (n == 0 || line[n-1] != '\n')
		goto oops;
	line[n-1] = 0;
	if (conn->display == NULL)
		conn->display = yw_strdup(line);
	
	conn->authority = yw_malloc(YW_COOKIE_SIZE);
	if (fread(conn->authority, 1, YW_COOKIE_SIZE, f) != YW_COOKIE_SIZE)
		goto oops;
		
	fclose(f);

	return 0;
oops:
	fclose(f);
	yw_int_return_err(conn, YW_ERR_BAD_AUTH_FILE);
}

void yw_int_init_rand(YwConnection *conn)
{
	static int rand_initialized;
	char pool[512];
	int i;
	FILE *f;
	char *fn;

	if (conn->authority_file == 0)
		conn->authority_file = yw_concat(conn->home, 
						 "/.Yauthority", 0);
		
	if (rand_initialized)
		return;
	
	rand_initialized = 1;

	/* try read ~/.Yrandseed file */
	if (conn->randseed == 0)
		conn->randseed = yw_concat(conn->home, "/.Yrandseed", 0);
		
	f = fopen(conn->randseed, "rb");
	if (f == 0) {
		fn = yw_concat(conn->home, 
				"/.ssh/random_seed", 0);
		f = fopen(fn, "rb");
		yw_free(fn);
	}
	if (f) {
		fread(pool, 1, sizeof(pool), f);
		fclose(f);
	}

	/* otherwise rely on what was on stack */
	yw_random_add(pool, sizeof(pool));
	
	/* then tryout what libc has for us */
#ifdef __unix__
	srand(time(0) ^ getpid());
#else
	srand(time(0));
#endif

	for (i = 0; i < (int)sizeof(pool); i++)
		pool[i] = rand();
	yw_random_add(pool, sizeof(pool));
	
	/* let's see whet the kernel has, this is optional */
	f = fopen(conn->random_file, "rb");
	if (f) {
		fread(pool, 1, sizeof(pool), f);
		fclose(f);
		yw_random_add(pool, sizeof(pool));
	}

	/* try to dumpout the random file... */
	f = fopen(conn->randseed, "wb");
	if (f) {
		yw_random_get(pool, sizeof(pool));
		fwrite(pool, 1, sizeof(pool), f);
		fclose(f);
	}
}

/**
 * {simple: yw/sock.h: open new connection to server}
 * When connection structure is allocated using yw_conn_new(3),
 * you need to use this function in order to connect to the server.
 * You can previously set some connection parameters using yw_set_flag(3).
 * {this} creates socket, and tries to connect it to the server.
 * Then it tries to authenticate, and does basic connection feature
 * negotation. Since it's quite complex, it can fail for several
 * reasons.
 * After you're done with this connection, you need to close it, and
 * release YwConnection structure using yw_conn_free(3).
 * {retval}
 *  The <function>yw_conn_open()</function> function returns 0 for success or
 *  one of the following values in case of error:
 *   <itemizedlist>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_ALREADY_CONNECTED</symbol> - connection is already
 *      active.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_NO_HOME</symbol> - <symbol>HOME</symbol> flag not set,
 *      and getenv("HOME") failed.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_NO_AUTH_FILE</symbol> - authority file hasn't been found.
 *      Check errno for details.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_BAD_AUTH_FILE</symbol> - authority file is invalid. This
 *      includes 'host not found' errors.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_SOCKET</symbol> - socket problem. 
 *      <function>socket()</function> or <function>connect()</function>
 *      call failed.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_PROTO</symbol> - protocol error.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_VERSION</symbol> - bad version of peer.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_BAD_AUTH</symbol> - peer didn't authenticate properly.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_EOF</symbol> - unexpected EOF reading socket.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_READ</symbol> - read error.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_WRITE</symbol> - write error.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_NO_ZLIB</symbol> - couldn't setup compressed link,
 *      this can fail for two reasons. One is that you set "ZLIB" flag to
 *      "yes" (instead of "maybe") and local libyw implementaion doesn't
 *      support zlib (you can check it, if it supports zlib preprocessor
 *      macro <symbol>YW_HAS_ZLIB</symbol> is defined), the other is that
 *      you set "ZLIB" to "yes" and peer doesn't support zlib compression.
 *     </para>
 *    </listitem>
 *    <listitem>
 *     <para>
 *      <symbol>YW_ERR_NO_CRYPT</symbol> - the other side doesn't support
 *      encryption at all (strange).
 *     </para>
 *    </listitem>
 *   </itemizedlist>
 *  It also sets internal state of <a>conn</a>
 *  to returned value. It can be checked later using
 *  yw_conn_state(3). It's impossible to reuse the
 *  same connection for second <function>yw_conn_open()</function> call.
 * </para>
 * <para>
 *  For more information about determing the way connection is 
 *  established, please refer to yw_conn_set_flag(3).
 */
int yw_conn_open(YwConnection *conn)
{
	if (conn->state == 0)
		return YW_ERR_ALREADY_CONNECTED;
		
	if (conn->state != YW_ERR_NOT_CONNECTED)
		return conn->state;
		
	if (conn->home == 0)
		yw_int_return_err(conn, YW_ERR_NO_HOME);

	yw_int_init_rand(conn);

	conn->state = 0;

	if (read_authority(conn))
		return conn->state;

	if (try_connect(conn))
		return conn->state;

	if (yw_int_conversation(conn))
		return conn->state;

	return 0;
}

/**
 * {simple: yw/sock.h: free connection structure}
 * {this} release any resources (memory, sockets, etc.)
 * associated with connection pointed by <a>conn</a> parameter.
 * After that <a>conn</a> pointer is invalid.
 */
void yw_conn_free(YwConnection *conn)
{
	YwPacket *pkt;
	
	while (conn->pkt_waiting) {
		pkt = conn->pkt_waiting;
		conn->pkt_waiting = pkt->next;
		yw_packet_free(pkt);
	}

	kill_socket(conn);
	
	yw_int_conn_crypt_free(conn);
#ifdef YW_HAS_ZLIB
	yw_int_conn_zlib_free(conn);
#endif /* YW_HAS_ZLIB */
	yw_free(conn->cookie);
	yw_free(conn->authority);
	yw_free(conn->home);
	yw_free(conn->authority_file);
	yw_free(conn->random_file);
	yw_free(conn->randseed);
	yw_free(conn->display);
	yw_free(conn);
}

/**
 * {simple: yw/sock.h: check state of connection}
 * {this} is used to check current state of connection.
 * yw_strerror(3) function can be used to translate returned value 
 * into human-readable string. yw_conn_get_state(3) function can be
 * used to return extended information about error, though
 * it's mainly useful for debugging library itself.
 * {retval}
 * The <function>yw_conn_state()</function> function returns one of YW_ERR_*
 * values. They are defined in &lt;<filename>yw/error.h</filename>&gt;.
 * YW_ERR_OK is 0.
 */
int yw_conn_state(YwConnection *conn)
{
	return conn->state;
}

/**
 * {simple: yw/sock.h: extensivly check state of connection}
 *  {this} is used to check current state of connection.
 *  Connection state is stored at *<a>state</a>,
 *  line and file of library, where the error was set, are
 *  stored at *<a>file</a> and *<a>line</a>. Any of these three is not
 *  stored if pointer is NULL.
 * </para>
 * <para>
 *  yw_strerror(3) function can be used to
 *  translate returned value into human-readable string.
 *  yw_conn_state(3) may be more handy
 *  in regular situation, when you don't want to debug
 *  the library. <userinput>yw_conn_state(conn)</userinput>
 *  is equivalent to <userinput>yw_conn_get_state(conn, NULL, 
 *  NULL, NULL)</userinput>.
 *
 * {retval}
 *  The <function>yw_conn_get_state()</function> function
 *  returns one of YW_ERR_* values. They are defined in
 *  &lt;<filename>yw/error.h</filename>&gt;. YW_ERR_OK is
 *  0. (note: the value is both returned and stored at
 *  *<a>state</a>).
 */
int yw_conn_get_state(YwConnection *conn, const char **file, 
		      int *line, int *state)
{
	if (file)
		*file = conn->err_file;
	if (line)
		*line = conn->err_line;
	if (state)
		*state = conn->state;

	return conn->state;
}
