/* $Id: string.c,v 1.6 2001/05/23 08:46:01 malekith Exp $ */

#include <yw/util.h>
#include <yw/string.h>
#include <yw/int/packet.h>
#include <string.h>

/**
 * {simple: yw/string.h: find character in string}
 * {this} looks for first occurance of <a>ch</a> in <a>str</a>.
 * {retval} Zero based index of character is returned, or 
 * <literal>-1</literal> if it is not found.
 * {see: strchr}
 */
int yw_string_idx(const YwString *str, uint32_t ch)
{
	int i;

	for (i = 0; i < str->len; i++)
		if (str->chars[i] == ch)
			return i;
	
	return -1;
}

/**
 * {simple: yw/string.h: append one string at the end of another}
 * {this} appends string <a>rest</a> to the end of <a>s</a>.
 * {see: yw_concat}
 */
void yw_string_concat(YwString *s, const YwString *rest)
{
	s->chars = yw_realloc(s->chars, 
			(s->len + rest->len + 1) * sizeof(uint32_t));
	memcpy(s->chars + s->len, rest->chars, rest->len * sizeof(uint32_t));
	s->len += rest->len;
	s->chars[s->len] = 0;
}

/**
 * {simple: yw/string.h: compare two strings}
 * {this} comparse strings given in <a>s1</a> and <a>s2</a> parameters.
 * It is eqivalent of strcmp(3), except that it works on UNICODE instead of
 * ASCII. Collating order used is locale independent and is simply order of
 * characters in UNICODE.
 * {retval} {this} returns integer less then, equal to, or greater then zero,
 * if <a>s1</a> is found respectivly less then, equal or greater then <a>s2</a>.
 * {see: strcmp}
 */
int yw_string_cmp(const YwString *s1, const YwString *s2)
{
	int i, n;

	n = s1->len < s2->len ? s1->len : s2->len;
		
	for (i = 0; i < n; i++)
		if (s1->chars[i] != s2->chars[i])
			break;
	if (i < n) {
		if (s1->chars[i] < s2->chars[i])
			return -1;
		else
			return 1;
	}
	
	if (s1->len == s2->len)
		return 0;
	
	if (s1->len < s2->len)
		return -1;
	else
		return 1;
}

/**
 * {simple: yw/string.h: assign empty string}
 * {this} initializes *<a>s</a> to contain empty string.
 * {retval} {this} always returns 0.
 * {see: yw_string_assign_one_char(3)}
 */
int yw_string_assign_empty(YwString *s)
{
	s->chars = yw_malloc_0(sizeof(uint32_t));
	s->len = 0;

	return 0;
}

/**
 * {simple: yw/string.h: assign one character copied n times}
 * {this} initializes *<a>s</a> to contain n characters <a>ch</a>.
 * {retval} {this} always returns 0.
 * {see: yw_string_assign_one_char(3)}
 */
int yw_string_assign_n_chars(YwString *s, uint32_t ch, int n)
{
	int i;
	
	s->chars = yw_malloc_0(sizeof(uint32_t) * (n + 1));
	s->len = n;

	for (i = 0; i < n; i++)
		s->chars[i] = ch;

	return 0;
}

/**
 * {simple: yw/string.h: assign one character long string}
 * {this} initializes *<a>s</a> to contain one character string
 * passed in <a>ch</a> parameter.
 * {retval} {this} always returns 0.
 * {see: yw_string_assign_empty(3)}
 */
int yw_string_assign_one_char(YwString *s, uint32_t ch)
{
	s->chars = yw_malloc_0(sizeof(uint32_t) * 2);
	s->len = 1;
	s->chars[0] = ch;

	return 0;
}

/**
 * {simple: yw/string.h: assign UTF32 string}
 * {this} initializes *<a>s</a> to contain UTF32 encoded string
 * passed in <a>chars</a> parameter, of length (in characters, not bytes) <a>len</a>.
 * <a>chars</a> can be NULL if <a>len</a> is <literal>0</literal>. 
 * If <a>len</a> is -1 then length of <a>chars</a> parameter is determined
 * based on 0 terminator.
 * String is encoded in host endian, 32 bit UNICODE, which is also internal
 * representation used by YwString datatype.
 * {retval} {this} always returns 0.
 * {see: yw_string_assign_cstring(3)}
 */
int yw_string_assign_utf32(YwString *s, const uint32_t *chars, int len)
{
	if (len == -1) {
		const uint32_t *p;
		len++;
		for (p = chars; *p; p++)
			len++;
	} else if (len == 0)
		return yw_string_assign_empty(s);
	
	s->chars = yw_malloc_0(len * sizeof(uint32_t) + sizeof(uint32_t));
	memcpy(s->chars, chars, len * sizeof(uint32_t));
	s->len = len;
	
	return 0;
}


/**
 * {simple: yw/string.h: assign one string to another}
 * {this} initializes *<a>s</a> to contain one string from <a>src</a>
 * parameter.
 * {retval} {this} always returns 0.
 * {see: yw_string_assign_cstring(3)}
 */
int yw_string_assign_string(YwString *s, const YwString *src)
{
	return yw_string_assign_utf32(s, src->chars, src->len);
}

/**
 * {simple: yw/string.h: assign UTF16 string}
 * {this} initializes *<a>s</a> to contain UTF16 encoded string
 * passed in <a>chars</a> parameter, of length (in characters, not bytes) <a>len</a>.
 * <a>chars</a> can be NULL if <a>len</a> is <literal>0</literal>. 
 * If <a>len</a> is -1 then length of <a>chars</a> parameter is determined
 * based on 0 terminator.
 * String is encoded in host endian, 16 bit UNICODE.
 * {retval} {this} returns 0 in case of success or one of yw_convert(3)
 * return codes when there is an error in parametrs encoding.
 * {see: yw_string_assign_cstring(3)}
 */
int yw_string_assign_utf16(YwString *s, const uint16_t *chars, int len)
{
	void *p;
	int l, err;
	
	if (len == -1) {
		const uint16_t *p;
		len++;
		for (p = chars; *p; p++)
			len++;
	} else if (len == 0)
		return yw_string_assign_empty(s);

	s->chars = NULL;
	s->len = 0;

	if ((err = yw_convert("UTF16", chars, len * sizeof(uint16_t),
			      YW_UTF32_HE, &p, &l)))
		return err;

	yw_assert(l % sizeof(uint32_t) == 0);
	s->len = l / sizeof(uint32_t);
	s->chars = p;

	return 0;
}

/**
 * {simple: yw/string.h: assign UTF8 string}
 * {this} initializes *<a>s</a> to contain UTF8 encoded string
 * passed in <a>chars</a> parameter, of length (in bytes) <a>len</a>.
 * <a>chars</a> can be NULL if <a>len</a> is <literal>0</literal>. 
 * If <a>len</a> is -1 then length of <a>chars</a> parameter is determined
 * based on 0 terminator.
 * {retval} {this} returns 0 in case of success or one of yw_convert(3)
 * return codes when there is an error in parametrs encoding.
 * {see: yw_string_assign_cstring(3)}
 */
int yw_string_assign_utf8(YwString *s, const uint8_t *chars, int len)
{
	void *p;
	int l, err;
	
	if (len == -1)
		len = strlen((const char*)chars);
	else if (len == 0)
		return yw_string_assign_empty(s);
	
	s->chars = NULL;
	s->len = 0;

	if ((err = yw_convert("UTF8", chars, len * sizeof(uint8_t),
			      YW_UTF32_HE, &p, &l)))
		return err;

	yw_assert(l % sizeof(uint32_t) == 0);
	s->len = l / sizeof(uint32_t);
	s->chars = p;

	return 0;
}

/**
 * {simple: yw/string.h: assign string in given encoding}
 * {this} initializes *<a>s</a> to contain string encoded using
 * <a>enc</a> parameter
 * passed in <a>chars</a> parameter, of length (in bytes) <a>len</a>.
 * <a>chars</a> can be NULL if <a>len</a> is <literal>0</literal>. 
 * If <a>len</a> is -1 then length of <a>chars</a> parameter is determined
 * based on 0 terminator.
 * When <a>enc</a> is NULL -- default, locale-based, encoding
 * is used (see yw_get_default_encoding(3) for more details, on how it is
 * determined).
 * {retval} {this} returns 0 in case of success or one of yw_convert(3)
 * return codes when there is an error in parametrs encoding.
 * {see: yw_string_free(3)}
 */
int yw_string_assign_cstring(YwString *s, const char *enc,
			     const char *chars, int len)
{
	void *p;
	int l, err;
	
	if (yw_default_encoding == NULL)
		yw_default_encoding = yw_get_default_encoding();
		
	if (len == -1)
		len = strlen(chars);
	else if (len == 0)
		return yw_string_assign_empty(s);
	
	if (enc == NULL)
		enc = yw_default_encoding;
		
	s->chars = NULL;
	s->len = 0;

	if ((err = yw_convert(enc, chars, len * sizeof(char),
			      YW_UTF32_HE, &p, &l)))
		return err;

	yw_assert(l % sizeof(uint32_t) == 0);
	s->len = l / sizeof(uint32_t);
	s->chars = p;

	return 0;
}

/**
 * {simple: yw/string.h: get value of string in UTF32}
 * {this} fetches value of *<a>s</a> in UTF32, host-endian, encoding and
 * stores pointer to yw_malloc()ed vactor of characters at *<a>chars</a>.
 * Length in characters (not bytes) is stored at *<a>len</a>, if it's not
 * NULL.
 * {retval} {this} always returns 0, as it only copy characters back and
 * fourth. Note that it's rather useless, as you have access to s->chars
 * and s->len.
 * Note, that it's up to the caller to yw_free(3) resulting vactor, which
 * *<a>chars</a> is pointing at.
 * {see: yw_string_get_cstring(3), yw_free(3)}
 */
int yw_string_get_utf32(const YwString *s, uint32_t **chars, int *len)
{
	*chars = yw_malloc_0(s->len * sizeof(uint32_t) + sizeof(uint32_t));
	*len = s->len;
	memcpy(*chars, s->chars, s->len * sizeof(uint32_t));
	
	return 0;
}

/**
 * {simple: yw/string.h: get value of string in UTF16}
 * {this} fetches value of *<a>s</a> in UTF16, host-endian, encoding and
 * stores pointer to yw_malloc()ed vactor of characters at *<a>chars</a>.
 * Length in characters (not bytes) is stored at *<a>len</a>, if it's not
 * NULL.
 * {retval} {this} returns 0 in case of success or one of yw_convert(3)
 * return codes when there is an error in parametrs encoding (in the later
 * case neither *<a>chars</a> nor *<a>len</a> is touched).
 * Note, that it's up to the caller to yw_free(3) resulting vactor, which
 * *<a>chars</a> is pointing at.
 * {see: yw_string_get_cstring(3), yw_free(3)}
 */
int yw_string_get_utf16(const YwString *s, uint16_t **chars, int *len)
{
	void *p;
	int l, err;
	
	if ((err = yw_convert(YW_UTF32_HE, s->chars, 
			s->len * sizeof(uint32_t), 
			"UTF16", &p, &l)))
		return err;
	
	l /= sizeof(uint16_t);

	if (len)
		*len = l;
	*chars = p;
	
	return 0;
}

/**
 * {simple: yw/string.h: get value of string in UTF8}
 * {this} fetches value of *<a>s</a> in UTF8 encoding and
 * stores pointer to yw_malloc()ed vactor of characters at *<a>chars</a>.
 * Length in bytes is stored at *<a>len</a>, if it's not
 * NULL.
 * {retval} {this} returns 0 in case of success or one of yw_convert(3)
 * return codes when there is an error in parametrs encoding (in the later
 * case neither *<a>chars</a> nor *<a>len</a> is touched).
 * Note, that it's up to the caller to yw_free(3) resulting vactor, which
 * *<a>chars</a> is pointing at.
 * {see: yw_string_get_cstring(3), yw_free(3)}
 */
int yw_string_get_utf8(const YwString *s, uint8_t **chars, int *len)
{
	void *p;
	int l, err;
	
	if ((err = yw_convert(YW_UTF32_HE, s->chars,
			s->len * sizeof(uint32_t), 
			"UTF8", &p, &l)))
		return err;
	
	l /= sizeof(uint8_t);
	
	if (len)
		*len = l;
	*chars = p;
	
	return 0;
}

/**
 * {simple: yw/string.h: get value of string in given encoding}
 * {this} fetches value of *<a>s</a> in encoding given in <a>enc</a> 
 * parameter and stores pointer to yw_malloc()ed vactor of characters 
 * at *<a>chars</a>.
 * Length in bytes is stored at *<a>len</a>, if it's not NULL.
 * When <a>enc</a> is NULL -- default, locale-based, encoding
 * is used (see yw_get_default_encoding(3) for more details, on how it is
 * determined).
 * {retval} {this} returns 0 in case of success or one of yw_convert(3)
 * return codes when there is an error in parametrs encoding (in the later
 * case neither *<a>chars</a> nor *<a>len</a> is touched).
 * Note, that it's up to the caller to yw_free(3) resulting vactor, which
 * *<a>chars</a> is pointing at.
 * {see: yw_free(3)}
 */
int yw_string_get_cstring(const YwString *s, const char *enc, 
			  char **chars, int *len)
{
	void *p;
	int l, err;

	if (yw_default_encoding == NULL)
		yw_default_encoding = yw_get_default_encoding();
	if (enc == NULL)
		enc = yw_default_encoding;
		
	if ((err = yw_convert(YW_UTF32_HE, s->chars,
			s->len * sizeof(uint32_t), 
			enc, &p, &l)))
		return err;
	
	if (len)
		*len = l;
	*chars = p;
	
	return 0;
}

/**
 * {simple: yw/string.h: release memory occupied by string}
 * {this} releases vectors of characters, *<a>s</a>->chars points to.
 * You should note that it <emphasis>doesn't</emphasis> free
 * <a>s</a> itself. Regular use:
 * <programlisting>
 *    YwString str;
 *    ...
 *    yw_string_assign_cstring(&amp;str, NULL, "Hello world!", -1);
 *    ...
 *    yw_string_concat(&amp;str, some_other);
 *    ...
 *    yw_string_free(&amp;str);
 * </programlisting>
 * {see: yw_string_assign_cstring(3), yw_free(3)}
 */
void yw_string_free(YwString *s)
{
	yw_free(s->chars);
	/* just in case ... */
	s->chars = NULL;
	s->len = 0;
}
