/* $Id: send.c,v 1.8 2001/05/05 11:36:18 dobrek Exp $ */

#include <yw/util.h>
#include <yw/sock.h>
#include <yw/int/packet.h>
#include <yw/int/conn.h>
#include <string.h>

#define add_int(kkk)					\
	do {						\
		*dst++ = (uint8_t)((kkk)>>24);		\
		*dst++ = (uint8_t)((kkk)>>16);		\
		*dst++ = (uint8_t)((kkk)>>8);		\
		*dst++ = (uint8_t)((kkk));		\
	} while (0)

static int write_packet(uint8_t * dst, YwPacket * pkt)
{
	int len, i = 0, size;
	YwWord *w;

	len = 6;

	if (dst) {
		*dst++ = pkt->type;
		*dst++ = 4;
		add_int(pkt->len);
	}

	for (w = pkt->words; w; w = w->next)
		switch (w->type) {
		case yw_key_word:
			size = strlen(w->val.ascii);
			yw_assert(size < 255);
			len += 2 + size;
			if (dst) {
				*dst++ = 'k';
				*dst++ = (uint8_t) size;
				for (i = 0; i < size; i++)
					*dst++ = w->val.ascii[i];
			}
			break;
		case yw_string_word:
			size = w->val.string.len;
			if (size > 63) {
				len += 3;
				if (dst) {
					*dst++ = 'S';
					add_int(size * 4);

				}
			} else if (dst) {
				*dst++ = 's';
				*dst++ = (uint8_t) (size * 4);
			}
			len += 2 + 4 * size;
			if (dst)
				for (i = 0; i < size; i++)
					add_int(w->val.string.chars[i]);
			break;
		case yw_int_word:
			len += 6;
			if (dst) {
				*dst++ = 'i';
				*dst++ = 4;
				add_int(w->val.i);
			}
			break;
		case yw_bindata_word:
			size = w->val.bin.len;
			if (size > 255) {
				len += 3;
				if (dst) {
					*dst++ = 'B';
					add_int(size);

				}
			} else if (dst) {
				*dst++ = 'b';
				*dst++ = (uint8_t) (size);
			}
			len += 2 + size;
			if (dst)
				for (i = 0; i < size; i++)
					*dst++ = w->val.bin.data[i];
			break;
		case yw_void_word:
			len += 2;
			if (dst) {
				*dst++ = 'n';
				*dst++ = '\0';
			}
			break;
		default:
			yw_halt();
		}

	return len;
}

/**
 * {simple: yw/sock.h: send packet to peer}
 * {this} sends packet specified in <a>pkt</a> to peer along 
 * <a>conn</a>. This is done without any interaction with queued 
 * packets. I.e. you can queue packets 2, 3, send packet 1, and 
 * queue 4, and peer will get packet 1 first, and packets 2, 3, 4 
 * at once, after flush on your side.  
 * {see: yw_conn_queue(3), yw_conn_send_and_free(3)}
 */
void yw_conn_send(YwConnection *conn, YwPacket *pkt)
{
	uint8_t *buf;
	int len;

	len = write_packet(0, pkt);
	buf = yw_malloc(len);
	write_packet(buf, pkt);

	conn->write(conn, buf, len);

	yw_free(buf);
}

/**
 * {simple: yw/sock.h: queue packet to send it to peer later}
 * {this} queues packet specified in <a>pkt</a> in order
 * to be send to peer along <a>conn</a>. yw_conn_queue_flush(3)
 * should be used to actually send queued packets.
 * Queueing packets serves efficiency of connection, i.e. when
 * compression is used, they are compressed altogether, which works
 * much better and faster then compressing them separatly.
 * Each YwConnection has separate queue.
 * {see: yw_conn_queue_and_free(3), yw_conn_queue_flush(3), yw_conn_send(3)}
 */
void yw_conn_queue(YwConnection *conn, YwPacket *pkt)
{
	int len, n;
	
	len = write_packet(0, pkt);
	n = conn->pkt_queue_size + len;
	conn->pkt_queue = yw_realloc(conn->pkt_queue, n);
	write_packet(conn->pkt_queue + conn->pkt_queue_size, pkt);
	conn->pkt_queue_size = n;
}

/**
 * {simple: yw/sock.h: flush (send) queued packets}
 * {this} sends queued packets to peer along <a>conn</a>.
 * {see: yw_conn_queue(3), yw_conn_send(3)}
 */
void yw_conn_queue_flush(YwConnection *conn)
{
	if (conn->pkt_queue == NULL)
		return;

	conn->write(conn, conn->pkt_queue, conn->pkt_queue_size);
	
	conn->pkt_queue = NULL;
	conn->pkt_queue_size = 0;
}

/**
 * {simple: yw/sock.h: queue and free packet to send it to peer later}
 * {this} does exactly the same as yw_conn_queue(3), but after packet
 * is queued, it's released using yw_packet_free(3) function.
 * {see: yw_conn_queue(3), yw_conn_queue_flush(3), yw_conn_send_and_free(3)}
 */
void yw_conn_queue_and_free(YwConnection *conn, YwPacket *pkt)
{
	yw_conn_queue(conn, pkt);
	yw_packet_free(pkt);
}

/**
 * {simple: yw/sock.h: send and free packet to peer}
 * {this} does exactly the same as yw_conn_send(3), but after packet
 * is send, it's released using yw_packet_free(3) function.
 * {see: yw_conn_send(3), yw_conn_queue_and_free(3)}
 */
void yw_conn_send_and_free(YwConnection *conn, YwPacket *pkt)
{
	yw_conn_send(conn, pkt);
	yw_packet_free(pkt);
}
