/* $Id: recv.c,v 1.12 2001/05/11 10:38:05 malekith Exp $ */

#include <yw/util.h>
#include <yw/sock.h>
#include <yw/int/conn.h>
#include <yw/int/calls.h>
#include <yw/int/packet.h>

#include <string.h>

typedef struct YwRecvState_struct YwRecvState;

struct YwRecvState_struct {
	uint8_t *buf;
	int size;	/* of buffer */
	int len;	/* of data */
};

#if YW_HOST_ENDIAN == YW_LITTLE_ENDIAN
# include <netinet/in.h>
#endif

/**
 * {simple: yw/util.h: swap bytes in 32bit words}
 * {this} swaps <a>n</a> 32 bit words pointed by <a>ptr</a>
 * converting between big and little endian or the other way
 * around.
 * If words were in little endian order, after treating with
 * this function they are in big endian, and vice versa.
 * You may want to look at macro YW_HOST_ENDIAN which evaluates
 * to YW_LITTLE_ENDIAN or YW_BIG_ENDIAN, which are defined in
 * &lt;<filename>yw/config.h</filename>&gt; file (included by all
 * yw&slash;*.h headers). Note that PDP endian is not supported (we're in
 * XXI centry, you know :^).
 */
void yw_swap_bytes(uint32_t *ptr, int n)
{
#if YW_HOST_ENDIAN == YW_LITTLE_ENDIAN
	/* use ntohl() since it seems to be optimized */
	while (n--) {
		*ptr = ntohl(*ptr);
		ptr++;
	}
#else
	/* generic version, it probably won't be used 
	 * very often... */
	while (n--) {
		*ptr = (((*ptr >> 24) & 0xff) <<  0) |
		       (((*ptr >> 16) & 0xff) <<  8) |
		       (((*ptr >>  8) & 0xff) << 16) | 
		       (((*ptr >>  0) & 0xff) << 24); 
		ptr++;
	}
#endif
}

static uint32_t get_int(uint8_t *p)
{
	register uint32_t r;

	r = *p++;
	r <<= 8;
	r |= *p++;
	r <<= 8;
	r |= *p++;
	r <<= 8;
	r |= *p;
	
	return r;
}

static int count_len(YwRecvState *ctx, int n)
{
	int p, i;
	
	p = 6;

	for (i = 0; i < n; i++) {
		if (p >= ctx->len)
			break;
		if (ctx->buf[p++] & ' ') {
			if (p >= ctx->len)
				break;
			p += ctx->buf[p] + 1;
		} else {
			if (p + 3 >= ctx->len)
				break;
			p += get_int(ctx->buf + p) + 4;
		}
		if (p > ctx->len)
			break;
	}
	
	if (i < n)
		return -1;
	
	return p;
}

static YwWord *add(YwConnection *conn, int type, uint8_t *data, int len)
{
	YwWord *w = NULL;
	
	if (conn->state)
		return NULL;
		
	switch (type) {
	case 'i':
		w = yw_word_new_int(get_int(data));
		break;
	case 'n':
		w = yw_word_new_void();
		break;
	case 'b':
		w = yw_word_new_bindata(data, len);
		break;
	case 'k':
		/* hack... */
		w = yw_word_new_void();
		w->val.ascii = yw_malloc(len + 1);
		memcpy(w->val.ascii, data, len);
		w->val.ascii[len] = 0;
		w->type = yw_key_word;
		break;
	case 's':
		w = yw_word_new_void();
		w->val.string.chars = yw_malloc(len + sizeof(uint32_t));
		w->val.string.len = len / sizeof(uint32_t);
		memcpy(w->val.string.chars, data, len);
		w->val.string.chars[w->val.string.len] = 0;
		w->type = yw_string_word;
#if YW_HOST_ENDIAN != YW_BIG_ENDIAN
		yw_swap_bytes(w->val.string.chars, w->val.string.len);
#endif
		break;
	default:
		w = yw_word_new_bad();
		break;
	}

	return w;
}

static YwPacket *make_packet(YwConnection *conn, int type, int n, int *len)
{
	int p, i, c, s;
	YwPacket *pkt;
	YwRecvState *ctx;
	YwWord *w;
	
	p = 6;
	pkt = yw_packet_new(type);
	ctx = conn->recv_state;

	for (i = 0; i < n; i++) {
		c = ctx->buf[p++];

		if ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z')) {
			if (c & ' ')
				s = ctx->buf[p++];
			else {
				s = get_int(ctx->buf + p);
				p += 4;
			}
			c |= ' ';
			
			w = add(conn, c, ctx->buf + p, s);
			if (w)
				yw_packet_append_word_no_copy(pkt, w);
			p += s;
			continue;
		}
		
		if (conn->state == 0)
			yw_int_set_err(conn, YW_ERR_PROTO);
		yw_packet_free(pkt);
		*len = p;
		return NULL;
	}

	*len = p;
	return pkt;
}

static YwPacket *try_construct(YwConnection *conn)
{
	int c, n, len, len2;
	YwRecvState *ctx;
	YwPacket *pkt;

	if (conn->state)
		return NULL;
	
	ctx = conn->recv_state;

	if (ctx->len < 6)
		return NULL;
	
	c = ctx->buf[0];

	switch (c) {
	case yw_call_packet:
	case yw_void_call_packet:
	case yw_reply_packet:
		break;
	case -1:
		yw_int_set_err(conn, YW_ERR_CONN_LOST);
		return NULL;
	default:
		/* XXX: we bomb in case of unknown call type... 
		 * possibly some sync error. */
		yw_int_set_err(conn, YW_ERR_PROTO);
		return NULL;
	}
	
	if (ctx->buf[1] != 4) {
		yw_int_set_err(conn, YW_ERR_PROTO);
		return NULL;
	}
	
	n = get_int(ctx->buf + 2);
	
	len = count_len(ctx, n);
	if (len == -1)
		return NULL;
	len2 = -1;
	pkt = make_packet(conn, c, n, &len2);

	yw_assert(len == len2);

	if (ctx->len != len) {
		ctx->len -= len;
		yw_memmove(ctx->buf, ctx->buf + len, ctx->len);
	} else
		ctx->len = 0;

	return pkt;
}

static YwPacket *recv_packet(YwConnection *conn, int timeout)
{
	YwPacket *pkt = NULL;
	int n;
	YwRecvState *ctx;

	if (conn->state)
		return NULL;
	
	ctx = conn->recv_state;
	if (ctx == NULL) {
		ctx = conn->recv_state = YW_NEW_0(YwRecvState);
		ctx->buf = yw_malloc(ctx->size = 2048);
	}

	if (ctx->len && (pkt = try_construct(conn)))
	    	return pkt;
		
	if (timeout != -1 && !conn->can_read(conn, timeout))
		return NULL;

	while (conn->state == 0) {
		if (ctx->size == ctx->len)
			ctx->buf = yw_realloc(ctx->buf, ctx->size += 2048);
			
		n = conn->read(conn, ctx->buf + ctx->len, ctx->size - ctx->len);
		
		if (n < 0)
			break;
			
		ctx->len += n;
	
		pkt = try_construct(conn);

		if (pkt == NULL && conn->can_read(conn, 0))
			continue;
			
		if (pkt || timeout != -1)
			return pkt;
	}

	return NULL;
}

/**
 * {simple: yw/sock.h: recive packet from peer}
 * {this} waits at most <a>timeout</a> miliseconds to fetch
 * packet from <a>conn</a>. The returned packet has to be
 * released using yw_packet_free(3), when it's no longer
 * needed. If <a>timeout</a> is <literal>-1</literal>
 * <function>yw_conn_recv()</function> blocks until packet is 
 * recived or error occurs.
 * {retval} The recived packet or NULL in case of error or
 * timeout.
 */
YwPacket *yw_conn_recv(YwConnection *conn, int timeout)
{
	YwPacket *pkt;

	if (conn->pkt_waiting) {
		pkt = conn->pkt_waiting;
		conn->pkt_waiting = pkt->next;
		return pkt;
	}
	
	if (conn->state)
		return NULL;
	
	return recv_packet(conn, timeout);
}

/**
 * {simple: yw/sock.h: recive reply packet from peer}
 * This function recives packet from peer along <a>conn</a> within
 * <a>timeout</a> miliseconds (see yw_conn_recv(3) for discussion about it).
 * However, when packet, which is not reply is recived, it's queued, to
 * be fetched uppon next call to yw_conn_recv(3). When reply packet is
 * fetched, it's returned immidietly. {this} can take more then <a>timeout</a>
 * miliseconds to complete, when there are packets avaialble which come in less
 * then <a>timeout</a> ms, but which are not of reply type.
 * {retval} The recived packet or NULL in case of error or
 * timeout.
 */
YwPacket *yw_conn_recv_reply(YwConnection *conn, int timeout)
{
	YwPacket *pkt, *p;

	for (;;) {
		pkt = recv_packet(conn, timeout);
		if (pkt == NULL)
			break;
		if (yw_packet_type(pkt) == yw_reply_packet)
			break;

		if (conn->pkt_waiting == NULL)
			conn->pkt_waiting = pkt;
		else {
			for (p = conn->pkt_waiting; p->next; p = p->next)
				/* nothing */ ;
			p->next = pkt;
		}
	}

	return pkt;
}
