/* $Id: random.c,v 1.6 2001/05/05 11:36:18 dobrek Exp $ */

/*
 * ripped from cftp (my project) --mal
 */
 
#include <yw/random.h>
#include <yw/util.h>
#include <yw/lock.h>
#include <string.h>

#define DIGEST_SIZE 16

/* have to be (POOL_SIZE%DIGEST_SIZE) == 0 */
#define POOL_SIZE 512

struct entropy_pool {
	uint8_t pool[POOL_SIZE + DIGEST_SIZE];
	int index;
	double last_time;
};

static struct entropy_pool oep;
static struct entropy_pool *ep = &oep;

static YwLock locked = YW_LOCK_INIT;

#define lock() yw_lock(&locked)
#define unlock() yw_unlock(&locked)

static void round(void)
{
	register int n;
	register uint8_t *src, *dst;
	
	yw_count_md5(ep->pool + POOL_SIZE, ep->pool, POOL_SIZE + DIGEST_SIZE);
	
	n = DIGEST_SIZE;
	src = ep->pool + POOL_SIZE;
	dst = ep->pool + ep->index;
	while (n--)
		*dst++ ^= *src++;
	ep->index += DIGEST_SIZE;
	ep->index %= POOL_SIZE;
}

/**
 * {simple: yw/random.h: add entropy to pool}
 * {this} adds <a>len</a> bytes of data, pointed at <a>buf</a>
 * to random entropy pool. It's advaiceable to call it from time
 * to time, with some user supplied data (like current time after
 * keystroke, or sth like that), to get trurly random results 
 * from yw_random_get(3) and other. Note that used hashing function
 * (MD5) allows you to add any data (including confidential or
 * predictable one, like output of cat /dev/zero :^), while having no visible
 * connection with results of random fetching function.
 * Also note, that random pool is initialized on startup with output
 * of <filename>/dev/urandom</filename>, whenever possible, current
 * time, and content of <filename>~/.Yrandseed</filename> file if present.
 * <filename>~/.Yrandseed</filename> is updated on each startup.
 * All random handling functions are threadsafe -- i.e. only one thread can
 * fetch/add to pool at once, which is enforced by mutex.
 * {see: yw_random_get(3)}
 */
void yw_random_add(const void *buf, int len)
{
	lock();
	for (; len > 0; buf = (uint8_t*)buf + DIGEST_SIZE) {
		memcpy(ep->pool + POOL_SIZE, buf, len < DIGEST_SIZE ? 
							len : DIGEST_SIZE);
		round();
		len -= DIGEST_SIZE;
	}
	unlock();
}

/**
 * {simple: yw/random.h: fetch entropy from pool}
 * {this} fetches <a>len</a> bytes of random data to buffer
 * at <a>buf</a>.
 * All random handling functions are threadsafe -- i.e. only one thread can
 * fetch/add to pool at once, which is enforced by mutex.
 * {see: yw_random_add(3), yw_random_byte(3), yw_random_int(3)}
 */
void yw_random_get(void *buf, int len)
{
	uint8_t *ptr = buf;
	lock();
	for (; len > 0; ptr += DIGEST_SIZE, len -= DIGEST_SIZE) {
		round();
		memcpy(ptr, ep->pool + POOL_SIZE, len < DIGEST_SIZE ? 
							len : DIGEST_SIZE);
	}
	unlock();
}

/**
 * {simple: yw/random.h: fetch one byte of entropy from pool}
 * {this} fetches one byte of random data.
 * All random handling functions are threadsafe -- i.e. only one thread can
 * fetch/add to pool at once, which is enforced by mutex.
 * {retval} Random byte.
 * {see: yw_random_add(3), yw_random_get(3), yw_random_int(3)}
 */
int yw_random_byte(void)
{
	uint8_t r;
	yw_random_get(&r, 1);
	return r;
}

/**
 * {simple: yw/random.h: fetch 4 bytes of entropy from pool}
 * {this} fetches 4 bytes of random data.
 * {retval} Random 32 bit unsigned integer formed from 4 bytes
 * from pool.
 * All random handling functions are threadsafe -- i.e. only one thread can
 * fetch/add to pool at once, which is enforced by mutex.
 * {see: yw_random_add(3), yw_random_get(3), yw_random_byte(3)}
 */
uint32_t yw_random_int(void)
{
	uint32_t r;
	yw_random_get(&r, 4);
	return r;
}
