/* $Id: packet.c,v 1.10 2001/05/21 09:58:51 malekith Exp $ */

#include <yw/util.h>
#include <yw/int/packet.h>
#include <string.h>

/**
 * {simple: yw/packet.h: make new empty packet}
 * {this} constructs new empty packet of type <a>type</a>.
 * <a>type</a> argument is of type YwPacketType, defined as follows:
 * <programlisting>
 *    typedef enum {
 *            yw_call_packet         = 'c',
 *            yw_void_call_packet    = 'v',
 *            yw_reply_packet        = 'r',
 *            yw_taglist_packet      = 't'
 *    } YwPacketType;
 * </programlisting>
 * When type argument is invalid this function bombs.
 * {retval} Pointer to newly allocated packet is returned. It should be
 * released using yw_packet_free(3) when no longed needed.
 * {see: yw_packet_make(3), yw_packet_free(3), yw_packet_copyof(3)}
 */
YwPacket *yw_packet_new(YwPacketType type)
{
	YwPacket *ret;

	ret = YW_NEW_0(YwPacket);

	switch (type) {
	case yw_call_packet:
	case yw_void_call_packet:
	case yw_reply_packet:
	case yw_taglist_packet:
		break;
	default:
		yw_halt();
	}
	
	ret->type = type;

	return ret;
}

/**
 * {simple: yw/packet.h: make copy of given packet}
 * {this} copies packet in <a>src</a> into freshly allocated
 * space.
 * {retval} Pointer to newly allocated packet, which is exact
 * copy of <a>src</a> is returned. It should be
 * released using yw_packet_free(3) when no longed needed.
 * {see: yw_packet_make(3), yw_packet_free(3), yw_packet_new(3)}
 */
YwPacket *yw_packet_copy_of(const YwPacket *src)
{
	YwPacket *r;

	r = yw_packet_new(src->type);
	yw_packet_append_words(r, src->words, -1);

	return r;
}

#if 0
YwWord *yw_int_append_word(YwPacket *pkt)
{
	YwWord *ret, *p;

	ret = YW_NEW_0(YwWord);

	if (pkt->words == 0)
		pkt->words = ret;
	else {
		for (p = pkt->words; p->next; p = p->next)
			/* nothing */ ;
		p->next = ret;
	}

	pkt->len++;

	return ret;
}
#endif

static void add_word(YwPacket *pkt, YwWord *w)
{
	YwWord *p;

	if (pkt->words == 0)
		pkt->words = w;
	else {
		for (p = pkt->words; p->next; p = p->next)
			/* nothing */ ;
		p->next = w;
	}

	pkt->len++;
}

void yw_packet_set_type(YwPacket *pkt, YwPacketType t)
{
	pkt->type = t;
}

/**
 * {simple: yw/packet.h: append keyword at the end of packet}
 * {this} appends one word of type <symbol>yw_key_word</symbol>
 * at the end of packet <a>pkt</a>. Word to be appended is taken
 * from '\0' terminated string at <a>keyword</a>. Argument is
 * copied, so <a>keyword</a> is not required by
 * <function>yw_packet_append_keyword()</function> to be valid,
 * after it returns (i.e. you can free it after
 * appending, and before sending the packet).
 * {see: yw_packet_append_string(3), yw_make_packet(3)}
 */
void yw_packet_append_keyword(YwPacket *pkt, const char *keyword)
{
	add_word(pkt, yw_word_new_keyword(keyword));
}

/**
 * {simple: yw/packet.h: append integer at the end of packet}
 * {this} appends one word of type <symbol>yw_int_word</symbol>
 * at the end of packet <a>pkt</a>. Appended word contains
 * integer <a>i</a>.
 * {see: yw_packet_append_ptr(3), yw_make_packet(3)}
 */
void yw_packet_append_int(YwPacket *pkt, int32_t i)
{
	add_word(pkt, yw_word_new_int(i));
}

/**
 * {simple: yw/packet.h: append void word at the end of packet}
 * {this} appends one word of type <symbol>yw_void_word</symbol>
 * at the end of packet <a>pkt</a>. Appended word doesn't contain
 * any data, except it's type.
 * {see: yw_packet_append_ptr(3), yw_make_packet(3)}
 */
void yw_packet_append_void(YwPacket *pkt)
{
	add_word(pkt, yw_word_new_void());
}

/**
 * {simple: yw/packet.h: append pointer at the end of packet}
 * {this} appends one word of type <symbol>yw_ptr_word</symbol>
 * at the end of packet <a>pkt</a>. Appended word contains
 * pointer <a>ptr</a>. Note that packets that contain ptr words
 * cannot be send over the network, they are only valid on
 * loopback. If you try send such broken packet over the network
 * yw_conn_send(3) will bomb.
 * {see: yw_packet_append_bindata(3), yw_make_packet(3)}
 */
void yw_packet_append_ptr(YwPacket *pkt, void *ptr)
{
	add_word(pkt, yw_word_new_ptr(ptr));
}

/**
 * {simple: yw/packet.h: append string at the end of packet}
 * {this} appends one word of type <symbol>yw_string_word</symbol>
 * at the end of packet <a>pkt</a>. Word to be appended is taken
 * from UNICODE string at *<a>str</a>. Argument is
 * copied, so <a>str</a> is not required by
 * <function>yw_packet_append_string()</function> to be valid,
 * after it returns (i.e. you can yw_string_free(3) it after
 * appending, and before sending the packet).
 * {see: yw_packet_append_keyword(3), yw_make_packet(3)}
 */
void yw_packet_append_string(YwPacket *pkt, const YwString *str)
{
	add_word(pkt, yw_word_new_string(str));
}

/**
 * {simple: yw/packet.h: append binary data word at the end of packet}
 * {this} appends one word of type <symbol>yw_bindata_word</symbol>
 * at the end of packet <a>pkt</a>. Word to be appended is taken
 * will contain copy of <a>len</a> bytes pointed by <a>ptr</a>.
 * You should note, that bindata words are to be used mainly
 * for data which are in fact binary, and hasn't got much internal
 * structure, like bitmaps, sound samples etc. Using it for
 * passing data structures is not sign of good style.
 * {see: yw_make_packet(3)}
 */
void yw_packet_append_bindata(YwPacket *pkt, const void *ptr, int len)
{
	add_word(pkt, yw_word_new_bindata(ptr, len));
}

/**
 * {simple: yw/packet.h: append copy of word at the end of packet}
 * {this} appends one word, which is exact copy of <a>word</a>
 * parameter.
 * {see: yw_make_packet(3)}
 */
void yw_packet_append_word(YwPacket *pkt, const YwWord *word)
{
	add_word(pkt, yw_word_copy_of(word));
}

/**
 * {simple: yw/packet.h: append word at the end of packet}
 * {this} appends <a>word</a> parameter. It is not copied.
 * You cannot free nor changed it anymore.
 * {see: yw_make_packet(3)}
 */
void yw_packet_append_word_no_copy(YwPacket *pkt, YwWord *word)
{
	add_word(pkt, word);
}

/**
 * {simple: yw/packet.h: append bunch of words at the end of a packet}
 * {this} appends <a>len</a> words
 * at the end of packet <a>pkt</a>. First word to be appended is taken
 * from *<a>src</a>, which is head of linked list of words.
 * If <a>len</a> is <literal>-1</literal> all available words
 * are appended. If <a>src</a> isn't long enough, only available
 * words are appended.
 * {see: yw_packet_copyof(3), yw_make_packet(3)}
 */
void yw_packet_append_words(YwPacket *pkt, const YwWord *src, int len)
{
	while (src && (len == -1 || len--)) {
		add_word(pkt, yw_word_copy_of(src));
		src = src->next;
	}
}

/**
 * {simple: yw/packet.h: make entire packet at once}
 * {this} is quite complex. First packet of <a>type</a> is
 * created (see yw_packet_new(3), for discusion about <a>type</a>
 * argument). Then arguments of char* type are processed
 * in order, they were passed. They specify words to be
 * appended. Type of word is determined based of it's first
 * character. When there is more then one character
 * additional word of type <symbol>yw_key_word</symbol> 
 * is appended first. This means, that 
 * <userinput>yw_packet_make(yw_call_packet, "k", "hello", "i", 
 * 10, YW_END)</userinput> and <userinput>yw_packet_make(yw_call_packet, 
 * "i" "hello", 10, YW_END)</userinput> are the same. 
 * This feature comes in handy when dealing with tag lists.
 * Then type-depended arguments are poped off the stack,
 * and processing restarts with new char* argument.
 * Words are appended until <symbol>YW_END</symbol>
 * is found (which is simply <literal>((char*)0)</literal>).
 * Following type characters are supported (format: type character,
 * type of appended word, arguments. Meaning of word appending
 * functions is same as meaning of yw_string_assign_* functions,
 * i.e. encoding can be NULL (use default), and length can be -1 
 * (compute it)):
 * <itemizedlist>
 *  <listitem><para>
 *    <symbol>"i"</symbol> - yw_int_word. arg: int. 
 *  </para></listitem><listitem><para>
 *    <symbol>"k"</symbol> - yw_key_word. arg: const char*.
 *  </para></listitem><listitem><para>
 *    <symbol>"p"</symbol> - yw_ptr_word. arg: void*.
 *  </para></listitem><listitem><para>
 *    <symbol>"v"</symbol> - yw_void_word. arg: none.
 *  </para></listitem><listitem><para>
 *    <symbol>"s"</symbol> - yw_string_word. arg: const YwString *.
 *  </para></listitem><listitem><para>
 *    <symbol>"u"</symbol> - yw_string_word. arg: const uint32_t *, int.
 *  </para></listitem><listitem><para>
 *    <symbol>"x"</symbol> - yw_string_word. arg: const uint16_t *, int.
 *  </para></listitem><listitem><para>
 *    <symbol>"8"</symbol> - yw_string_word. arg: const uint8_t *, int.
 *  </para></listitem><listitem><para>
 *    <symbol>"e"</symbol> - yw_string_word. arg: const char*, 
 *    const char*, int. First arg is encoding (can be NULL), second -
 *    characters, and the last one - length (can be -1 -- compute length).
 *  </para></listitem><listitem><para>
 *    <symbol>"g"</symbol> - yw_string_word. arg: const uint8_t*. Same
 *    as '8', arg, -1.
 *  </para></listitem><listitem><para>
 *    <symbol>"d"</symbol> - yw_string_word. arg: const char *. Same
 *    as 'e', NULL, arg, -1.
 *  </para></listitem>
 * </itemizedlist>
 * {retval} Pointer to freshly created packet is returned, or NULL
 * in case of error (i.e. bad character for given encoding, or
 * sth like that. This functions bombs when there is error in
 * passed arguments).
 * {sect:EXAMPLES}
 * Here you can find some examples of use:
 * <programlisting>
 *   yw_packet_make(yw_taglist_packet,
 *   			"i" "width", 20,
 *   			"i" "height", 10,
 *   			"d" "title", "Hello world",
 *   			YW_END);
 *   			
 *   yw_packet_make(yw_call_packet,
 *   			"k", "frob", YW_END);
 *   			
 *   yw_packet_make(yw_call_packet,
 *   			"k", "frob_with_tags", 
 *   			"i" "foo", 20,
 *   			"v" "bumzor_present", 
 *   			&slash;* void word appended here ^^^ *&slash; 
 *   			"s" "bar", &amp;str,
 *   			&slash;* assume _() to return UTF8: *&slash;
 *   			"g" "baz", _("Hello world!"), 
 *   			&slash;* or maybe it would be wiser to assume 
 *   			 * _() to return current default 
 *   			 * encoding?? *&slash;
 *   			"d" "baz_local", _("Hmm... Locale?"),
 *   			&slash;* use longer versions *&slash;
 *   			"k", "ala_ma", "d", "kota",
 *   			"k", "kot_ma", "i", 20,
 *   			YW_END);
 * </programlisting>
 * {bugs} Confusing manpage.
 */
YwPacket *yw_packet_make(YwPacketType type, ...)
{
	va_list ap;
	YwPacket *ret;

	va_start(ap, type);
	ret = yw_packet_make_v(type, ap);
	va_end(ap);

	return ret;
}


/**
 * {simple: yw/packet.h: make entire packet at once}
 * {this} is same as yw_packet_make(3) except that it takes
 * arguments vector instead of varargs.
 * {retval} Pointer to freshly created packet is returned, or NULL
 * in case of error (i.e. bad character for given encoding, or
 * sth like that. This functions bombs when there is error in
 * passed arguments).
 */
YwPacket *yw_packet_make_v(YwPacketType type, va_list ap)
{
	YwPacket *pkt;
	YwString str;
	const char *tag;
	int err = 0;
	int len;

	pkt = yw_packet_new(type);
	
	for (tag = va_arg(ap, const char *); 
	     tag; 
	     tag = va_arg(ap, const char *)) {
	     	if (tag[1])
			yw_packet_append_keyword(pkt, tag + 1);
			
		str.chars = NULL;	
		
	     	switch (*tag) {
		case 'i': /* int data */
			yw_packet_append_int(pkt, va_arg(ap, int));
			break;
		case 'k': /* const char *data */
			yw_packet_append_keyword(pkt, va_arg(ap, const char *));
			break;
		case 's': /* const YwString * (string in default enc) */
			yw_string_assign_string(&str, 
					va_arg(ap, const YwString *));
			break;
		case 'u': /* const uint32_t *data (utf32), int len */
		{
			const uint32_t *ptr = va_arg(ap, const uint32_t *);
			len = va_arg(ap, int);
			if (yw_string_assign_utf32(&str, ptr, len))
				err++;
			break;
		}
		case 'x': /* const uint16_t *data (utf16), int len */
		{
			const uint16_t *ptr = va_arg(ap, const uint16_t *);
			len = va_arg(ap, int);
			if (yw_string_assign_utf16(&str, ptr, len))
				err++;
			break;
		}
		case '8': /* const uint8_t *data (utf8), int len */
		{
			const uint8_t *ptr = va_arg(ap, const uint8_t *);
			len = va_arg(ap, int);
			if (yw_string_assign_utf8(&str, ptr, len))
				err++;
			break;
		}
		case 'e': /* const char *enc, const char *chars, int len */
		{
			const char *enc = va_arg(ap, const char *);
			const char *ptr = va_arg(ap, const char *);
			len = va_arg(ap, int);
			if (yw_string_assign_cstring(&str, enc, ptr, len))
				err++;
			break;
		}
		case 'g': /* const uint8_t *data (utf8), compute length, */
			if (yw_string_assign_utf8(&str, 
					va_arg(ap, const uint8_t *), -1))
				err++;
			break;
		case 'd': /* char *str, in default encoding, NUL term. */
			if (yw_string_assign_cstring(&str, NULL,
					va_arg(ap, const char *), -1))
				err++;
			break;
		case 'p': /* void *ptr */
			yw_packet_append_ptr(pkt, va_arg(ap, void*));
			break;
		case 'v': /* nothing */
			yw_packet_append_void(pkt);
			break;
		default:
			yw_halt();
		}

		if (str.chars) {
			yw_packet_append_string(pkt, &str);
			yw_string_free(&str);
		}
		
		if (err) {
			yw_packet_free(pkt);
			return NULL;
		}
	}

#if 0
	yw_assert(pkt->len > 0);
#endif

	return pkt;
}

/**
 * {simple: yw/packet.h: release packet and associated resources}
 * {this} releases all memory occupied by packet <a>pkt</a> and
 * its words. All values retrived using yw_packet_word(3),
 * yw_packet_fetch_string(3), yw_packet_fetch_keyword(3),
 * etc. from this very packet are from nowon invalid.
 */
void yw_packet_free(YwPacket *pkt)
{
	YwWord *w;

	if (pkt == NULL)
		return;

	while (pkt->words) {
		w = pkt->words;
		pkt->words = pkt->words->next;
		yw_word_free(w);
	}

	yw_free(pkt);
}

/**
 * {simple: yw/packet.h: get type of packet}
 * {this} retrives type of <a>pkt</a>.
 * {retval} Type of packet. See yw_packet_new(3) for disussion
 * about possible packet types.
 */
YwPacketType yw_packet_type(YwPacket *pkt)
{
	return pkt->type;
}

/**
 * {simple: yw/packet.h: get length of packet}
 * {this} retrives length of <a>pkt</a>.
 * {retval} Length of packet in words, including command word.
 */
int yw_packet_length(YwPacket *pkt)
{
	return pkt->len;
}

/**
 * {simple: yw/packet.h: get specified word of packet}
 * {this} retrives <a>nth</a> word of <a>pkt</a>. YwWord is
 * opaque pointer.
 * {retval} <a>nth</a> word of <a>pkt</a>.
 */
YwWord *yw_packet_word(YwPacket *pkt, int nth)
{
	YwWord *w;

	if (pkt == NULL)
		return NULL;

	for (w = pkt->words; w; w = w->next)
		if (nth-- <= 0)
			break;

	return w;
}
