/* $Id: map.c,v 1.5 2001/05/05 11:36:18 dobrek Exp $ */

#include <yw/map.h>
#include <yw/util.h>
#include <yw/hash.h>
#include <yw/lock.h>
#include <string.h>

/*
 * FIXME:
 * Hmmm... some sorted array an bsearch would be possibly better...
 */

struct entry {
	struct entry *next;
	char *key;
	int val;
};

struct YwMapCache_struct {
	YwHash *hash;
	int def;
};

static YwMapCache *make_cache(const YwMapEntry *entries)
{
	struct entry *e;
	YwHash *h;
	YwMapCache *c;

	h = yw_hash_new();
	
	while (entries->key) {
		e = yw_malloc(sizeof(struct entry) + strlen(entries->key) + 1);
		e->key = (char*)(e) + sizeof(struct entry);
		strcpy(e->key, entries->key);
		e->val = entries->value;
		yw_hash_add(h, e);
		entries++;
	}

	c = yw_malloc(sizeof(YwMapCache));
	c->hash = h;
	c->def = entries->value;

	return c;
}

static void free_entry(YwHashEntry *entry)
{
	yw_free(entry);
}

/**
 * {simple: yw/map.h: free keyword -> int mapping cache}
 * {this} frees resoures assiciated with *<a>cache</a>, and sets
 * it to NULL.
 */
void yw_map_free_cache(YwMapCache **cache)
{
	if (cache == NULL || *cache == NULL)
		return;
	yw_hash_free((*cache)->hash, free_entry);
	yw_free(*cache);
	*cache = NULL;
}

/**
 * {simple: yw/map.h: lookup string in keyword -> int mapping}
 * {this} searches <a>entries</a> for <a>key</a>, and returns value
 * field of entry, that matched. If <a>cache</a> is not NULL, it
 * is used to store cache of <a>entries</a>. YwMapEntry datatype
 * is defined as follows:
 * <programlisting>
 *     typedef struct YwMapEntry_struct YwMapEntry;
 *     struct YwMapEntry_struct {
 *             const char *key;
 *             int value;
 *     };
 * </programlisting>
 * Last entry has key NULL, it's value is returned when no other entry
 * matched. Example of use:
 * <programlisting>
 *     enum {
 *            ev_born,
 *            ev_eat,
 *            ev_die,
 *            ev_oops
 *     };
 *     
 *     int lookup(const char *key)
 *     {
 *            YwMapEntry e[] = {
 *                   { "born", ev_born },
 *                   { "eat", ev_eat },
 *                   { "die", ev_die },
 *                   { NULL, ev_oops }
 *            };
 *            static YwMapCache *cache = NULL;
 *     
 *            return yw_map(key, e, &amp;cache);
 *     }
 * </programlisting>
 * Note, that <userinput>cache</userinput> is initialized to NULL,
 * before passing to yw_map, it is also static, so caching makes sense :^)
 * In this example cache is never released -- it's OK for most cases, when
 * caches are static variables, used throughout lifetime of a program.
 * This function is multithread safe -- i.e. concurently running threads
 * won't create *<a>cache</a> at the same time.
 *
 * {retval} value of entry, that matched <a>key</a>, or value of entry with
 * key of NULL, when no entry matched.
 *
 * {see: yw_map_free_cache(3)}
 */
int yw_map(const char *key, const YwMapEntry *entries, YwMapCache **cache)
{
	struct entry *e;
	int r;
	static YwLock lock = YW_LOCK_INIT;

	if (cache) {
		yw_lock(&lock);
		if (*cache == NULL)
			*cache = make_cache(entries);
		e = yw_hash_find((*cache)->hash, key);
		r = e ? e->val : (*cache)->def;
		yw_unlock(&lock);
		return r;
	}

	/* otherwise no locking needed */
	while (entries->key) {
		if (strcmp(entries->key, key) == 0)
			return entries->value;
		entries++;
	}

	return entries->value;
}
