/* $Id: lock.c,v 1.4 2001/05/04 11:04:23 malekith Exp $ */

#include <yw/lock.h>
#include <yw/util.h>

#ifdef YW_HAS_PTHREAD
# define lock(m) pthread_mutex_lock(&m)
# define unlock(m) pthread_mutex_unlock(&m)
# define init(m) pthread_mutex_init(&m, NULL)
# define destroy(m) pthread_mutex_destroy(&m)
#else
# if 0
/* poor man's locking, disabled, as it is not safe... */
#  define lock(m) 			\
	do { 				\
		while (m)		\
			yw_sleep(10);	\
		m++;			\
	} while (0)
#  define unlock(m) m--
# else
#  define lock(m) do {} while (0)
#  define unlock(m) do {} while (0)
# endif
# define init(m) (m = 0)
# define destroy(m) (m = -1)
#endif


/**
 * {simple: yw/lock.h: initialize YwLock structure}
 * {this} initializes YwLock pointed to by <a>l</a> parameter.
 * YwLock is kind of mutual exclusion device a.k.a. mutex.
 * LibYW implements read/write (or shared/exclusive) locks.
 * I.e. just one thread can hold write lock on given mutex,
 * but when no thread holds write lock, any number of threads
 * can hold read lock.
 * {note} In current implementation locks are implemented on
 * the base of pthread mutexes. When they are not available,
 * locking won't work (i.e. locking will always suceed).
 * As a result -- no function in LibYW will be multithread safe.
 * {see: yw_lock_ro(3), yw_lock_rw(3), yw_lock_destroy(3)}
 * {see: pthread_mutex_init(3)}
 */
void yw_lock_init(YwLock *l)
{
	l->rd_cnt = 0;
	init(l->m);
}

/**
 * {simple: yw/lock.h: release any resources associated with lock}
 * {this} frees any data associated with lock <a>l</a>.
 * {see: yw_lock_init(3), pthread_mutex_destroy(3)}
 */
void yw_lock_destroy(YwLock *l)
{
	destroy(l->m);
}

/**
 * {simple: yw/lock.h: wait for shared lock}
 * {this} waits until shared locking of <a>l</a> is possible,
 * and then, atomicly locks it.
 * {see: yw_lock_init(3), yw_unlock_ro(3), yw_lock_rw(3)}
 */
void yw_lock_ro(YwLock *l)
{
	lock(l->m);
	l->rd_cnt++;
	unlock(l->m);
}

/**
 * {simple: yw/lock.h: release shared lock}
 * {this} releases shared lock on <a>l</a>.
 * {see: yw_lock_init(3), yw_lock_ro(3), yw_unlock_rw(3)}
 */
void yw_unlock_ro(YwLock *l)
{
	l->rd_cnt--;
}

/**
 * {simple: yw/lock.h: wait for exclusive lock}
 * {this} waits until exclusive locking of <a>l</a> is possible,
 * and then, atomicly locks it.
 * {see: yw_lock_init(3), yw_unlock_rw(3), yw_lock_ro(3)}
 */
void yw_lock_rw(YwLock *l)
{
	lock(l->m);
	while (l->rd_cnt)
		yw_sleep(10);
	l->rd_cnt = -1;
}

/**
 * {simple: yw/lock.h: release exclusive lock}
 * {this} releases exclusive lock on <a>l</a>.
 * {see: yw_lock_init(3), yw_lock_rw(3), yw_unlock_ro(3)}
 */
void yw_unlock_rw(YwLock *l)
{
	l->rd_cnt = 0;
	unlock(l->m);
}

/**
 * {simple: yw/lock.h: wait for exclusive lock}
 * {this} is exactly the same as yw_lock_rw(3).
 * {see: yw_lock_init(3), yw_lock_rw(3), yw_unlock(3)}
 */
void yw_lock(YwLock *lock)
{
	yw_lock_rw(lock);
}

/**
 * {simple: yw/lock.h: release exclusive lock}
 * {this} is exactly the same as yw_unlock_rw(3).
 * {see: yw_lock_init(3), yw_unlock_rw(3), yw_lock(3)}
 */
void yw_unlock(YwLock *lock)
{
	yw_unlock_rw(lock);
}
