/* $Id: crypt.c,v 1.6 2001/05/05 11:36:18 dobrek Exp $ 
 *
 * The Arcfour cipher is believed to be compatible with the RC4 cipher. 
 * RC4 is a registered trademark of RSA Data Security Inc.
 */

#include <yw/util.h>
#include <yw/int/conn.h>
#include <string.h>

struct arcfour_ctx {
  uint8_t S[256];
  uint8_t i, j;
};


#define SWAP(a,b) do { int _t = a; a = b; b = _t; } while(0)

static void arcfour_set_key(struct arcfour_ctx *ctx, const uint8_t *key, uint32_t len)
{
  register uint8_t j; /* Depends on the eight-bitness of these variables. */
  unsigned i;
  uint32_t k;

  /* Initialize context */
  i = 0;
  do ctx->S[i] = i; while (++i < 256);

  /* Expand key */
  i = j = k = 0;
  do {
    j += ctx->S[i] + key[k];
    SWAP(ctx->S[i], ctx->S[j]);
    k = (k+1) % len; /* Repeat key if needed */
  } while(++i < 256);
  
  ctx->i = ctx->j = 0;
}

static void arcfour_crypt(struct arcfour_ctx *ctx, uint8_t *p, uint32_t len)
{
  register uint8_t i, j;

  i = ctx->i; j = ctx->j;
  while(len--)
    {
      i++; /* i &= 0xff; */
      j += ctx->S[i]; /* j &= 0xff; */
      SWAP(ctx->S[i], ctx->S[j]);
      *p++ ^= ctx->S[ (ctx->S[i] + ctx->S[j]) & 0xff ];
    }
  ctx->i = i; ctx->j = j;
}

struct arcfour_state {
	struct arcfour_ctx read_ctx;
	struct arcfour_ctx write_ctx;
	
	void *buf;
	int buf_size;

	int (*read)(YwConnection *conn, void *buf, int n);
	int (*write)(YwConnection *conn, const void *buf, int n);
};

static void init_keys(YwConnection *conn, struct arcfour_ctx *c0, 
		      struct arcfour_ctx *c1)
{
	uint8_t key0[YW_COOKIE_SIZE], key1[YW_COOKIE_SIZE];
	int i, n;
	uint8_t *p;

	i = 0;
	n = YW_COOKIE_SIZE/2;
	p = conn->authority;
	while (n--) {
		key0[i] = *p++;
		key1[i] = *p++;
		i++;
	}
	n = YW_COOKIE_SIZE/2;
	p = conn->cookie;
	while (n--) {
		key0[i] = *p++;
		key1[i] = *p++;
		i++;
	}
	
	arcfour_set_key(c0, key0, YW_COOKIE_SIZE);
	arcfour_set_key(c1, key1, YW_COOKIE_SIZE);
}

static int arcfour_read(YwConnection *conn, void *buf, int size)
{
	int r;
	struct arcfour_state *ctx;

	ctx = conn->crypt_state;
	r = ctx->read(conn, buf, size);
	if (r > 0)
		arcfour_crypt(&ctx->read_ctx, buf, r);

	return r;
}

static int arcfour_write(YwConnection *conn, const void *buf, int size)
{
	int r;
	struct arcfour_state *ctx;

	ctx = conn->crypt_state;
	
	if (size > ctx->buf_size)
		ctx->buf = yw_realloc(ctx->buf, size * 2);

	memcpy(ctx->buf, buf, size);
	
	arcfour_crypt(&ctx->write_ctx, ctx->buf, size);
	
	r = ctx->write(conn, ctx->buf, size);

	return r;
}

void yw_int_conn_crypt_free(YwConnection *conn)
{
	struct arcfour_state *ctx;
	
	if ((ctx = conn->crypt_state)) {
		yw_free(ctx->buf);
		yw_free(ctx);
	}
}

void yw_int_enable_encryption(YwConnection *conn)
{
	struct arcfour_state *ctx;

	ctx = YW_NEW_0(struct arcfour_state);

	ctx->read = conn->read;
	ctx->write = conn->write;

	if (conn->server)
		init_keys(conn, &ctx->read_ctx, &ctx->write_ctx);
	else
		init_keys(conn, &ctx->write_ctx, &ctx->read_ctx);

	conn->read = arcfour_read;
	conn->write = arcfour_write;
	conn->crypt_state = ctx;
}
