/* $Id: buffer.c,v 1.6 2001/05/24 14:53:57 malekith Exp $ */

#include <yw/util.h>
#include <yw/int/buffer.h>
#include <yw/map.h>

YwBuffer *yw_buffer_new(YwBufferCallback *callback, void *userdata,
			int w, int h, int syncable)
{
	YwBuffer *r;
	int i;

	r = YW_NEW_0(YwBuffer);

	r->w = w;
	r->h = h;
	r->cb = callback;
	r->cb_data = userdata;

	r->lines = yw_malloc(sizeof(Line) * h);
	for (i = 0; i < h; i++) {
		r->lines[i].chars = yw_malloc_0((w + 1) * sizeof(Char));
	}
	
	yw_buffer_reset_attr(r);
	yw_buffer_clr(r);
	
	yw_assert(callback->char_width); 
	
	if (syncable) {
		yw_assert(callback->remote_puts && 
			  callback->remote_attr &&
			  callback->remote_goto);
		r->second = yw_buffer_new(callback, userdata, w, h, 0);
	}
	
	r->nul_attr.fg = 7;
	r->nul_attr.bg = 0;
	r->nul_attr.flags = F_CHANGED;
	r->nul_attr.ch = ' ';

	return r;
}

void yw_buffer_free(YwBuffer *buf)
{
	int i;

	if (buf == NULL)
		return;
	
	yw_buffer_free(buf->second);
	
	for (i = 0; i < buf->h; i++) 
		yw_free(buf->lines[i].chars);
		
	yw_free(buf->lines);
	yw_free(buf);
}

int yw_buffer_get_x(YwBuffer *buf)
{
	return buf->x;
}

int yw_buffer_get_y(YwBuffer *buf)
{
	return buf->y;
}

void yw_buffer_goto(YwBuffer *buf, int x, int y)
{
	if (x >= 0 && x < buf->w)
		buf->x = x;
	if (y >= 0 && y < buf->h)
		buf->y = y;
	buf->last = NULL;
}

void yw_buffer_set_attr(YwBuffer *buf, YwBufAttr attr, int val)
{
	switch (attr) {
	case yw_bufattr_fg:
	case yw_bufattr_bg:
		yw_assert(val >= YW_COLOR_ANSI_BLACK && 
			  val <= YW_COLOR_ANSI_WHITE);
		if (attr == yw_bufattr_fg)
			buf->attr.fg = (uint8_t)(val & 0xff);
		else
			buf->attr.bg = (uint8_t)(val & 0xff);
		break;
	case yw_bufattr_bold:
		if (val)
			buf->attr.flags |= F_BOLD;
		else
			buf->attr.flags &= ~F_BOLD;
		break;
	case yw_bufattr_underline:
		if (val)
			buf->attr.flags |= F_UNDERLINE;
		else
			buf->attr.flags &= ~F_UNDERLINE;
		break;
	case yw_bufattr_reset:
		yw_buffer_set_fg(buf, YW_COLOR_ANSI_LIGHTGRAY);
		yw_buffer_set_bg(buf, YW_COLOR_ANSI_BLACK);
		buf->attr.flags = F_CHANGED;
		buf->attr.ch = ' ';
		break;
	default:
	case yw_bufattr_bad:
		break;
	}
}

void yw_buffer_set_fg(YwBuffer *buf, int color)
{
	yw_buffer_set_attr(buf, yw_bufattr_fg, color);
}

void yw_buffer_set_bg(YwBuffer *buf, int color)
{
	yw_buffer_set_attr(buf, yw_bufattr_bg, color);
}

void yw_buffer_set_bold(YwBuffer *buf, int on_off)
{
	yw_buffer_set_attr(buf, yw_bufattr_bold, on_off);
}

void yw_buffer_set_underline(YwBuffer *buf, int on_off)
{
	yw_buffer_set_attr(buf, yw_bufattr_underline, on_off);
}

void yw_buffer_reset_attr(YwBuffer *buf)
{
	yw_buffer_set_attr(buf, yw_bufattr_reset, 0);
}

static void kill_addons(Char *c)
{
	Char *n;
	
	if (c->addons == NULL)
		return;
		
	n = c->addons;
	c->addons = NULL;
	c->flags |= F_CHANGED;
		
	c = n;
	while (c) {
		n = c->addons;
		yw_free(c);
		c = n;
	}
}

static Char *will_write_at(YwBuffer *buf, int x, int y, int w)
{
	Char *p, *i;
	int k;

	/* check whatever we're at screen boundary, if so - refuse to work */
	if (x < 0 || y < 0 || y >= buf->h || x + w > buf->w)
		return NULL;
		
	p = buf->lines[y].chars + x;
	
	for (i = p; i->flags & F_NULL; i--)
		*i = buf->attr;
		
	if (p != i) {
		kill_addons(i);
		*i = buf->attr;
	}
		
	for (i = p + (w == 0 ? 1 : w); i->flags & F_NULL; i++)
		*i = buf->attr;

	for (k = 0; k < w; k++) {
		kill_addons(&p[k]);
		p[k] = buf->attr;
		p[k].flags |= F_NULL;
	}
		
	return p;
}

#define same_attr(a,b)	\
	((a)->fg == (b)->fg && (a)->bg == (b)->bg &&	\
	 ((a)->flags & F_DRAWABLE) == ((b)->flags & F_DRAWABLE))

static int same_addons(Char *a, Char *b)
{
	Char *p, *x;

	for (p = a->addons; p; p = p->addons) {
		for (x = b->addons; x; x = x->addons)
			if (x->ch == p->ch)
				break;
		if (x == NULL)
			return 0;
	}
	
	for (p = b->addons; p; p = p->addons) {
		for (x = a->addons; x; x = x->addons)
			if (x->ch == p->ch)
				break;
		if (x == NULL)
			return 0;
	}

	return 1;
}

void yw_buffer_putchar(YwBuffer *buf, uint32_t ch)
{
	int w;
	Char *p;

	w = buf->cb->char_width(buf->cb_data, ch);

	if (w == -2)
		return;

	if (w == 0) {
		if (buf->last == NULL)
			buf->last = will_write_at(buf, buf->x, buf->y, 0);
		for (p = buf->last; p->addons; p = p->addons)
			if (p->addons->ch == ch)
				break;
		if (p->addons == NULL) {
			p->addons = YW_NEW(Char);
			*p->addons = buf->attr;
			p->addons->ch = ch;
		}
		return;
	}
		
	p = will_write_at(buf, buf->x, buf->y, w);

	buf->last = p;
	buf->x += w;
	
	if (p == NULL)
		return;

	*p = buf->attr;
	p->ch = ch;
}

int yw_buffer_strlen(YwBuffer *buf, const YwString *str)
{
	int i, len = 0;

	if (buf->cb->cache_widths)
		buf->cb->cache_widths(buf->cb_data, str);
	for (i = 0; i < str->len; i++)
		len += buf->cb->char_width(buf->cb_data, str->chars[i]);
	return len < 0 ? -2 : len;
}

/* FIXME: maybe some faster implementations :> 
 * Everything below is quite generic... 
 * Also optimzation when updating is far from the ideal :^)
 */
void yw_buffer_clr(YwBuffer *buf)
{
	int i, j;
	
	for (i = 0; i < buf->h; i++) {
		yw_buffer_goto(buf, 0, i);
		for (j = 0; j < buf->w; j++)
			yw_buffer_putchar(buf, buf->attr.ch);
	}
	
	yw_buffer_goto(buf, 0, 0);
}

void yw_buffer_puts(YwBuffer *buf, const YwString *str)
{
	int i;

	if (buf->cb->cache_widths)
		buf->cb->cache_widths(buf->cb_data, str);
	for (i = 0; i < str->len; i++)
		yw_buffer_putchar(buf, str->chars[i]);
}

static void copy_addons(Char *d, const Char *s)
{
	while (s->addons) {
		d->addons = YW_NEW(Char);
		*(d->addons) = *(s->addons);
		d = d->addons;
		s = s->addons;
	}
}

static void copy_chars(YwBuffer *dst, int dst_x, int dst_y, 
	   	       const YwBuffer *src, int src_x, int src_y, int n) 
{
	Char *d, *s;
	int beg = 1;

	if (dst_y >= dst->h)
		return;

	n++;
	do {
		d = will_write_at(dst, dst_x, dst_y, --n);
	} while (d == NULL && n > 1);
	
	s = (src_y >= src->h || src_x >= src->w) ? 
				NULL : &src->lines[src_y].chars[src_x];
				
	while (dst_x < dst->w && n--) {
		if (s == NULL || (beg && (s->flags & F_NULL))) {
			*d = dst->attr;
		} else {
			beg = 0;
			*d = *s;
			copy_addons(d, s);
		}
		
		if (src_x++ >= src->w)
			s = NULL;
		else
			s++;
		d++;
		dst_x++;
	}
}

void yw_buffer_putbuffer(YwBuffer *dst, int dst_x, int dst_y, 
					int dst_w, int dst_h,
		   const YwBuffer *src, int src_x, int src_y) 
{
	int dy, y, x;
		
	y = src_y;
	
	for (dy = 0; dy < dst_h; dy++, y++) {
		copy_chars(dst, dst_x, dst_y + dy,
			   src, src_x, y, dst_w); 
	}
	
	x = dst_x + src->x - src_x;
	y = dst_y + src->y - src_y;

	if (x >= dst_x && x <= dst_x + dst_w &&
	    y >= dst_y && y <= dst_y + dst_h) {
	    	dst->x = x;
		dst->y = y;
	}
}

/* these are mainly useful for emulating vt-whatever :&) */
void yw_buffer_delline(YwBuffer *buf)
{
	int y;
	Char *p;

	p = buf->lines[buf->y].chars;
	
	for (y = buf->y; y < buf->h - 1; y++)
		buf->lines[y].chars = buf->lines[y+1].chars;
		
	buf->lines[y].chars = p;
	y = buf->w;
	while (y--) {
		kill_addons(p);
		*p++ = buf->nul_attr;
	}
}

void yw_buffer_insline(YwBuffer *buf)
{
	int y;
	Char *p;

	p = buf->lines[buf->h - 1].chars;
	
	for (y = buf->h - 1; y > buf->y; y--)
		buf->lines[y].chars = buf->lines[y-1].chars;
		
	buf->lines[buf->y].chars = p;
	y = buf->w;
	while (y--) {
		kill_addons(p);
		*p++ = buf->nul_attr;
	}
}

void yw_buffer_inschar(YwBuffer *buf)
{
	int x;
	Char *p;

	will_write_at(buf, buf->x, buf->y, 1);

	p = buf->lines[buf->y].chars;
	
	kill_addons(&p[buf->w - 1]);

	for (x = buf->w - 1; x > buf->x; x--)
		p[x] = p[x-1];

	p[buf->x] = buf->nul_attr;
}

void yw_buffer_delchar(YwBuffer *buf)
{
	int x;
	Char *p;

	will_write_at(buf, buf->x, buf->y, 1);
	
	p = buf->lines[buf->y].chars;
	
	kill_addons(&p[buf->x]);

	for (x = buf->x; x < buf->w - 1; x++)
		p[x] = p[x+1];

	p[buf->w - 1] = buf->nul_attr;
}

static void flush_attrs(YwBuffer *buf, Char *cur, const Char *n)
{
	if (cur->fg != n->fg)
		buf->cb->remote_attr(buf->cb_data, yw_bufattr_fg, 
				     (n->fg | YW_COLOR_ANSI_BLACK));
	if (cur->bg != n->bg)
		buf->cb->remote_attr(buf->cb_data, yw_bufattr_bg, 
				     (n->bg | YW_COLOR_ANSI_BLACK));
	if ((cur->flags & F_BOLD) != (n->flags & F_BOLD))
		buf->cb->remote_attr(buf->cb_data, yw_bufattr_bold, 
			(n->flags & F_BOLD) != 0);
	if ((cur->flags & F_UNDERLINE) != (n->flags & F_UNDERLINE))
		buf->cb->remote_attr(buf->cb_data, yw_bufattr_underline, 
			(n->flags & F_UNDERLINE) != 0);
	*cur = *n;
}

static int process_line(YwBuffer *buf, Line *l, int x, int y, Char *cur)
{
#define BUFSIZE 200
	uint32_t xbuf[BUFSIZE];
	YwString str;
	int beg = 1;
	int stop_marker = 0;

	str.len = 0;
	str.chars = xbuf;

	for (; x < buf->w; x++)
		if ((l->chars[x].flags & F_CHANGED) || 
		    (l->chars[x].flags & F_NULL))
			break;
			
	if (x == buf->w)
		return x;
		
	/* FIXME: not always needed... */
	buf->cb->remote_goto(buf->cb_data, x, y);
	
	for (str.len = 0; str.len < BUFSIZE - 10 && x < buf->w; x++) {
		if (l->chars[x].flags & F_NULL)
			continue;
		if (beg) {
			beg = 0;
			flush_attrs(buf, cur, &l->chars[x]); 
		}
		if (!same_attr(cur, &l->chars[x]))
			break;
		if (l->chars[x].flags & F_CHANGED)
			stop_marker = str.len;
		else if (str.len - stop_marker > 8)
			break;
			
		str.chars[str.len++] = l->chars[x].ch;
		
		if (l->chars[x].addons) {
			Char *p;

			for (p = l->chars[x].addons; 
			     str.len < BUFSIZE - 10 && p; 
			     p = p->addons)
			     	str.chars[str.len++] = p->ch;
			if (p == NULL) {
				if (l->chars[x].flags & F_CHANGED)
					stop_marker = str.len - 1;
			} else {
				if (l->chars[x].flags & F_CHANGED)
					stop_marker--;
				x--;
				break;
			}
		}
	}
	
	str.len = stop_marker + 1;
	
	if (str.len)
		buf->cb->remote_puts(buf->cb_data, &str);
	
	return x;
}


static int cmp_line(YwBuffer *buf, int y)
{
	Line *l, *s;
	int i, r = 0;

	l = buf->lines + y;
	s = buf->second->lines + y;

	for (i = 0; i < buf->w; i++)
		if (!same_attr(l->chars + i, s->chars + i) ||
		    l->chars[i].ch != s->chars[i].ch ||
		    !same_addons(l->chars + i, s->chars + i)) {
		    	l->chars[i].flags |= F_CHANGED;
			kill_addons(s->chars + i);
			s->chars[i] = l->chars[i];
			copy_addons(s->chars + i, l->chars + i);
			r++;
		} else {
		    	l->chars[i].flags &= ~F_CHANGED;
		}
	return r;
}

void yw_buffer_sync(YwBuffer *buf)
{
	int x, y, end, start;
	Line *l;
	Char cur;
	int did = 0;

	buf->last = NULL;
	
	for (y = 0; y < buf->h; y++) {
		l = buf->lines + y;
		if (!cmp_line(buf, y))
			continue;	/* nothing to do */
		if (!did) {
			buf->cb->remote_attr(buf->cb_data, 
					yw_bufattr_reset, 0);
			cur = buf->nul_attr;
			did++;
		}
		for (start = 0; start < buf->w; start++)
			if (l->chars[start].flags & F_CHANGED)
				break;
		for (end = buf->w - 1; end > start; end--)
			if ((l->chars[end].flags & F_CHANGED))
				break;
		
		
		for (x = start; x <= end; )
			x = process_line(buf, l, x, y, &cur);
	}
	
	if (did || buf->last_x != buf->x || buf->last_y != buf->y) {
		buf->last_x = buf->x;
		buf->last_y = buf->y;
		buf->cb->remote_goto(buf->cb_data, buf->x, buf->y);
		if (buf->cb->remote_flush)
			buf->cb->remote_flush(buf->cb_data);
	}
}

/* 
 * Must be in the same order as in <yw/buffer.h> !!!
 */
static YwMapEntry names[] = {
	{ "bad", yw_bufattr_bad },
	{ "fg", yw_bufattr_fg },
	{ "bg", yw_bufattr_bg },
	{ "underline", yw_bufattr_underline },
	{ "bold", yw_bufattr_bold },
	{ "reset", yw_bufattr_reset },
	{ NULL, yw_bufattr_bad }
};

const char *yw_bufattr_name(YwBufAttr attr)
{
	int a;

	a = (int)attr;
	if (a >= (int)(sizeof(names) / sizeof(names[0])))
		return yw_bufattr_bad;
		
	return names[a].key;
}

YwBufAttr yw_bufattr_from_keyword(const char *name)
{
	static YwMapCache *cache = NULL;
	
	return yw_map(name, names, &cache);
}
