/* $Id: term.c,v 1.3 2001/05/23 08:46:00 malekith Exp $ */

/* this is for setenv(), but note that we also use BSD forkpty() 
 * function. In BSD it is in <util.h> I guess. */

#define _BSD_SOURCE 1	

#include <yw/util.h>
#include <yw/sock.h>
#include <yw/window.h>
#include <yw/keys.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/poll.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include <pty.h>

#define NPAR 16

YwConnection *conn;
int term_fd;
int term_w, term_h;
int x, y;
char *tab_stops;
int scroll_up, scroll_down;
int save_x, save_y;
YwBuffer *buff;
int underline_color, dim_color;

int child_died;

#define BOOL int

struct {
	BOOL ins_mode;
	BOOL display_control;
	BOOL lf_nl;
	BOOL cursor_o;
	BOOL mode132;
	BOOL reverse_video;
	BOOL scroll_relative;
	BOOL auto_wrap;			/* def on */
	BOOL kbd_autorepeat;		/* def on */
	BOOL x10_mouse;
	BOOL x11_mouse;
	BOOL cursor_visible;		/* def on */
} mode;

typedef struct {
	int fore;
	int back;
	int intensity;		/* 0, 1, 2 */
	BOOL reverse;
	BOOL underline;
} sgr_mode_t;

sgr_mode_t sgr_mode, saved_sgr;

int did_set_size = 0;

void sig_chld(int sig)
{
	(void)sig;
	child_died++;
}

void set_size()
{
	struct winsize ws;
	
	ws.ws_row = term_h;
	ws.ws_col = term_w;
	ws.ws_xpixel = term_w * 8;
	ws.ws_ypixel = term_h * 8;
	
	ioctl(term_fd, TIOCSWINSZ, &ws);

	did_set_size++;
}

void xwrite(unsigned char *buf, int n)
{
	int c;
	
	printf("'");
	while (n--) {
		c = *buf++;
		if (c >= ' ')
			printf("%c", c);
		else
			printf("\\x%02x", c);
	}
	printf("'\n");
}

void state()
{
	const char *file;
	int line;
	int state;
	
	yw_conn_get_state(conn, &file, &line, &state);
	printf("state = %s [%d], %s:%d\n", yw_strerror(state), 
		state, file, line);
}

void check_state()
{
	if (yw_conn_state(conn)) {
		state();
		exit(1);
	}
}

void dispatch(YwPacket *pkt)
{
	YwString *str;
	int flags;
	char *s = NULL;
	int l = 0;
	char buf[20];

	if (yw_key_normal(pkt, &str, &flags) == 0) {
		yw_string_get_cstring(str, NULL, &s, &l);
		if (flags & yw_key_ctrl && s) {
			if ((s[0] ^ ' ') < ' ')
				s[0] ^= ' ';
		}
		if (s == 0) {
			s = yw_strdup("?");
			l = 1;
		}
	} else if (yw_key_special(pkt, &l, &flags) == 0) {
		s = buf;
		if (YW_KEY_IS_F(l)) {
			l = YW_KEY_FVAL(l);
			
			if (l < 6)
				sprintf(buf, "\033[%c%c", 
					mode.cursor_o ? 'O' : '[',
					'A' + l - 1);
			else
				sprintf(buf, "\033[%d~", l + 11);
		} else {
			switch (l) {
			case yw_key_up:
			case yw_key_down:
			case yw_key_right:
			case yw_key_left:
				sprintf(buf, "\033[%c", 'A' + l - yw_key_up);
				break;
			case yw_key_home:
			case yw_key_ins:
			case yw_key_del:
			case yw_key_end:
			case yw_key_pgup:
			case yw_key_pgdn:
				sprintf(buf, "\033[%d~", l - yw_key_home + 1);
				break;
			case yw_key_keypad5:
				s = "\033[G";
				break;
			case yw_key_backspace:
				s = "\b";
				break;
			default:
				s = "";
				break;
			}
		}
		s = yw_strdup(s);
		l = strlen(s);
	} else {
		yw_packet_free(pkt);
		return;
	}

	if (!did_set_size)
		set_size();
	if (flags & yw_key_meta) {
		if (l + 2 > (int)sizeof(buf)) {
			write(term_fd, "\033", 1);
			write(term_fd, s, l);
		} else {
			buf[0] = '\033';
			memcpy(buf + 1, s, l);
			write(term_fd, buf, l + 1);
		}
	} else {
		write(term_fd, s, l);
	}
	
	yw_free(s);
	
	yw_packet_free(pkt);
}

int get_color(int c)
{
	return (c & 7) + YW_COLOR_ANSI_BLACK;
} 

void go()
{
	yw_buffer_goto(buff, x, y);
}

void rethink_sgr()
{
	int fg, bg, tmp;
	
	fg = sgr_mode.fore;
	bg = sgr_mode.back;
	
	if (sgr_mode.underline) {
		fg = underline_color;
		if (sgr_mode.intensity == 2)
			fg |= 8;
	} else { 
		switch (sgr_mode.intensity) {
		case 0:
			fg = dim_color;
			break;
		case 2:
			fg |= 8;
			break;
		}
	}
	
	if (sgr_mode.reverse) {
		tmp = fg;
		fg = bg;
		bg = tmp;
	}

	yw_buffer_set_fg(buff, fg);
	yw_buffer_set_bg(buff, bg);
}

void sgr(int n)
{
	switch (n) {
	case 0:
		sgr_mode.fore = get_color(7);
		sgr_mode.back = get_color(0);
		sgr_mode.intensity = 1;
		sgr_mode.reverse = 0;
		sgr_mode.underline = 0;
		break;
	case 1:
		sgr_mode.intensity = 2;
		break;
	case 2:
		sgr_mode.intensity = 0;
		break;
	case 21:
	case 22:
		sgr_mode.intensity = 0;
		break;
	case 4:
		sgr_mode.underline = 1;
		break;
	case 24:
		sgr_mode.underline = 0;
		break;
	case 7:
		sgr_mode.reverse = 1;
		break;
	case 27:
		sgr_mode.reverse = 0;
		break;
	case 10:
	case 11:
	case 12:
		/* FIXME */
		break;
	case 38:
		sgr_mode.underline = 1;
		sgr_mode.fore = get_color(7);
		break;
	case 39:
		sgr_mode.underline = 0;
		sgr_mode.fore = get_color(7);
		break;
	case 49:
		sgr_mode.back = get_color(0);
		break;
	default:
		if (n >= 30 && n <= 37)
			sgr_mode.fore = get_color(n - 30);
		else if (n >= 40 && n <= 47)
			sgr_mode.back = get_color(n - 40);
		break;
	}
	rethink_sgr();
}

void reset()
{
	yw_free(tab_stops);
	tab_stops = yw_malloc_0(term_w + 1);
	for (x = 0; x < term_w; x += 8)
		tab_stops[x] = 1;
	x = 0;
	y = 0;
	scroll_up = 0;
	scroll_down = term_h - 1;
	memset(&mode, 0, sizeof(mode));
	mode.auto_wrap = 1;
	mode.kbd_autorepeat = 1;
	mode.cursor_visible = 1;
	underline_color = get_color(6);
	dim_color = get_color(0) | 8;
	sgr(0);
	go();
}

void lf()
{
	y++;
	if (y == scroll_down + 1) {
		y--;
		yw_buffer_goto(buff, 0, scroll_up);
		yw_buffer_delline(buff);
	} else if (y == term_h) {
		y--;
		yw_buffer_goto(buff, 0, 0);
		yw_buffer_delline(buff);
	}
	go();
}

void cr()
{
	x = 0;
	go();
}

void print_normal(int c)
{
	if (mode.auto_wrap && x == term_w) {
		cr();
		lf();
	}
	yw_buffer_putchar(buff, c);
	x++;
}

void set_mode(int n, int q, int set)
{
	if (q)
		switch (n) {
		case 1:
			mode.cursor_o = set;
			break;
		case 3:
			mode.mode132 = set;
			break;
		case 5:
			mode.reverse_video = set;
			break;
		case 6:
			mode.scroll_relative = set;
			break;
		case 7:
			mode.auto_wrap = set;
			break;
		case 8:
			mode.kbd_autorepeat = set;
			break;
		case 25:
			mode.cursor_visible = set;
			break;
		case 9:
			mode.x10_mouse = set;
			break;
		case 1000:
			mode.x11_mouse = set;
			break;
		}
	else 
		switch (n) {
		case 3:
			mode.display_control = set;
			break;
		case 4:
			mode.ins_mode = set;
			break;
		case 20:
			mode.lf_nl = set;
			break;
		}
}

void save_state()
{
	save_x = x;
	save_y = y;
}

void restore_state()
{
	x = save_x;
	y = save_y;
}

void exec_csi(int ch, int nargs, int *args, int q)
{
	int n = args[0];
	int sx, sy, tmp, save;

	sx = x;
	sy = y;
	
	switch (ch) {
	case 'X':
		while (n--)
			yw_buffer_putchar(buff, ' ');
		break;
	case '@':
		while (n--)
			yw_buffer_inschar(buff);
		break;
	case 'P':
		while (n--)
			yw_buffer_delchar(buff);
		break;
	case 'F':
	case 'A':
		y -= n;
		if (ch == 'F')
			x = 0;
		break;
	case 'E':
	case 'B':
	case 'e':
		y += n;
		if (ch == 'E')
			x = 0;
		break;
	case 'c':
		write(term_fd, "\033[?6c", 5);
		break;
	case 'a':
	case 'C':
		x += n;
		break;
	case 'D':
		x -= n;
		break;
	case '`':
	case 'G':
		x = n - 1;
		break;
	case 'd':
		y = n - 1;
		break;
	case 'f':
	case 'H':
		x = args[1] - 1;
		y = args[0] - 1;
		break;
	case 'J':
		tmp = y * term_w + x;
		switch (n) {
		case 0:
			tmp = term_w * term_h - tmp;
			break;
		case 2:
			tmp = term_w * term_h;
		case 1:
			x = y = 0;
			go();
			break;
		}
		save = mode.auto_wrap;
		mode.auto_wrap = 1;
		while (tmp--)
			print_normal(' ');
		mode.auto_wrap = save;
		x = sx;
		y = sy;
		break;
	case 'K':
		tmp = x;
		switch (n) {
		case 0:
			tmp = term_w - tmp;
			break;
		case 2:
			tmp = term_w;
		case 1:
			x = 0;
			go();
			break;
		}
		while (tmp--)
			yw_buffer_putchar(buff, ' ');
		x = sx;
		y = sy;
		break;
	case 'L':
		while (n--)
			yw_buffer_insline(buff);
		break;
	case 'M':
		while (n--)
			yw_buffer_delline(buff);
		break;
	case 'g':
		tab_stops[x] = 0;
		if (n == 3) {
			for (n = 0; n < term_w; n++)
				tab_stops[n] = 0;
		}
		break;
	case 'r':
		scroll_up = args[0] - 1;
		scroll_down = args[1] - 1;
		if (scroll_up < 0)
			scroll_up = 0;
		if (scroll_up > term_h - 2)
			scroll_up = term_h - 2;
		if (scroll_down <= scroll_up)
			scroll_down = scroll_up + 1;
		if (scroll_down >= term_h)
			scroll_down = term_h - 1;
		break;
	case 'q':
		/* LEDs.. */
		break;
	case 'h':
	case 'l':
		set_mode(n, q, ch == 'h');
		break;
	case 'm':
		for (n = 0; n < nargs; n++)
			sgr(args[n]);
		break;
	case 's':
		save_state();
		break;
	case 'u':
		restore_state();
		break;
	case 'n':
		if (n == 5)
			write(term_fd, "\033[0n", 4);
		else if (n == 6) {
			char buf[100];
			sprintf(buf, "\033[%d;%dR", y + 1, x + 1);
			write(term_fd, buf, strlen(buf));
		}
		break;
	case ']':
		if (args[0] == 1)
			underline_color = get_color(args[1]);
		else if (args[0] == 2)
			dim_color = get_color(args[1]);
		break;
	}
	
	if (x < 0)
		x = 0;
	if (x >= term_w)
		x = term_w - 1;
	if (y < 0)
		y = 0;
	if (y >= term_h)
		y = term_h - 1;
	go();
}

int csi(unsigned char *buf, int n)
{
	int args[NPAR];
	int nargs = 0;
	int q = 0;
	int v;
	unsigned char *bp;

	bp = buf;

	if (n < 1)
		return -1;
	if (buf[0] == '?') {
		q++;
		buf++;
		n--;
		if (n < 1)
			return -1;
	}

	if (*buf == '[')	/* ignore function key */
		return (n == 0) ? -1 : 2;
	
	for (v = 0; v < NPAR; v++)
		args[v] = 0;

	for (;;) {
		if (*buf >= '0' && *buf <= '9') {
			v = *buf++ - '0';
			while (--n && (*buf >= '0' && *buf <= '9')) {
				v *= 10;
				v += *buf++ - '0';
			}
			
			if (n && *buf == ';') {
				buf++;
				n--;
			}
			
			if (nargs < NPAR)
				args[nargs++] = v;
		} else if (*buf == ';') {
			if (nargs < NPAR)
				args[nargs++] = 0;
			buf++;
		} else
			break;
			
		if (n == 0)
			return -1;
	}

	exec_csi(*buf++, nargs, args, q);

	return buf - bp;
}

int esc(unsigned char *buf, int n)
{
	int r = 1;
	
	(void)n;
	switch (buf[0]) {
	case 'c':
		reset();
		break;
	case 'D':
		lf();
		break;
	case 'E':
		cr();
		lf();
		break;
	case 'H':
		tab_stops[x] = 1;
		break;
	case 'M':
		/* rev lf */
		break;
	case 'Z':
		write(term_fd, "\033[?6c", 5);
		break;
	case '7':
		save_state();
		break;
	case '8':
		restore_state();
		break;
	}

	return r;
}

int process_esc(unsigned char *buf, int n)
{
	int r = 1;
	
	(void)n;
	switch (buf[0]) {
	case '\r':	/* CR */
		cr();
		break;
	case '\n':	/* LF */
	case 0x0b:	/* VT */
	case 0x0c:	/* FF */
		lf();
		if (mode.lf_nl)
			cr();
		break;
	case '\b':	/* BS */
		if (x)
			x--;
		go();
		break;
	case '\t':	/* HT */
		while (x < term_w && tab_stops[x] == 0)
			x++;
		go();
		break;
	case 0x9b:	/* CSI */
		r = csi(buf + 1, n - 1);
		if (r > 0)
			r++;
		break;
	case 0x1b:	/* ESC */
		if (n == 1)
			return -1;
		if (buf[1] == '[') {
			r = csi(buf + 2, n - 2);
			if (r > 0)
				r++;
		} else
			r = esc(buf + 1, n - 1);
		if (r > 0)
			r++;
		break;
	}

	return r;
}

int process_input(unsigned char *buf, int n)
{
	int i, c;

	if (!did_set_size)
		set_size();
			
	for (i = 0; i < n; i++) {
		c = buf[i];
		if (c == 0x7f || c == 0x07 || c == 0x00)
			continue;
		if (c < ' ') {
			c = process_esc(buf + i, n - i - 1);
			i--;
			if (c <= 0)
				break;
			i += c;
		} else
			print_normal(c);
	}

	return i;
}

void event_loop()
{
	struct pollfd fds[2];
	unsigned char buf[16000];
	int pos = 0;
	YwPacket *pkt;
	int n;

	for (;;) {
		fds[0].fd = term_fd;
		fds[1].fd = yw_conn_fd(conn);
		fds[0].events = fds[1].events = POLLIN;
		fds[0].revents = fds[1].revents = 0;

		check_state();
		poll(fds, 2, 100);
		
		while ((pkt = yw_conn_recv(conn, 0)))
			dispatch(pkt);
		if (fds[0].revents & POLLIN) {
			n = read(term_fd, buf + pos, sizeof(buf) - 1 - pos);
			if (n <= 0)
				return;
			n += pos;
			pos = n;
			n = process_input(buf, n);
			if (n != pos) {
				yw_memmove(buf, buf + n, pos - n);
				pos -= n;
			} else {
				pos = 0;
			}
			if (n)
				yw_buffer_sync(buff);
		}

		if (child_died)
			break;
	}
}

void fork_shell()
{
	int r;
	char *shell, *c;

	r = forkpty(&term_fd, NULL, NULL, NULL);

	if (r == -1) {
		perror("forkpty");
		exit(1);
	}

	if (r)
		return;
	
	shell = yw_strdup(getenv("SHELL") ? getenv("SHELL") : "/bin/sh");
	c = yw_strdup(shell);
	setenv("TERM", "linux", 1);
	*c = '-';

	execlp(shell, c, NULL);
	exit(127);
}

int main()
{
	YwWindow *win;
	
	conn = yw_conn_new();
	yw_conn_open(conn);
	check_state();

	term_w = 80;
	term_h = 24;
	
	win = yw_window_new_and_free(conn, yw_packet_make(yw_taglist_packet,
				"i" "width", term_w,
				"i" "height", term_h,
				"d" "wm_title", "Y Terminal",
				YW_END));
	
	buff = yw_window_buffer(win);

	reset();

	signal(SIGCHLD, sig_chld);
	fork_shell();
	
	event_loop();

	return 0;
}
