/* $Id: curses.c,v 1.5 2001/05/11 10:38:02 malekith Exp $ */


/* ywcurses - bare curses display */

/* #define DOUBLE_ESC */

#ifndef DOUBLE_ESC
/* for setenv() */
# define _BSD_SOURCE 1
#endif

#include <yw/display.h>
#include <yw/unigfx.h>
#include <yw/util.h>

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <sys/ioctl.h>

#include <curses.h>

YwConnection *conn;
int s_width, s_height;
int attr_map[256];
YwDisplayCallbacks *cb;
int attr_changed;
int quit;
int color_support;

/***
 *** Curses intialization
 ***/
 
/*
 * Heh... i wonder where did i get it from ... :^)
 */
static void resize_handler(int x)
{
	struct winsize size;
	
	(void)x;
	
	s_height = LINES;
	s_width = COLS;
	
	if (ioctl(1, TIOCGWINSZ, &size) == 0) {
		resizeterm(size.ws_row, size.ws_col);
		s_height = LINES;
		s_width = COLS;
	}
	
	attr_changed++;
	
	signal(SIGWINCH, resize_handler);
}

static void init_color_colors()
{
	int vgac[] = {
		COLOR_BLACK, COLOR_RED, COLOR_GREEN, COLOR_YELLOW,
		COLOR_BLUE, COLOR_MAGENTA, COLOR_CYAN, COLOR_WHITE
	};
	int back, fore, bold, i, pair;
	
	start_color();
	
	for (i = 0; i < 256; i++) {
		back = (i >> 4) & 0x07;
		bold = i & 0x08;
		fore = i & 0x07;
		pair = ((back << 3) | fore) + 1;
		init_pair(pair, vgac[fore], vgac[back]);
		attr_map[i] = COLOR_PAIR(pair);
		if (bold)
			attr_map[i] |= A_BOLD;
	}

	color_support++;
}

static void init_mono_colors()
{
	int i;
	/* subjective brightness of vga colors to decide whatever */
	/* to use A_REVERSE and/or A_BOLD */
	int vgab[] = {
		/*BLACK*/ 0, /*RED*/ 3, /*GREEN*/ 5, /*BROWN*/ 2, 
		/*BLUE*/ 4, /*MAGENTA*/ 7, /*CYAN*/ 8, /*LIGHTGRAY*/ 11, 
		/*DARKGRAY*/ 1, /*LIGHTRED*/ 9, /*LIGHTGREEN*/ 12, 
		/*YELLOW*/ 14, /*LIGHTBLUE*/ 6, /*LIGHTMAGENTA*/ 10, 
		/*LIGHTCYAN*/ 13, /*WHITE*/ 15
	};
	
	for (i = 0; i < 256; i++) {
		attr_map[i] = vgab[(i>>4)&0xf] > vgab[i&0xf] ? A_REVERSE : 0;
		if ((i&0xf) > 11)
			attr_map[i] |= A_BOLD;
	}
}

void init_curses()
{
	/* allow iso-8859-[12] chars - we don't want vga rom frames and some */
	/* ugly progs (e.g. BitchX ;) tend to left console in such state */
	if (getenv("TERM") && strcmp(getenv("TERM"),"linux") == 0)
		printf("\033(B"); 

#ifndef DOUBLE_ESC
	setenv("ESCDELAY", "100", 0);
#endif

	initscr();
	keypad(stdscr, TRUE);
	raw();
	noecho(); 
	timeout(0);
	refresh();
	meta(stdscr, TRUE);
	idlok(stdscr, TRUE);
	
	/* mousemask(REPORT_MOUSE_POSITION, 0); */
	
	if (has_colors())
		init_color_colors();
	else
		init_mono_colors();
	resize_handler(0);
}

void kill_curses()
{
	endwin();
	printf("\r");
}

/***
 *** curses tranlations
 ***/
int get_key(char **n, int *alt, int *ctrl)
{
	int k;
	char *name = NULL;
	static char buf[20];
#ifdef DOUBLE_ESC
	static int last;
#endif

	k = getch();

	if (k == ERR)
		return 0;
	
	/* XXX: temprorary, until we've got working
	 * WM, and can exit nicly on server command 
	 */
	if (k == ('Q' ^ 0x40)) {
		kill_curses();
		exit(0);
	}

	switch (k) {
	case KEY_SUSPEND: k = ('Z' ^ 0x40); break;
	case KEY_LEFT: name = "left"; break;
	case KEY_RIGHT: name = "right"; break;
	case KEY_UP: name = "up"; break;
	case KEY_DOWN: name = "down"; break;
	case KEY_HOME: name = "home"; break;
	case KEY_END: name = "end"; break;
	case KEY_IC: name = "ins"; break;
	case KEY_DC: name = "del"; break;
	case KEY_NPAGE: name = "pgdn"; break;
	case KEY_PPAGE: name = "pgup"; break;
	case KEY_B2: name = "keypad5"; break;
	case '\r':
	case KEY_ENTER: k = '\n'; break;
/*	case KEY_MOUSE: break; */
	case 127:
	case '\b':
	case KEY_BACKSPACE: name = "backspace"; break;
	default:
		if (k >= KEY_F(0) && k <= KEY_F(30)) {
			sprintf(buf, "f%d", k - KEY_F(0));
			name = buf;
		}
		break;
	}
	
	*n = name;
	*alt = *ctrl = 0;

	if (k < ' ' && yw_strchr("\t\n\033", k) == NULL) {
		k ^= 0x40;
		*ctrl = 1;
	}
	
#ifdef DOUBLE_ESC
	if (k == 27) {
		if (last == 27) {
			last = 0;
			*n = "esc";
			return 27;
		} else {
			last = 27;
			return 0;
		}
	} else {
		if (last == 27)
			*alt = 1;
		last = k;
		return k;
	}
#else
	if (k == 27) {
		k = getch();
		if (k != ERR && k != 27)
			*alt = 1;
		else {
			*n = "esc";
			k = 27;
		}
	}
	return k;
#endif
}

int unigfx2curses(int c)
{
	switch (c) {
	case 0x23BA:
		return ACS_S1;
	case 0x23BD:
		return ACS_S9;
#ifdef ACS_S3
	case 0x23BB:
		return ACS_S3;
	case 0x23BC:
		return ACS_S7;
	case 0x2264:
		return ACS_LEQUAL;
	case 0x2265:
		return ACS_GEQUAL;
	case 0x2260:
		return ACS_NEQUAL;
	case 0x03C0:
		return ACS_PI;
	case 0x00A3:
		return ACS_STERLING;
#endif
	case 0x00B0:
		return ACS_DEGREE;
	case 0x00B1:
		return ACS_PLMINUS;
	case YW_UNIGFX_BULLET:
		return ACS_BULLET;
	case YW_UNIGFX_FILL_1:
		return ACS_BOARD;
	case YW_UNIGFX_BOX_DV_DH:
		return ACS_LANTERN;
	case YW_UNIGFX_DIAMOND:
		return ACS_DIAMOND;
	case YW_UNIGFX_BLOCK_MIDDLE:
		return ACS_BLOCK;
	case YW_UNIGFX_ARROW_UP:
		return ACS_UARROW;
	case YW_UNIGFX_ARROW_DOWN:
		return ACS_DARROW;
	case YW_UNIGFX_ARROW_LEFT:
		return ACS_LARROW;
	case YW_UNIGFX_ARROW_RIGHT:
		return ACS_RARROW;
	case YW_UNIGFX_FILL_3:
		return ACS_CKBOARD;
	case YW_UNIGFX_BOX_H:
		return ACS_HLINE;
	case YW_UNIGFX_BOX_V:
		return ACS_VLINE;
	case YW_UNIGFX_BOX_D_R:
		return ACS_ULCORNER;
	case YW_UNIGFX_BOX_D_L:
		return ACS_URCORNER;
	case YW_UNIGFX_BOX_U_R:
		return ACS_LLCORNER;
	case YW_UNIGFX_BOX_U_L:
		return ACS_LRCORNER;
	case YW_UNIGFX_BOX_V_R:
		return ACS_LTEE;
	case YW_UNIGFX_BOX_V_L:
		return ACS_RTEE;
	case YW_UNIGFX_BOX_D_H:
		return ACS_TTEE;
	case YW_UNIGFX_BOX_U_H:
		return ACS_BTEE;
	case YW_UNIGFX_BOX_V_H:
		return ACS_PLUS;
	default:
		return -1;
	}
}

/***
 *** callbacks
 ***/
 
/* ^L:	wrefresh(curscr); */
void do_flush(YwDisplayCallbacks *cb)
{
	(void)cb;
	refresh();
}

void do_goto(YwDisplayCallbacks *cb, int x, int y)
{
	(void)cb;
	move(y, x);
}

void do_quit(YwDisplayCallbacks *cb)
{
	(void)cb;
	quit++;
}

void do_attr(YwDisplayCallbacks *self, YwBufAttr attr, int val)
{
	static int color = 0x07;
	
	(void)self;

	switch (attr) {
	case yw_bufattr_reset:
		color = 0x07;
		attrset(attr_map[color]);
		break;
	case yw_bufattr_bold:
	case yw_bufattr_bad:
	case yw_bufattr_underline:
		break;
	case yw_bufattr_fg:
		if ((val & 0xff000000) == 0x01000000)
			color = (color & 0xf0) | (val & 0x0f);
		attrset(attr_map[color]);
		break;
	case yw_bufattr_bg:
		if ((val & 0xff000000) == 0x01000000)
			color = (color & 0x0f) | ((val & 0x0f) << 4);
		attrset(attr_map[color]);
		break;
	}
}

void do_print(YwDisplayCallbacks *self, const YwString *str)
{
	int i, ch;
	
	(void)self;

	for (i = 0; i < str->len; i++) {
		ch = str->chars[i];
		if (unigfx2curses(ch) != -1)
			addch(unigfx2curses(ch));
		else if (ch >= ' ' && ch < 127)
			addch(ch);
		else {
			YwString tmp;
			char *p;
			int l;
			
			p = NULL;
			l = 0;
			yw_string_assign_one_char(&tmp, ch);
			yw_string_get_cstring(&tmp, NULL, &p, &l);
			yw_string_free(&tmp);
			
			if (l == 0)
				addch('?');
				
			for (ch = 0; ch < l; ch++)
				if (p[ch] >= ' ')
					addch(p[ch]);
				else
					addch('?');
			yw_free(p);
		}
	}
}

int char_width(YwDisplayCallbacks *self, uint32_t ch)
{
	YwString str;
	char *p = NULL;
	int l;
	
	(void)self;

	if (unigfx2curses(ch) != -1)
		return 1;
		
	yw_string_assign_one_char(&str, ch);
	yw_string_get_cstring(&str, NULL, &p, &l);
	
	if (p == NULL)
		return -1;	/* couldn't convert to current charset */
		
	yw_string_free(&str);
	yw_free(p);

	return l;
}

void check_state()
{
	const char *file;
	int line;
	int state;
	
	if (yw_conn_state(conn)) {
		yw_conn_get_state(conn, &file, &line, &state);
		kill_curses();
		printf("error = %s [%d], %s:%d\n", yw_strerror(state), 
			state, file, line);
		exit(1);
	}
}

void send_caps()
{
	cb->width = s_width;
	cb->height = s_height;
	cb->caps = (color_support ? YW_DISPLAY_CAP_COLOR : 0); 
	yw_display_send_caps(cb);
	attr_changed = 0;
}

void handle_keyboard()
{
	int c;
	int alt, ctrl;
	char *n;
	YwPacket *pkt;
	YwString s;
	char buf[2];

	for (;;) {
		alt = ctrl = 0;
		n = NULL;
		
		c = get_key(&n, &alt, &ctrl);

		if (n == NULL && c > 0xff)
			continue;	/* unknown character */
		
		if (c == 0)
			return;
			
		pkt = yw_packet_make(yw_void_call_packet,
			"k", "kbd", YW_END);
			
		if (ctrl)
			yw_packet_append_keyword(pkt, "control");
		if (alt)
			yw_packet_append_keyword(pkt, "meta");

		if (n) {
			yw_packet_append_keyword(pkt, n);
		} else {
			if (c < 127)
				yw_string_assign_one_char(&s, c);
			else {
				buf[0] = c;
				buf[1] = 0;
				yw_string_assign_cstring(&s, NULL, buf, 1);
			}
			yw_packet_append_string(pkt, &s);
			yw_string_free(&s);
		}
		
		yw_conn_send_and_free(conn, pkt);
	}
}

void event_loop()
{
	YwPacket *pkt;

	for (;;) {
		pkt = yw_conn_recv(conn, 10);
		while (pkt) {
			yw_display_dispatch_packet(cb, pkt);
			pkt = yw_conn_recv(conn, 0);
		}
		check_state();
		if (quit)
			return;
			
		if (attr_changed)
			send_caps();
		handle_keyboard();
	}
}

int main()
{
	int n;
	
	conn = yw_conn_new();
	yw_conn_open(conn);
	check_state();

	cb = YW_NEW_0(YwDisplayCallbacks);
	cb->char_width = char_width;
	cb->do_goto = do_goto;
	cb->do_print = do_print;
	cb->do_attr = do_attr;
	cb->do_flush = do_flush;
	cb->do_quit = do_quit;
	cb->conn = conn;

	init_curses();
	
	do_attr(NULL, yw_bufattr_reset, 0);
	n = COLS * LINES;
	move(0, 0);
	while (n--)
		addch(' ');
	move(0, 0);
	wrefresh(curscr);
	/*refresh();*/
	
	send_caps();

	event_loop();

	kill_curses();

	yw_conn_free(conn);

	return 0;
}
