/* $Id: executor.c,v 1.21 2000/09/30 19:30:02 malekith Exp $ */

#include "h.h"

static pid_t *pids, *pid_ptr;
void clear_export_once(void);
void delete_export_once(void);
int is_builtin(const char *name); /* return 1 in case of FOUND */
void exec_builtin(void); /* exec prev. found builtin */
void execute(struct arg *head, int do_fork);

int last_bg_pid = -1; /* $! */

/* wait and set last_exit */
void xwait(int pid)
{
	int r;
	wait4(pid, &r, 0, 0);
	last_exit = (r >> 8);
}

static void make_argv(struct arg *cur)
{
	argv = alloc(sizeof(char*));
	for (argc = 0; cur->type != T_TERMINATOR; cur = cur->next) {
		if (cur->type == T_NORMAL || cur->type == T_ASSIGN) {
			alloc(sizeof(char*));
			argv[argc++] = cur->val;
		}
	}
	argv[argc] = 0;
}

static struct arg *find_paren(struct arg *head)
{
	while (head) {
		if ((head = skip_command(head)) == 0)
			break;
			
		if (head->terminator == ')')
			return head;
		if ((head = head->next) == 0)
			break;
		if (head->terminator == ')')
			return head;
	}
	return 0;
}

struct arg *skip_command(struct arg *head)
{	
	WIZARD(2, "sc: '%i' '%s' == ", head->terminator, head->val);
	
	if (head->terminator == '(') {
		head = find_paren(head->next);
		if (head) {
			WIZARD(2, "'%i'\n", head->next->terminator);
			return head->next;
		} else {
			WIZARD(2, "croak\n");
			return head;
		}
	}
	
	if (find_control(head)) {
		if ((head = ctl_find_end(head)))
			head = head->next;
	} else
		while (head && !head->terminator)
			head = head->next;
	
	WIZARD(2, "norm '%i'\n", head ? head->terminator : 'x');
	return head;
}

static void do_redir(struct arg *a)
{
	int fd;
	
	if (a->target_fd != -1) {
		if (dup2(a->target_fd, a->redir_fd) == -1) {
			perror("dup2(%d,%d)", a->target_fd, a->redir_fd);
			exit(127);
		}
	} else {
		fd = hopen(a->val, a->flags, 0666);
		if (fd == -1) {
			perror("%s", a->val);
			exit(127);
		}
		if (dup2(fd, a->redir_fd) == -1) {
			perror("dup2[%s->%d]", a->val, a->redir_fd);
			exit(127);
		}
		close(fd);
	}
	fcntl(a->redir_fd, F_SETFD, 0); /* clr close-on-exec */
}

/* ;) */
static void do_child(struct arg *head, int rfd, int wfd)
{
	char *p, *n;
	struct arg *a;
		
	if (rfd != -1)
		dup2(rfd, 0);
		
	if (wfd != -1)
		dup2(wfd, 1);
		
	if (head->terminator) {
		/* a subshell */
		find_paren((head = head->next))->terminator = '\n';
		execute(head, 0);
		/* never returns */
	}
	
	if ((p = getenv("PATH")) == 0)
		p = DEF_PATH; /* rather unusual thing ... */
		
	p = strdup(p);
	
	for (a = head; a->type != T_TERMINATOR; a = a->next)
		if (a->type == T_REDIR)
			do_redir(a);
			
	make_argv(head);
	
	if (strcmp(argv[0], "exec") == 0)
		argv++, argc--;
		
	putenv("_", argv[0]);
	if (script)
		putenv("LINENO", sprintf("%d", line));
	
	environ = buildenv();
		
	if (*argv[0] == '.' || *argv[0] == '/') {
		execve(argv[0], argv, environ);
		if (errno != ENOENT) {
			perror("%s: - cannot execute", argv[0]);
			exit(126);
		}
	} else
		while (p && *p) {
			char *e;
			e = strchr(p, ':');
			if (e)
				*e++ = 0;
			n = sprintf("%s/%s", p, argv[0]);
			execve(n, argv, environ);
			if (errno != ENOENT)
				goto croak;
			p = e;
		}
		
	perr("no, no bad magic word ``%s''", argv[0]);
	
	exit(127);

croak:
	perror("%s: - cannot execute", n);
	exit(126);
}

extern int highest_fd;

static void do_ctl(struct arg *head, int rfd, int wfd, int do_fork)
{
	struct arg *a;
	int hfd = 0, i;
	
	if (do_fork == 1) {
		hfd = highest_fd + 1;
		for (i = 0; i <= 10; i++)
			dup2(i, hfd+i);
	}
	
	if (rfd != -1)
		dup2(rfd, 0);
		
	if (wfd != -1)
		dup2(wfd, 1);
		
	for (a = ctl_find_end(head); a->type != T_TERMINATOR; a = a->next)
		if (a->type == T_REDIR)
			do_redir(a);
	
	ctl_exec(head);
	
	if (do_fork != 1)
		exit(last_exit);
	
	/* restore descriptors */
	for (i = 0; i <= 10; i++)
		dup2(hfd+i, i);
}

void exec_one(struct arg *head, int rfd, int wfd, int do_fork)
{
	struct arg *a;
	int builtin = 1;
	int pid;
	
	for (a = head; a->type == T_ASSIGN || a->type == T_REDIR; a = a->next)
		if (a->type == T_ASSIGN)
			putenv_eq(a->val, ENV_EXPORT_ONCE);
		else	/* don't use builtins when doing redirs */
			builtin = 0;
	
	if (strcmp(a->val,"exec") == 0 && do_fork != 2) {
		a = a->next;
		while (a->type == T_REDIR)
			a = a->next;
		if (a->type == T_TERMINATOR) {
			/* special case of exec : only redirs */
			/* assigmenst should take place : */
			clear_export_once(); 
			/* do the redirs : */
			for (a = head; a->type != T_TERMINATOR; a = a->next)
				if (a->type == T_REDIR)
					do_redir(a);
			last_exit = 0;
			return;
		} else
			do_fork = 0;
	}
	
	if (head->terminator != '(' && a->type == T_TERMINATOR) {
		/* there were only assigns in cmd line. bye */
		close(rfd);
		close(wfd);
		if (do_fork == 2) 
			/* if we HAD to fork, we cannot set any env vars
			 * we had to e.g. if ``echo | x=123''
			 * which shouldn't change env */
			delete_export_once(); /* remove them */
		else /* normal assigment */
			clear_export_once();
		return;
	}
	
	if (head->terminator || rfd != -1 || wfd != -1)
		builtin = 0;
		
	if (builtin && is_builtin(a->val)) {
		make_argv(head);
		exec_builtin();
		if (!do_fork)
			exit(last_exit);
		else
			return;
	}

	if (do_fork == 2 || (!head->ctl && do_fork)) {
		pid = fork();
		if (pid == 0)
			loop_lev = 0;
	} else
		pid = 0;	/* pretend we're the child */

	/* we don't need pipes anymore in parent (or if failed) */
	if (pid) {
		close(rfd);
		close(wfd);
	}
	
	if (pid == -1) {
		perror("fork");
		return;
	}
	
	if (pid) {
		*pid_ptr++ = pid;
		a_alloc(a_arg, sizeof(pid_t));
		return;
	} else if (head->ctl)
		do_ctl(head, rfd, wfd, do_fork);
	else
		do_child(head, rfd, wfd);
}

/* tree of commands:
 * - simple command  do_child 
 * - ( sub shell )   exec_one
 * - { xxx }         handled supartly in expand()
 * - pipeline |      do_pipline
 * - && ||
 * - ; &
 */

/* does everything, upto pipeline */
static int do_pipeline(struct arg *head, int do_fork)
{
	int fd[2] = { -1, -1 };
	int left_pipe = -1;
	struct arg *end;

	/* use diffrent allocator, a_arg doesn't have much to do with pids,
	 * but it won't be used anywhere here, so it's safe */
	pid_ptr = pids = a_alloc(a_arg, sizeof(pid_t));
	*pids = 0;
	
	for (;;) {
		end = skip_command(head);
		
		if (end->terminator == '|') {
			if (pipe(fd)) {
				perror("pipe");
				return -1;
			}
			exec_one(head, left_pipe, fd[1], 2);
			left_pipe = fd[0];
		} else
			break;
		
		head = end->next;
	}
	
	/* execute last command */
	exec_one(head, left_pipe, -1, left_pipe == -1 ? do_fork : 2);
	
	for (; *pids; pids++)
		xwait(*pids);
	
	if (!do_fork)
		exit(last_exit);

	return 0;
}

static int do_and_or(struct arg *head, int do_fork)
{
	struct arg *end;
	pid_t *pp;
	
	pp = pids;
	
	for (;;) {
		/* skip pipelines & subshells */
		for (end = skip_command(head); end; end = skip_command(end)) {
			if (end->terminator != '|')
				break;
			end = end->next;
		}
		
		if (end->terminator != 'a' && end->terminator != 'o')
			break;
		
		if (do_pipeline(head, 1))
			return -1;
				
		if ((end->terminator == 'a' && last_exit) ||
		    (end->terminator == 'o' && last_exit == 0))
		    	return 1;
			
		head = end->next;
		
		if (continue_lev || break_lev || error_lev)
			break;
	}
	
	if (continue_lev || break_lev || error_lev)
		return -1;
		
	return do_pipeline(head, do_fork);
}

void do_bgd(struct arg *head)
{
	int pid;
	
	pid = fork();
	
	if (pid == -1)
		perror("fork");
	
	if (pid)
		return;
	
	close(0);
	open("/dev/null", O_RDWR, 0);
	
	do_and_or(head, 0);
	
	exit(127);
}

static int check_list(struct arg *head)
{
	struct arg *p;
	int lev = 0;
	
	/* check for ;; -- it cannot get here from case ... 
	 * also check for unmatched ()
	 */
restart:
	for (p = head; p->terminator != '\n'; p = p->next)
		switch (p->terminator) {
		case 'c':
			perr("unexpected token `;;'");
			return -1;
		case '(':
			lev++;
			break;
		case 0:
			/* cannot have `` blah ( shshs ) '' */
			if (p->next->terminator == '(') {
				perr("word `%s' is followed by a `('", p->val);
				return -1;
			}
			break;
		case ')':
			if (lev == 0) {
				perr("unmatched `)'");
				return -1;
			}
			/* cannot have `` blah ) blah '' */
			if (p->next->terminator == 0) {
				perr("`)' is followed by a word `%s'", 
				     p->next->val);
				return -1;
			}
			lev--;
			break;
		default:
			break;
		}
	if (lev) {
		p->terminator = ';';
		if ((head = p->next = parse(getline(1))) == 0) {
			perr("unmatched `('");
			return -1;
		}
		goto restart;
	}
	return 0;
}

void execute(struct arg *head, int do_fork)
{
	struct arg *end;
	static int lev;
	
	lev++;
	
	check_list(head);

#ifndef SMALL	
	{
		struct arg *a;
		for (a = head; a; a = a->next)
			WIZARD(1, "ex[%d]: '%i' '%s'\n", lev, 
				a->terminator, a->val);
	}
#endif
	
	for (;;) {
		/* skip pipelines, subshells, &&, ||, words */
		for (end = skip_command(head); end; end = skip_command(end)) {
			if (strchr("|ao", end->terminator) == 0)
				break;
			end = end->next;
		}
		
		/* check for parse error in control structure */
		if (head->ctl && end == 0)  {
			if (do_fork) {
				last_exit = 1;
				lev--;
				return;
			} else
				exit(1);
		}
		
		assert(end);
		
		if (head == end)	/* avoid null commands */
			goto skip;
			
		switch (end->terminator) {
		case '&':
			/* fork a child on purpouse etc. */
			do_bgd(head);
			break;
		case ';': 
			if (end->next->terminator != '\n') {
				do_and_or(head, 1);
				break;
			}
			/* fall through */
		case '\n':
			do_and_or(head, do_fork);
			goto out;
			/* this should be the last item */
		default: oops();
		}
	
	skip:
		head = end->next;
		
		if (continue_lev || break_lev || error_lev)
			break;
	}
	
out:
	if (!do_fork)
		/* ``cmd & \n'' result: */
		exit(0);
	
	lev--;
}
