#include <stdio.h>
#include <expat.h>
#include <glib.h>

#include "convgen.h"
extern XML_Parser parser;
extern int multiple_files_option;
extern int be_quiet_option;
extern int debug_option;

enum FontStyles {
  NO_FONT,
  R,
  B,
  I
};
enum ListsTypes {
  ITEMIZEDLIST,
  ORDEREDLIST,
  VARIABLELIST,
  ADMONITION
};

static XML_Char *lang = "";	/* refentry language (lang='?') */
static GString *manvolnum;
static GString  *refentrytitle;
static GString  *refentry_date;
static GString  *refentry_source;
static GString  *refentry_manual;

static int refentry_date_found = 0;
static int refentry_source_found = 0;
static int refentry_manual_found = 0;

static int refname_count = 0;	/* needed to separate refnames with commas */
static int paramdef_count = 0;	/* needed to separate function params with commas */
static int arg_optional = 0;	/* put braces around command arguments? */
static int linespecific = 0;	/* vebatim environment? */
static int currentfont = NO_FONT;
static int spacebuffer = -1;	/* 1 : the last character was one of " \t\n"
				   -1: printing spaces is suppressed
				   0: any other char
				*/
static int in_citerefentry = 0;

#define MAXLISTDEPTH 6
static int listlevel = 0;	/* current list nesting level */
static int listtype[MAXLISTDEPTH]; /* stack for types of nested list*/
static int listcounter[MAXLISTDEPTH]; /* stack for numberedlists' counters */
static int listindent[MAXLISTDEPTH]; /* stack for relative indents */


#define ITEMIZEDLIST_INDENT 2
#define ORDEREDLIST_INDENT 4
#define VARIABLELIST_INDENT 8
#define ADMONITION_INDENT 4


/*
static int in_itemizedlist = 0;
static int in_orderedlist = 0; 
static int in_variablelist = 0; 
static int indent;
*/


/* *****************************************************************/
/* ********************* utility functions *************************/
/* *****************************************************************/

void len_printf(const XML_Char *string, int len) {
  int i;
  char s;
  //if (len==0) return;
  //printf("#");
  //printf("%d",linespecific);
  for(i = 0; i<len; i++) {
    s = *string;
    switch (s) {
    case ' ': case '\t': case '\n':
      if (linespecific) {
	if (s=='\t') printf("    ");
	else if (s=='\n') printf("\n.br\n");
	else putchar(' ');
      }
      else {
	/* don't print multiple whitespaces */
	if (spacebuffer == 0) {
	  spacebuffer = 1;
	}
      }
      break;
    default:
      /* any else than whitespace */
      /* first check if there is some space buffered */
      if (spacebuffer == 1) {
	putchar(' ');
      }
      spacebuffer = 0;
      switch (s) {
      case '-':	printf("\\-");	break;
      case '|':	printf("\\|");	break;
      case '\\': printf("\\ "); break; /* TODO */
      default: putchar(s);
      }
    }
    string++;
  }
}


const XML_Char *attr_value(const XML_Char **attr, const XML_Char *name) {
  int i;
  for (i = 0; attr[i]; i += 2) {
    if (strcmp(attr[i],name) == 0) return attr[i+1];
  }
  return "";
}

void font_change(int font) {
  if (font == currentfont) return;
  /* if (spacebuffer ) maybe flushing space will not be necessary */
  switch (font) {
  case R: printf("\\fR"); break;
  case B: printf("\\fB"); break;
  case I: printf("\\fI"); break;
  }
  currentfont = font;
}

void flush_space() {
  if (spacebuffer == 1) {
    putchar(' ');
    spacebuffer = 0;
  }
}

/* this changes font and prints data */
#define FONT(FF) {font_change(FF); len_printf(s, len); }
#define BLOCK_MIXED spacebuffer = -1; currentfont = NO_FONT;
#define BLOCK BLOCK_MIXED


/* *****************************************************************/
/* ************************** root element *************************/
/* *****************************************************************/
START(refentry)
{
  if(!be_quiet_option) fprintf(stderr,"refentry language found: lang='%s'\n", attr_value(attr,"lang"));
  lang = g_strndup(attr_value(attr,"lang"), 10);

  if (multiple_files_option) {
    /* in most cases the given sizes should be enough */
    refentrytitle = g_string_sized_new(16);
    manvolnum = g_string_sized_new(4);
    refentry_date = g_string_sized_new(16);
    refentry_source = g_string_sized_new(16);
    refentry_manual = g_string_sized_new(64);
    refname_count = 0;	
    paramdef_count = 0;	
    arg_optional = 0;	
    linespecific = 0;
    currentfont = NO_FONT;
    spacebuffer = -1;
    listlevel = 0;
    refentry_date_found = 0;
    refentry_source_found = 0;
    refentry_manual_found = 0;
    in_citerefentry = 0;
  }
  
}
END(refentry)
{
  /* fprintf(stderr,__FUNCTION__"\n"); */

  /* ignore input until next refentry */
  if (multiple_files_option) {
    g_string_free(manvolnum,1);
    g_string_free(refentrytitle,1);
    g_free(lang);
    XML_SetElementHandler(parser, wait_for_root_element_start, NULL);
  }
  else {
    exit(0);			/* one refentry is enough */
  }
}


/* *****************************************************************/
/* ************************ refmeta tags ***************************/
/* *****************************************************************/

START(refmeta) { BLOCK_MIXED; if (!multiple_files_option) printf(".TH "); }  /* no newline before! */
END(refmeta) {
  /* now we know enough to put output to file */
  const XML_Char *tmp;
  BLOCK_MIXED;
  if (multiple_files_option) {
    tmp = g_strdup_printf("%s.%s",refentrytitle->str, manvolnum->str);
    if (freopen(tmp, "w", stdout) == NULL) {
      fprintf(stderr, "cannot freopen() file %s\n",tmp);
      exit(1);
    }
    g_free((void *)tmp);
    printf(".TH \"%s\" \"%s\" \"\" \"\" \"\"",
	   refentrytitle->str, manvolnum->str );
  }
  else {
    printf(" \"\" \"\" \"\"");
  }
}

START(refentrytitle) { if (!multiple_files_option && !in_citerefentry) printf("\""); } 
END(refentrytitle) { if (!multiple_files_option && !in_citerefentry) printf("\" "); } 
DATA(refentrytitle) {
  int i;
  if (in_citerefentry) {
    FONT(B);
  }
  else {
    if (multiple_files_option) {
      for(i=0; i<len; i++) g_string_append_c(refentrytitle, *s++);
    }
    else {
      FONT(R);
    }
  }
}
 
START(manvolnum) {
  if (in_citerefentry) { font_change(R); putchar('('); }
  else if (!multiple_files_option) printf("\"");
} 
END(manvolnum) {
  if (in_citerefentry) { putchar(')'); }
  else if (!multiple_files_option) { printf("\" "); }
} 
DATA(manvolnum) {
  int i;
  if (in_citerefentry) {
    FONT(R);
  }
  else {
    if (multiple_files_option) {
      for(i=0; i<len; i++) g_string_append_c(manvolnum, *s++);
    }
    else {
      FONT(R);
    }
  }
} 

START(refmiscinfo) {
  if (multiple_files_option) {
  }
  else {
    if(!in_citerefentry) printf("\" ");
  }
} 
END(refmiscinfo) { if (!multiple_files_option && !in_citerefentry ) printf("\" "); } 
DATA(refmiscinfo) {
  if (!in_citerefentry) {
    if (multiple_files_option) {
    }
    else {
      len_printf(s,len);
    }
  }
}

/* *****************************************************************/
/* ************************ refnamediv tags ************************/
/* *****************************************************************/

START(refnamediv) { BLOCK_MIXED; /*printf("\n");*/ }
END(refnamediv) { BLOCK_MIXED; /*printf("\n");*/ }

START(refname) { if (refname_count++) printf(", "); else printf("\n.SH NAME\n"); }
DATA(refname) { len_printf(s,len); }

START(refpurpose) { printf(" \\- "); }
DATA(refpurpose) { len_printf(s,len); }

/* *****************************************************************/
/* *********************** refsynopsisdiv tags *********************/
/* *****************************************************************/
START(refsynopsisdiv) { printf("\n.SH SYNOPSIS"); }

START(cmdsynopsis) { BLOCK_MIXED; printf("\n"); /* printf("\n.sp\n");*/ }
END(cmdsynopsis) { BLOCK_MIXED; /* printf("\n"); */ }

START(arg) {
  const XML_Char *a;
  a = attr_value(attr,"choice");
  printf(" ");
  /* if "plain" don't put braces */
  if(a) if (strcmp(a,"plain")==0) { arg_optional = 0; return; }
  arg_optional = 1;
  font_change(R); printf("[ ");
}
END(arg) { if(arg_optional) { font_change(R); printf(" ]"); } }
DATA(arg) FONT(B)

START(sbr) { printf("\n.br\n"); }	/* line break in cmdsynopsis */



START(funcsynopsis) { BLOCK_MIXED; /*printf("\n.sp\n");*/ }
END(funcsynopsis) { BLOCK_MIXED;/* printf("\n");*/ }

START(funcsynopsisinfo) { BLOCK_MIXED; linespecific = 1; printf("\n.P\n"); }
END(funcsynopsisinfo) { BLOCK_MIXED; linespecific = 0; /* printf("\n");*/ }
DATA(funcsynopsisinfo) FONT(B)

START(funcprototype) { BLOCK_MIXED; printf("\n.HP\n"); }
END(funcprototype) { BLOCK_MIXED; font_change(B); printf(")"); }	/* closing paren */

END(funcdef) { paramdef_count = 0; font_change(B); printf("("); } /* opening paren */
DATA(funcdef) FONT(B)
     
START(paramdef) { if(paramdef_count++) printf(", "); }
DATA(paramdef) FONT(B)
     

START(funcparams) { font_change(B); printf("("); }
END(funcparams) { font_change(B); printf(")"); }
DATA(funcparams) FONT(I)

START(void) { printf("void"); }
START(varargs) { if(paramdef_count++) printf(", "); font_change(I); printf("\\&..."); }


START(synopsis) { BLOCK_MIXED; linespecific = 1; printf("\n.sp\n"); }
END(synopsis) { BLOCK_MIXED; linespecific = 0; /* printf("\n");*/ }
DATA(synopsis) FONT(R)

START(optional) { font_change(R); printf(" [ "); }
END(optional) { font_change(R); printf(" ] "); }
DATA(optional) FONT(B)

     
/* refsect* */
START(refsect1) { printf("\n.SH "); }
START(refsect2) { printf("\n.SS "); }
START(refsect3) { printf("\n.SS \" \""); } /* ugly */


/* TODO: title can be child of many other elements */
DATA(title) FONT(B)

/* *****************************************************************/
/* ************************** block elements ***********************/
/* *****************************************************************/

START(para) { BLOCK_MIXED if(!listlevel) printf("\n.P\n"); else printf("\n"); }
END(para) { BLOCK_MIXED if(listlevel) printf("\n.br"); /* else printf("\n");*/ }
DATA(para) FONT(R);

START(example) {
  BLOCK_MIXED;
  if(!listlevel)
    printf("\n.P\n");
  else printf("\n.br\n");
  font_change(I); printf("Example: ");
}



/* function for lists */
void start_list(int type, int indent) {
  ++listlevel;
  listtype[listlevel] = type;
  listcounter[listlevel] = 0;
  if (listlevel > 1) printf("\n.RS %d",listindent[listlevel-1]);
  listindent[listlevel] = indent;
}
void end_list() {
  if (listlevel > 1) printf("\n.RE");
  listlevel--;
}


START(itemizedlist) {
  BLOCK;
  start_list(ITEMIZEDLIST, ITEMIZEDLIST_INDENT);
}
END(itemizedlist) {
  BLOCK;
  end_list();
}
START(orderedlist) {
  BLOCK;
  start_list(ORDEREDLIST, ORDEREDLIST_INDENT);
}
END(orderedlist) {
  BLOCK;
  end_list();
}
START(variablelist) {
  BLOCK;
  start_list(VARIABLELIST, VARIABLELIST_INDENT);
}
END(variablelist) {
  BLOCK;
  end_list();
}


START(varlistentry) {
  printf("\n.IP \"");
  /* rest is in <term> element */
}



START(listitem) {
  switch (listtype[listlevel]) {
  case ITEMIZEDLIST:
    printf("\n.IP \\(bu %d", ITEMIZEDLIST_INDENT);
    break;
  case ORDEREDLIST:
    printf("\n.IP \"%2d.\" %d",++listcounter[listlevel], ORDEREDLIST_INDENT);
    break;
  case VARIABLELIST:
    printf("\" %d", VARIABLELIST_INDENT);
    break;

  }
}

START(screen) { BLOCK_MIXED linespecific = 1; printf("\n.br\n");}
END(screen) { BLOCK_MIXED linespecific = 0; printf("\n.br");}
DATA(screen) FONT(R)

START(programlisting) { BLOCK_MIXED linespecific = 1; printf("\n.br\n");}
END(programlisting) { BLOCK_MIXED linespecific = 0; printf("\n.br"); }
DATA(programlisting) FONT(R)

START(literallayout) { BLOCK_MIXED linespecific = 1; printf("\n.br\n");}
END(literallayout) { BLOCK_MIXED linespecific = 0; printf("\n.br");}
DATA(literallayout) FONT(R)
     
DATA(term) FONT(B)

START(ulink) {
  const XML_Char *a;
  a = attr_value(attr,"url");
  printf("\n.UR \"");
  if (a) printf("%s",a);
  printf("\"\n");
}

END(ulink) { printf("\n.UE\n"); }
DATA(ulink) FONT(B)

START(quote) { flush_space(); printf("\""); }
END(quote) { printf("\""); }
DATA(quote) { len_printf(s, len); }

START(comment) { BLOCK_MIXED; printf("\n.br\n"); font_change(B); printf("[Comment: "); }
END(comment) { BLOCK_MIXED; font_change(B); printf("]"); }
DATA(comment) FONT(R);


START(warning) { BLOCK; printf("\n.RS %d", ADMONITION_INDENT);
  start_list(ADMONITION, ADMONITION_INDENT);
  printf("\n.P\n"); font_change(B); printf("Warning! ");
}
END(warning) { BLOCK; end_list(); printf("\n.RE"); }

START(caution) {BLOCK; printf("\n.RS %d", ADMONITION_INDENT);
  start_list(ADMONITION, ADMONITION_INDENT);
  printf("\n.P\n"); font_change(B); printf("Caution! ");
}
END(caution) { BLOCK; end_list(); printf("\n.RE");}

START(note) {BLOCK; printf("\n.RS %d", ADMONITION_INDENT);
  start_list(ADMONITION, ADMONITION_INDENT);
  printf("\n.P\n"); font_change(B); printf("Note: ");
}
END(note) { BLOCK; end_list(); printf("\n.RE");}

START(tip) {BLOCK; printf("\n.RS %d", ADMONITION_INDENT);
  start_list(ADMONITION, ADMONITION_INDENT);
  printf("\n.P\n"); font_change(B); printf("Tip: ");
}
END(tip) { BLOCK; end_list(); printf("\n.RE");}

START(citerefentry) { in_citerefentry = 1; }
END(citerefentry) { in_citerefentry = 0; }


START(abstract) {BLOCK; printf("\n.RS 4\n"); }
END(abstract) {BLOCK; printf("\n.RE"); }

START(subtitle) {printf(" \\- "); }
DATA(abstract) FONT(R);


START(highlights) {}

/* *****************************************************************/
/* ********************* some inline elements **********************/
/* *****************************************************************/

DATA(acronym) FONT(B)		/* should be small */
DATA(application) FONT(B)
DATA(command) FONT(B)
DATA(email) FONT(I)
DATA(emphasis) FONT(I)
DATA(envar) FONT(B)
DATA(errorcode) FONT(B)
DATA(function) FONT(B)
DATA(foreignphrase) FONT(I)
DATA(filename) FONT(I)
DATA(literal) FONT(B)
DATA(option) FONT(B)
DATA(parameter) FONT(I)
DATA(property) FONT(I)
DATA(replaceable) FONT(I)
DATA(returnvalue) FONT(B)
DATA(symbol) FONT(B)
DATA(structfield) FONT(I)
DATA(structname) FONT(B)
DATA(userinput) FONT(B)
DATA(varname) FONT(I)
DATA(type) FONT(B)


     
/* default handler - for all unspecified elements */
START(default) {
  if(debug_option) printf("<%s>",name);
  if(!be_quiet_option) fprintf(stderr,"%s element unsupported\n", name);
}
END(default) {
  if(debug_option) printf("</%s>",name);
}
DATA(default) { len_printf(s, len); }
