/*
 * dml - Dialog Markup Language
 *
 * $Id: slang.c,v 1.2 2001/04/12 08:54:27 malekith Exp $
 * Author: Michal Moskal <malekith@pld.org.pl>
 * include COPYING-GNU
 */

#include "pi.h"
#include <stdio.h>
#include <slang/slang.h>

#define C_BLACK "black"
#define C_GRAY "gray"
#define C_RED "red" 
#define C_LIGHTRED "brightred"
#define C_GREEN "green"
#define C_LIGHTGREEN "brightgreen"
#define C_BROWN "brown"
#define C_YELLOW "yellow"
#define C_BLUE "blue"
#define C_LIGHTBLUE "brightblue"
#define C_MAGENTA "magenta"
#define C_LIGHTMAGENTA "brightmagenta"
#define C_CYAN "cyan"
#define C_LIGHTCYAN "brightcyan"
#define C_LIGHTGRAY "lightgray"
#define C_WHITE "white"

struct pi_object *_pi_root_object;

int pi_screen_height, pi_screen_width;

static void init_colors()
{
	SLtt_set_color(pi_color_bg, 0, C_YELLOW, C_BLUE);
	SLtt_set_color(pi_color_label, 0, C_LIGHTGRAY, C_BLACK);
	SLtt_set_color(pi_color_window, 0, C_CYAN, C_BLACK);
	SLtt_set_color(pi_color_window_hl, 0, C_LIGHTCYAN, C_BLACK);
	SLtt_set_color(pi_color_button, 0, C_GREEN, C_BLACK);
	SLtt_set_color(pi_color_abutton, 0, C_BLACK, C_GREEN);
	SLtt_set_color(pi_color_button_hl, 0, C_LIGHTGREEN, C_BLACK);
	SLtt_set_color(pi_color_abutton_hl, 0, C_LIGHTGREEN, C_GREEN);
	SLtt_set_color(pi_color_input, 0, C_BLACK, C_CYAN);
	SLtt_set_color(pi_color_input_hl, 0, C_WHITE, C_CYAN);
	SLtt_set_color(pi_color_input_deadly, 0, C_WHITE, C_BLUE);
	SLtt_set_color(pi_color_input_arrow, 0, C_WHITE, C_GREEN);
	SLtt_set_color(pi_color_textarea_c1, 0, C_LIGHTGRAY, C_BLUE);
	SLtt_set_color(pi_color_textarea_c2, 0, C_YELLOW, C_BLUE);
	SLtt_set_color(pi_color_textarea_c3, 0, C_LIGHTCYAN, C_BLUE);
	SLtt_set_color(pi_color_textarea_scrollbar, 0, C_BLUE, C_BLUE);
	SLtt_set_color(pi_color_atextarea_scrollbar_marked, 0, C_GREEN, C_GREEN);
	SLtt_set_color(pi_color_atextarea_scrollbar, 0, C_BLUE, C_BLUE);
	SLtt_set_color(pi_color_textarea_scrollbar_marked, 0, C_CYAN, C_CYAN);
	SLtt_set_color(pi_color_menu_1, 0, C_LIGHTGRAY, C_BLUE);
	SLtt_set_color(pi_color_menu_2, 0, C_YELLOW, C_BLUE);
	SLtt_set_color(pi_color_menu_hl_1, 0, C_BLACK, C_CYAN);
	SLtt_set_color(pi_color_menu_hl_2, 0, C_LIGHTCYAN, C_CYAN);
	SLtt_set_color(pi_color_menu_hl_active_1, 0, C_BLACK, C_GREEN);
	SLtt_set_color(pi_color_menu_hl_active_2, 0, C_LIGHTGREEN, C_GREEN);
	SLtt_set_color(pi_color_amenu_scrollbar, 0, C_BLUE, C_BLUE);
	SLtt_set_color(pi_color_amenu_scrollbar_marked, 0, C_CYAN, C_CYAN);
	SLtt_set_color(pi_color_menu_scrollbar_marked, 0, C_GREEN, C_GREEN);
	SLtt_set_color(pi_color_menu_scrollbar, 0, C_BLUE, C_BLUE);
	SLtt_set_color(pi_color_title1, 0, C_GREEN, C_BLACK);
	SLtt_set_color(pi_color_title2, 0, C_LIGHTGREEN, C_BLACK);
	SLtt_set_color(pi_color_checkbox, 0, C_WHITE, C_BLACK);
	SLtt_set_color(pi_color_checkbox_hl, 0, C_LIGHTCYAN, C_BLUE);
	SLtt_set_color(pi_color_fileselector_1, 0, C_LIGHTGRAY, C_BLUE);
	SLtt_set_color(pi_color_fileselector_2, 0, C_YELLOW, C_BLUE);
	SLtt_set_color(pi_color_fileselector_hl_1, 0, C_BLACK, C_CYAN);
	SLtt_set_color(pi_color_fileselector_hl_2, 0, C_LIGHTCYAN, C_CYAN);
	SLtt_set_color(pi_color_fileselector_hl_active_1, 0, C_BLACK, C_GREEN);
	SLtt_set_color(pi_color_fileselector_hl_active_2, 0, C_LIGHTGREEN, C_GREEN);
	SLtt_set_color(pi_color_afileselector_scrollbar, 0, C_BLUE, C_BLUE);
	SLtt_set_color(pi_color_afileselector_scrollbar_marked, 0, C_CYAN, C_CYAN);
	SLtt_set_color(pi_color_fileselector_scrollbar_marked, 0, C_GREEN, C_GREEN);
	SLtt_set_color(pi_color_fileselector_scrollbar, 0, C_BLUE, C_BLUE);
}

static void int_handler(int x)
{
	(void)x;
	pi_die("interrupt");
}

void pi_cls()
{
	SLsmg_cls();
}

void pi_addstr_fill_alt(const char *s, int n, int c1, int c2)
{
	pi_color(c1);
	
	while (*s && n--) {
		if (*s == '_' && s[1]) {
			pi_color(c2);
			pi_putch(*++s);
			pi_color(c1);
			s++;
		} else
			pi_putch(*s++);
	}

	pi_color(c1);
	while (n-- > 0)
		pi_putch(' ');
}

void pi_addstr_alt(const char *s, int n, int c1, int c2)
{
	pi_color(c1);
	
	while (*s && n--) {
		if (*s == '_' && s[1]) {
			pi_color(c2);
			pi_putch(*++s);
			pi_color(c1);
			s++;
		} else
			pi_putch(*s++);
	}
	
	pi_color(c1);
}

int pi_strlen_alt(const char *s)
{
	int n;
	
	for (n = 0; *s; s++, n++) 
		if (*s == '_' && s[1])
			s++;
	return n;
}

void pi_init()
{
	SLtt_get_terminfo();
	if (SLkp_init() == -1)
		pi_die("Unable to initialize SLkp.");
		
	if (SLang_init_tty(-1, 0, 0) == -1)
		pi_die("Unable to initialize SLang terminal.");
	SLsmg_init_smg();
	
	SLang_set_abort_signal(int_handler);

	init_colors();

	_pi_root_object = pi_new_object(0);

	pi_screen_height = SLtt_Screen_Rows;
	pi_screen_width = SLtt_Screen_Cols;
}

void pi_goto(int x, int y)
{
	SLsmg_gotorc(y, x);
}

void pi_frame(int x, int y, int w, int h)
{
	SLsmg_draw_box(y, x, h, w);
}

void pi_putch(int c)
{
	SLsmg_write_char(c & 0xff);
}

void pi_addstr(const char *c)
{
	SLsmg_write_string((char*)c);
}

void pi_addstrn(const char *c, int n)
{
	SLsmg_write_nstring((char*)c, n);
}

void pi_addstrn_fill(const char *c, int n)
{
	int k;
	
	if ((k = strlen(c)) > n)
		SLsmg_write_nchars((char*)c, n);
	else {
		SLsmg_write_string((char*)c);
		SLsmg_write_nstring(0, n - k);
	}
}

void pi_color(int c)
{
	SLsmg_set_color(c);
}

void pi_refresh()
{
	SLsmg_refresh();
}

void pi_fini()
{
	SLsmg_reset_smg();
	SLang_reset_tty();
}

void pi_die(const char *msg)
{
	pi_fini();
	fprintf(stderr, "%s\n", msg);
	exit(1);
}

void pi_fill(int c, int x, int y, int w, int h)
{
	SLsmg_fill_region(y, x, h, w, c);
}

int pi_getch()
{
	int k;

	if (SLang_input_pending(0) <= 0)
		return 0;
	
	k = SLkp_getkey();

	switch (k) {
	case SL_KEY_ERR:
		return 0;
	case SL_KEY_UP:
		return pi_key_up;
	case SL_KEY_DOWN:
		return pi_key_down;
	case SL_KEY_LEFT:
		return pi_key_left;
	case SL_KEY_RIGHT:
		return pi_key_right;
	case SL_KEY_PPAGE:
		return pi_key_pgup;
	case SL_KEY_NPAGE:
		return pi_key_pgdn;
	case 0x7f:
	case SL_KEY_BACKSPACE:
		return '\b';
	case '\r':
	case SL_KEY_ENTER:
		return '\n';
	case SL_KEY_DELETE:
		return pi_key_del;
	case SL_KEY_HOME:
		return pi_key_home;
	case SL_KEY_END:
		return pi_key_end;
	default:
		if (k < 256)
			return k;
		if (k > SL_KEY_F0 && k < SL_KEY_F0 + 24)
			return pi_key_f(k - SL_KEY_F0);
		return 0;
	}
}
