/*
 * dml - Dialog Markup Language
 *
 * $Id: pi.h,v 1.6 2001/06/08 09:31:08 malekith Exp $
 * Author: Michal Moskal <malekith@pld.org.pl>
 * include COPYING-GNU
 */

#define PI_FSF_ALLOW_DIR	0x0001
#define PI_FSF_ALLOW_REG	0x0002
#define PI_FSF_ALLOW_DEV	0x0004
#define PI_FSF_ALLOW_BROKEN	0x0008
#define PI_FSF_ALLOW_OTHER	0x0010
#define PI_FSF_ALLOW_ALL	0x00ff

enum {
	pi_key_first = 0x100,
	pi_key_up, pi_key_down, pi_key_left, pi_key_right,
	pi_key_pgup, pi_key_pgdn, pi_key_del, pi_key_home,
	pi_key_end, pi_key_f0 = 0x200,
};

#define pi_key_f(a) (pi_key_f0 + (a))

struct pi_object;
typedef void (*pi_handler_void)(struct pi_object *);
typedef void (*pi_handler_int)(struct pi_object *, int);
typedef const char *(*pi_handler_rstring)(struct pi_object *);

struct pi_funcs {
	int can_focus;
	pi_handler_void draw;
	pi_handler_int handle_cc;
	pi_handler_int handle_key;
	pi_handler_void kill;
	pi_handler_rstring fetch;
};

struct pi_rect {
	int x, y, w, h;
};

#define PI_COMMON_DATA(type) \
	struct pi_object *next, *children, *parent, *active;	\
	struct pi_rect pos;					\
	struct pi_funcs *f

struct pi_object {
	PI_COMMON_DATA(pi_object);
};

enum {
	pi_color_std = 0,
	pi_color_bg,
	pi_color_label,
	pi_color_window,
	pi_color_window_hl,
	pi_color_button,
	pi_color_button_hl,
	pi_color_abutton,
	pi_color_abutton_hl,
	pi_color_input,
	pi_color_input_hl,
	pi_color_input_deadly,
	pi_color_input_arrow,
	pi_color_textarea_c1,
	pi_color_textarea_c2,
	pi_color_textarea_c3,
	pi_color_textarea_scrollbar,
	pi_color_textarea_scrollbar_marked,
	pi_color_atextarea_scrollbar,
	pi_color_atextarea_scrollbar_marked,
	pi_color_menu_1,
	pi_color_menu_2,
	pi_color_menu_hl_1,
	pi_color_menu_hl_2,
	pi_color_menu_hl_active_1,
	pi_color_menu_hl_active_2,
	pi_color_menu_scrollbar,
	pi_color_menu_scrollbar_marked,
	pi_color_amenu_scrollbar,
	pi_color_amenu_scrollbar_marked,
	pi_color_title1,
	pi_color_title2,
	pi_color_checkbox,
	pi_color_checkbox_hl,
	pi_color_fileselector_1,
	pi_color_fileselector_2,
	pi_color_fileselector_hl_1,
	pi_color_fileselector_hl_2,
	pi_color_fileselector_hl_active_1,
	pi_color_fileselector_hl_active_2,
	pi_color_fileselector_scrollbar,
	pi_color_fileselector_scrollbar_marked,
	pi_color_afileselector_scrollbar,
	pi_color_afileselector_scrollbar_marked,
};

extern struct pi_object *_pi_root_object;
extern int pi_screen_height, pi_screen_width;
extern char *pi_title_string;

/* Objects: */

struct pi_id_info {
	struct pi_id_info *next;
	struct pi_object *obj;
	char *name;
};

struct pi_window {
	PI_COMMON_DATA(pi_window);

	char *title;
	int modal_result;
	struct pi_id_info *id_info;
	char *text_res;
};

struct pi_label {
	PI_COMMON_DATA(pi_label);
	
	char *caption;
};

struct pi_button {
	PI_COMMON_DATA(pi_button);
	
	char *caption;
	int cc;
	char *text_res;
};

struct pi_input {
	PI_COMMON_DATA(pi_input);

	char *buf;
	int top, cur, blen;
	int deadly, pass;
};

struct pi_textarea {
	PI_COMMON_DATA(pi_textarea);

	char **lines;
	int nlines;
	int top;
};

struct pi_menu_item {
	char *id;
	char *caption;
	char *group;
	enum {
		pi_menu_item_normal,
		pi_menu_item_checked,
		pi_menu_item_unchecked
	} state;
};

struct dbuf;

struct pi_menu {
	PI_COMMON_DATA(pi_menu);

	struct pi_menu_item **items;
	int nlines, hl, top;
	struct dbuf *items_db;
	int cc;
	enum {
		pi_menu_normal,
		pi_menu_checkable,
		pi_menu_checkable_signaled
	} type;
};

struct pi_checkbox {
	PI_COMMON_DATA(pi_checkbox);

	int checked;
};

struct pi_fileselector {
	PI_COMMON_DATA(pi_fileselector);

	struct pi_fileselector_entry *root;
	int nlines, hl, top;
	char *full_name;
};

struct pi_fsbutton {
	struct pi_button super;
	struct pi_input *input;
	char *title;
	int flags;
};

/* logic.c */
const char *pi_object_fetch(struct pi_object *self);
void pi_object_draw(struct pi_object *self);
void pi_object_cc(struct pi_object *self, int data);
void pi_object_kill(struct pi_object *self);
void pi_object_key(struct pi_object *self, int key);
struct pi_object *pi_new_object(int size);
void pi_add_child(struct pi_object *self, struct pi_object *o);
void pi_kill(struct pi_object *self);
void pi_add_window(struct pi_window *p);
int pi_exec(struct pi_window *win);
void pi_loop(void);
void pi_repaint();
/* slang.c */
void pi_cls();
void pi_addstr_alt(const char *s, int n, int c1, int c2);
int pi_strlen_alt(const char *s);
void pi_init(void);
void pi_goto(int x, int y);
void pi_frame(int x, int y, int w, int h);
void pi_putch(int c);
void pi_addstr(const char *c);
void pi_addstrn(const char *c, int n);
void pi_addstrn_fill(const char *c, int n);
void pi_color(int c);
void pi_refresh(void);
void pi_fini(void);
void pi_die(const char *msg);
void pi_fill(int c, int x, int y, int w, int h);
int pi_getch(void);
void pi_addstr_fill_alt(const char *s, int n, int c1, int c2);
/* util.c */
void pi_center(struct pi_window *w);
void pi_draw(struct pi_object *p);
void pi_locate(struct pi_object *self, int *xx, int *yy);
void pi_setstr(char **p, const char *s);
int pi_is_active(struct pi_object *self);
void pi_setpos(struct pi_object *p, int x, int y, int w, int h);
void pi_scrollbar(int i, int top, int h, int nlines, int color, int color_hl);
/* window.c */
void pi_window_draw(struct pi_window *self);
void pi_window_kill(struct pi_window *self);
void pi_window_cc(struct pi_window *self, int cc);
struct pi_window *pi_new_window(int size);
/* label.c */
void pi_label_draw(struct pi_label *self);
void pi_label_kill(struct pi_label *self);
struct pi_label *pi_new_label(int size);
struct pi_label *pi_add_label(struct pi_window *par, int x, int y, const char *s);
/* button.c */
void pi_button_draw(struct pi_button *self);
void pi_button_kill(struct pi_button *self);
void pi_button_key(struct pi_button *self, int key);
const char *pi_button_fetch(struct pi_button *self);
struct pi_button *pi_new_button(int size);
struct pi_button *pi_add_button(struct pi_window *par, int x, int y, const char *s, int cc);
/* xalloc.c */
void *xmalloc(int size);
void xfree(void *p);
void *xrealloc(void *ptr, int s);
/* input.c */
void pi_input_draw(struct pi_input *self);
void pi_input_kill(struct pi_input *self);
void pi_input_key(struct pi_input *self, int key);
const char *pi_input_fetch(struct pi_input *self);
struct pi_input *pi_new_input(int size);
void pi_input_set_caption(struct pi_input *i, const char *s);
struct pi_input *pi_add_input(struct pi_window *par, int x, int y, int w, const char *s);
/* textarea.c */
void pi_textarea_draw(struct pi_textarea *self);
void pi_textarea_kill(struct pi_textarea *self);
void pi_textarea_set_caption(struct pi_textarea *self, char *parsed);
void pi_textarea_key(struct pi_textarea *self, int key);
struct pi_textarea *pi_new_textarea(int size);
/* ml.c */
struct pi_window *pi_make_ml_window(const char *ml);
/* output.c */
void pi_output(struct pi_window *win, int fd);
/* parser.c */
char *pi_parse_ml(const char *ml, int mask);
/* menu.c */
void pi_menu_set_hl(struct pi_menu *self, const char *id);
void pi_menu_draw(struct pi_menu *self);
void pi_menu_add(struct pi_menu *self, const char *caption, 
		 const char *id, const char *checked, const char *group);
void pi_menu_finish_adding(struct pi_menu *self);
void pi_menu_kill(struct pi_menu *self);
void pi_menu_key(struct pi_menu *self, int key);
const char *pi_menu_fetch(struct pi_menu *self);
struct pi_menu *pi_new_menu(int size);
void pi_menu_set_caption(struct pi_menu *self, char *p);
void pi_menu_set_default_size(struct pi_menu *self);
void pi_menu_set_checked(struct pi_menu *self, const char *ids);
/* nls.c */
char *pi_nls_pp(const char *p, const char *file, const char *dir);
/* checkbox.c */
void pi_checkbox_draw(struct pi_checkbox *self);
void pi_checkbox_key(struct pi_checkbox *self, int key);
const char *pi_checkbox_fetch(struct pi_checkbox *self);
struct pi_checkbox *pi_new_checkbox(int size);
/* fileselector.c */
void pi_fileselector_draw(struct pi_fileselector *self);
void pi_fileselector_kill(struct pi_fileselector *self);
void pi_fileselector_set_root(struct pi_fileselector *self, const char *root);
void pi_fileselector_key(struct pi_fileselector *self, int key);
const char *pi_fileselector_fetch(struct pi_fileselector *self);
struct pi_fileselector *pi_new_fileselector(int size);
char *pi_select_file(const char *def, int flags, const char *title);
/* fsbutton.c */
void pi_fsbutton_kill(struct pi_fsbutton *self);
void pi_fsbutton_key(struct pi_fsbutton *self, int key);
struct pi_fsbutton *pi_new_fsbutton(int size);
/* text_res.c */
const char *pi_text_res_fetch(struct pi_object *self);
struct pi_object *pi_new_text_res(int size);
