/*************************************************
**      To: qt_nvclock.cpp
** Project: nvclock
**  Author: Jan Prokop
**  e-mail: jprokop@ibl.sk
*************************************************/

#include "qt_nvclock.moc"
#include "../../config.h"

#include <qlabel.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qmessagebox.h>

#include "nv.xpm"

static const char *txtLinks = {
    "<qt><b>NVidia</b><br>"
    "<ul>"
    "<li>home page http://www.nvidia.com</li>"
    "<li>news http://nvclock.sourceforge.net</li>"
    "<li>news http://news.nvidia.com</li>"
    "</ul>"
    "<b>Accelerated games</b><ul>"
    "<li>Quake2</li>"
    "<li>Quake3 arena</li>"
    "<li>Return To Castle Wolfrestein</li>"
    "</ul>"
    "</qt>"
};

static const char *txtAbout = {
    "<qt>Authors<br><br>"
    "<b>Roderick Colenbrander thunderbird2k@gmx.nospam.net</b>"
    "<ul>"
    "<li>author of nvclock</li>"
    "<li>overclocking/pci code</li>"
    "<li>support for multiple cards</li>"
    "<li>GTK gui</li>"
    "<li>documentation</li>"
    "</ul>"
    "<b>Jan Prokop jprokop@ibl.nospam.sk</b>"
    "<ul>"
    "<li>Qt dialog</li>"
    "</ul>"
    "<b>Christian Zander phoenix@minion.nospam.de</b>"
    "<br>(Christian only worked on nvclock 0.1 and 0.2)"
    "<ul>"
    "<li>cleanups</li>"
    "<li>autoconf/automake</li>"
    "<li>restructuring</li>"
    "</ul>"
    "</qt>"
};

//********************************************************************************

CTabNVidia::CTabNVidia(QWidget *parent, char *name)
    : QWidget(parent, name)
{
    l = new QGridLayout(this, 2, 3, 4, 4);

    int yy = 0;
    addLabel(tr("Card number"), yy);
    comboCardNo = new QComboBox(this, "comboCardNo");
    l->addWidget(comboCardNo, yy++, 1);

    addLabel(tr("Memory speed"), yy);
    spinMemory = new QSpinBox(100, 1000, 1, this, "spinMemory");
    spinMemory->setSuffix("MHz");
    l->addWidget(spinMemory, yy++, 1);

    addLabel(tr("Core speed"), yy);
    spinCore = new QSpinBox(100, 1000, 1, this, "spinCore");
    spinCore->setSuffix("MHz");
    l->addWidget(spinCore, yy++, 1);

    checkDebug = new QCheckBox(tr("Debug info"), this, "checkDebug");
    l->addWidget(checkDebug, yy++, 1);
    
    QPushButton *button = new QPushButton(tr("GO"), this);
    l->addMultiCellWidget(button, 0, 2, 2, 2);
    connect(button, SIGNAL(clicked()), SLOT(slotGo()));

    connect(comboCardNo, SIGNAL(activated(int)),
            SLOT(slotLoad(int)));

    initCardInfo(yy++);
    initAGPInfo(yy++);

    if(getCards()) slotLoad(0);
}

//********************************************************************************

void
CTabNVidia::addLabel(const QString &text, int yy)
{
    QLabel *label = new QLabel(text, this);
    l->addWidget(label, yy, 0, AlignRight | AlignVCenter);
}

//********************************************************************************

void
CTabNVidia::initCardInfo(int posy)
{
#define ADD(lname, txt) \
    layout->addWidget(new QLabel(txt, cardInfo), yy, 0, Qt::AlignRight | Qt::AlignVCenter); \
    lname = new QLabel(cardInfo); \
    lname->setFont(f); \
    layout->addWidget(lname, yy, 1); yy++;

    cardInfo = new QGroupBox(tr("Card info"), this);
    l->addMultiCellWidget(cardInfo, posy, posy, 0, 2);

    int yy = 0;
    QGridLayout *layout = new QGridLayout(cardInfo, 5, 2, 4, 2);
    layout->addRowSpacing(yy++, 10);

    QFont f(font());
    f.setBold(TRUE);

    QLabel *label = NULL;
    ADD(labelGPUName, tr("GPU Name"));
    ADD(labelBustype, tr("Bustype"));
    ADD(labelMemorySize, tr("Memory size"));
    ADD(labelMemoryType, tr("Memory type"));
#undef ADD
}

//********************************************************************************

void
CTabNVidia::initAGPInfo(int posy)
{
#define ADD(lname, txt) \
    layout->addWidget(new QLabel(txt, agpInfo), yy, 0, Qt::AlignRight | Qt::AlignVCenter); \
    lname = new QLabel(agpInfo); \
    lname->setFont(f); \
    layout->addWidget(lname, yy, 1); yy++;

    agpInfo = new QGroupBox(tr("AGP info"), this);
    l->addMultiCellWidget(agpInfo, posy, posy, 0, 2);

    int yy = 0;
    QGridLayout *layout = new QGridLayout(agpInfo, 6, 2, 4, 2);
    layout->addRowSpacing(yy++, 10);

    QFont f(font());
    f.setBold(TRUE);

    QLabel *label = NULL;
    ADD(labelAGPstatus, tr("AGP status"));
    ADD(labelSupAGPrates, tr("Supported AGP rates"));
    ADD(labelAGPrate, tr("AGP rate"));
    ADD(labelFWstatus, tr("FW status"));
    ADD(labelSBAstatus, tr("SBA status"));
#undef ADD
}

//********************************************************************************

int
CTabNVidia::getCards()
{
    //*** get list of cards ***
    total_cards = FindAllCards();

    QString s;
    switch(total_cards) {
    case -1: s = tr("No nVidia card found in your system !"); break;
    case -2: s = tr("The NVdriver isn't loaded and the user isn't root."); break;
    }
    setEnabled(total_cards >= 0);

    if(!s.isEmpty()) {
        QMessageBox::warning(this, tr("Error"), s);
        return(FALSE);
    }

    for(int i=0; i <= total_cards; i++)
        comboCardNo->insertItem(GetCardInfo(i)->device_string);

    return(TRUE);
}

//********************************************************************************

int
CTabNVidia::loadCardInfo(const struct card_info *nv_info)
{
    labelGPUName->setText(nv_info->device_string);
    labelBustype->setText(nv_info->bus_type);
    labelMemorySize->setText(QString("%1 MB").arg(nv_info->memory_size));
    labelMemoryType->setText(QString("%1 bit %2")
                             .arg(nv_info->memory_width)
                             .arg(nv_info->memory_type));

    QToolTip::add(spinMemory, tr("%1 MHz - %2 MHz")
                  .arg(nv_info->memclk_min)
                  .arg(nv_info->memclk_max));

    QToolTip::add(spinCore, tr("%1 MHz - %2 MHz")
                  .arg(nv_info->nvclk_min)
                  .arg(nv_info->nvclk_max));
    return(TRUE);
}

//********************************************************************************

int
CTabNVidia::loadAGPInfo(const struct card_info *nv_info)
{
    labelAGPstatus->setText(nv_info->agp_status);
    labelSupAGPrates->setText(nv_info->agp_support_rate);

    QString s;
    if(nv_info->agp_rate == 0) s = "-";
    else s = QString("%1X").arg(nv_info->agp_rate);
    labelAGPrate->setText(s);

    labelFWstatus->setText(QString("%1, %2").arg(nv_info->fw_support).arg(nv_info->fw_status));
    labelSBAstatus->setText(QString("%1, %2").arg(nv_info->sba_support).arg(nv_info->sba_status));
    return(TRUE);
}

//********************************************************************************

int
CTabNVidia::slotLoad(int card_number)
{
    float memclk, nvclk;
    const struct card_info *nv_info = GetCardInfo(card_number);
    
    if(GetSpeeds(&memclk, &nvclk, card_number,
                checkDebug->isChecked()) != 0) {
        QMessageBox::warning(this, tr("Warning"), tr("Can't load data !"));
        return(FALSE);
    }

    spinMemory->setRange(nv_info->memclk_min, nv_info->memclk_max);
    spinCore->setRange(nv_info->nvclk_min, nv_info->nvclk_max);


    spinMemory->setValue((int)memclk);
    spinCore->setValue((int)nvclk);

    loadCardInfo(nv_info);
    loadAGPInfo(nv_info);
    return(TRUE);
}

//********************************************************************************

int
CTabNVidia::slotGo()
{
    int id = comboCardNo->currentItem();
    qDebug("overclocking card: %d", id);

    if(!ChangeSpeeds(spinMemory->value(),
                     spinCore->value(),
                     id,
                     checkDebug->isChecked())) return(FALSE);
    
    return(slotLoad(id));
}

//********************************************************************************

CNVclock::CNVclock(QWidget *parent, char *name)
: QTabDialog(parent, name)
{
    setCaption(tr("Linux overclocker - NVclock QT " VERSION));
    
    setOkButton(0);
    setCancelButton(tr("&Quit"));

    CTabNVidia *tabNVidia  = new CTabNVidia(this, "tabNVidia");
    addTab(tabNVidia, QPixmap(nv_xpm), tr("NVidia"));

    QTextBrowser *browser = new QTextBrowser(this, "tabAbout");
    browser->setText(txtLinks);
    addTab(browser, tr("Links"));

    browser = new QTextBrowser(this, "tabAbout");
    browser->setText(txtAbout);
    addTab(browser, tr("About"));
}
