/* NVClock 0.6.1 - Linux overclocker for NVIDIA cards
 *
 * site: http://nvclock.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "backend.h"
#include "nvclock.h"

/* Missing are 0x31*, 0x32* and 0x33* since I don't know what cards that are */
/* Perhaps NV30 MX and integrated cards?? */

static struct pci_ids ids[] = 
{ { 0x20, "nVidia Riva TnT", 75, 135, 75, 135, 1 },
{ 0x28, "nVidia Riva TnT 2 Pro", 125, 200, 125, 175, 1 },
{ 0x2a, "nVidia Riva TnT 2", 100, 175, 100, 150, 1 },
{ 0x2b, "nVidia Riva TnT 2", 100, 175, 100, 175, 1 },
{ 0xa0, "nVidia Riva TnT 2 Aladdin (Integrated)", 100, 175, 100, 175, 0 },
{ 0x2c, "nVidia Riva TnT 2 VANTA", 100, 175, 100, 175, 1 },
{ 0x2d, "nVidia Riva TnT 2 M64", 100, 175, 100, 175, 1 },
{ 0x2e, "nVidia Riva TnT 2 VANTA", 100, 175, 100, 175, 1 },
{ 0x2f, "nVidia Riva TnT 2 VANTA", 100, 175, 100, 175, 1 },	    
{ 0x29, "nVidia Riva TnT 2 Ultra", 150, 225, 150, 225, 1 },
{ 0x100, "nVidia Geforce 256 SDR", 150, 200, 100, 150, 1 },
{ 0x101, "nVidia Geforce 256 DDR", 250, 350, 100, 150, 1 },
{ 0x103, "nVidia Quadro", 250, 350, 100, 150, 1 },
{ 0x110, "nVidia Geforce 2 MX/MX400", 150, 250, 150, 250, 1 },
{ 0x111, "nVidia Geforce 2 MX 100/200", 150, 250, 125, 250, 1 },
{ 0x112, "nVidia Geforce 2 GO", 0, 0, 0, 0, 0 },
{ 0x113, "nVidia Quadro 2 MXR/EX/GO", 0, 0, 0, 0, 0 },
{ 0x1a0, "nVidia Geforce 2 MX integrated", 0, 0, 0, 0, 0 },  // unsupported for the moment
{ 0x150, "nVidia Geforce 2 GTS/PRO", 250, 450, 175, 250, 1 },
{ 0x151, "nVidia Geforce 2 Ti", 300, 500, 175, 275, 1 },
{ 0x152, "nVidia Geforce 2 Ultra", 400, 550, 200, 275, 1 },
{ 0x153, "nVidia Quadro 2 Pro", 250, 400, 150, 250, 1 },
{ 0x170, "nVidia Geforce 4 MX460", 450, 600, 275, 350, 1 },
{ 0x171, "nVidia Geforce 4 MX440", 350, 500, 250, 325, 1 },
{ 0x172, "nVidia Geforce 4 MX420", 150, 250, 225, 300, 1 },
{ 0x174, "nVidia Geforce 4 440 Go", 0, 0, 0, 0, 0 },
{ 0x175, "nVidia Geforce 4 420 Go", 0, 0, 0, 0, 0 },
{ 0x176, "nVidia Geforce 4 420 Go 32M", 0, 0, 0, 0, 0 },
{ 0x177, "nVidia Geforce 4 460 Go", 0, 0, 0, 0, 0 },
{ 0x178, "nVidia Quadro 4 500 XGL", 150, 600, 150, 350, 1 }, // needs to be adjusted when the real speeds are known
{ 0x179, "nVidia Geforce 4 440 Go 64M", 0, 0, 0, 0, 0 },
{ 0x17a, "nVidia Quadro 4 200/400 NVS", 150, 250, 225, 300, 1 }, // needs to be adjusted when the real speeds are known
{ 0x17b, "nVidia Quadro 4 550 XGL", 375, 500, 250, 325, 1 },
{ 0x17c, "nVidia Quadro 4 GoGL", 0, 0, 0, 0, 0 },
{ 0x17d, "nVidia Geforce 410 Go", 0, 0, 0, 0, 0 },
{ 0x180, "nVidia Geforce 4 MX440 8X", 300, 600, 200, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x181, "nVidia Geforce 4 MX440 8X", 300, 600, 200, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x182, "nVidia Geforce 4 MX440SE 8X", 300, 600, 150, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x183, "nVidia Geforce 4 MX420 8X", 300, 600, 150, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x186, "NV18", 300, 600, 150, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x187, "NV18", 300, 600, 150, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x188, "nVidia Quadro 4 580 XGL", 300, 600, 200, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x18a, "nVidia Quadro 4 280 NVS", 300, 600, 150, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x18b, "nVidia Quadro 4 380 XGL", 300, 600, 200, 400, 1}, // needs to be adjusted when the real speeds are known
{ 0x1f0, "nVidia Geforce 4 MX integrated",0, 0, 0, 0, 0}, // unsupported for the moment
{ 0x200, "nVidia Geforce 3", 350, 550, 175, 275, 1 },
{ 0x201, "nVidia Geforce 3 Titanium 200", 350, 500, 175, 275, 1 },
{ 0x202, "nVidia Geforce 3 Titanium 500", 450, 600, 200, 300, 1 },
{ 0x203, "nVidia Quadro DCC", 350, 550, 175, 275, 1 },
{ 0x250, "nVidia Geforce 4 Ti 4600", 625, 750, 300, 400, 1 },
{ 0x251, "nVidia Geforce 4 Ti 4400", 450, 600, 275, 375, 1 },
{ 0x253, "nVidia Geforce 4 Ti 4200", 350, 550, 225, 325, 1 },
{ 0x258, "nVidia Quadro 4 900 XGL", 550, 700, 275, 350, 1 },
{ 0x259, "nVidia Quadro 4 750 XGL", 525, 650, 250, 325, 1 },
{ 0x25a, "nVidia Quadro 4 600 XGL", 525, 650, 250, 325, 1 }, // needs to be adjusted when the real speeds are known
{ 0x25b, "nVidia Quadro 4 700 XGL", 450, 600, 200, 300, 1 }, // needs to be adjusted when the real speeds are known
{ 0x280, "nVidia Geforce 4 Ti 4800", 400, 900, 250, 450, 1 }, // needs to be adjusted when the real speeds are known
{ 0x281, "nVidia Geforce 4 Ti 4200 8X", 400, 900, 250, 450, 1 }, // needs to be adjusted when the real speeds are known
{ 0x282, "nVidia Geforce 4 Ti 4800SE", 400, 900, 250, 450, 1 }, // needs to be adjusted when the real speeds are known
{ 0x286, "nVidia Geforce 4 4000 GO", 0, 0, 0, 0, 0 },
{ 0x288, "nVidia Quadro 4 980 XGL", 400, 900, 250, 450, 1 }, // needs to be adjusted when the real speeds are known
{ 0x289, "nVidia Quadro 4 780 XGL", 400, 900, 250, 450, 1 }, // needs to be adjusted when the real speeds are known
{ 0x28c, "nVidia Quadro 4 700 GoGL", 0, 0, 0, 0, 0 },
{ 0x300, "nVidia GeforceFX 5800 Ultra", 250, 550, 550, 1100, 1 }, // A big range is needed for FX cards because of the speed throttle
{ 0x301, "nVidia GeforceFX 5800", 250, 500, 750, 1000, 1 },
{ 0x302, "nVidia GeforceFX", 400, 550, 850, 1100, 1 },
{ 0x308, "nVidia QuadroFX 2000", 250, 550, 550, 1100, 1 },
{ 0x309, "nVidia QuadroFX 1000", 250, 550, 750, 1000, 1 },
{ 0x2a0, "nVidia Xbox GPU", 0, 0, 0, 0, 0 },
{ 0, NULL, 0, 0, 0, 0, 0 }
};


/* This function is needed because nVidia creates other chips than GPU's now too.
*  Assuming a card is a videocard is impossible now. */
int CheckDeviceId(int device_id)
{
    struct pci_ids *nv_ids;
    int found;

    nv_ids = ids;
    while(nv_ids->id != 0)
    {
	if(nv_ids->id == device_id)
	{
	    found = 1;
	    return found;
	}

	nv_ids++;
    }	
    
    return 0;
}

/* Get all info about a card */
const struct card_info *GetCardInfo(int number)
{
    struct card_info *nv_info;
    struct pci_ids *nv_ids;
    riva_hw nv_card;
    int fd, i, found = 0;
    unsigned long base;
    int agp_rate, agp_status;
    
    base = (long)card[number].reg_address;

    nv_info = (struct card_info *) malloc(sizeof(struct card_info));

    nv_info->device_id = card[number].device_id;
    nv_info->reg_address = base;

    nv_ids = ids;
    while(nv_ids->id != 0)
    {
	if(nv_ids->id == nv_info->device_id)
	{
	    nv_info->device_string = (char*)strdup(nv_ids->name);
	    nv_info->memclk_min = nv_ids->mem_min;
	    nv_info->memclk_max = nv_ids->mem_max;
	    nv_info->nvclk_min = nv_ids->nv_min;
	    nv_info->nvclk_max = nv_ids->nv_max;
	    nv_info->supported = nv_ids->supported;
	    found = 1;
	    break;
	}

	nv_ids++;
    }	
    
    if(found == 0) return 0;


    if((fd = openDevMem(number)) == -1 )
    {
	printf("Can't open /dev/mem or /dev/nvidia%d\n", number);
    	return 0;
    }

    /* Map the registers of the nVidia chip */
    nv_card.PMC     = mapDevMem(fd, base + 0x000000, 0x2000);
    nv_card.PEXTDEV = mapDevMem(fd, base + 0x101000, 0x1000);
    nv_card.PFB     = mapDevMem(fd, base + 0x100000, 0x1000);
    nv_card.PRAMDAC = mapDevMem(fd, base + 0x680000, 0x2000);


    /* Memory info */
    nv_info->memory_type = (char*)strdup((nv_card.PFB[0x200/4] & 0x01) ? "DDR" : "SDR");
    nv_info->memory_width = (nv_card.PEXTDEV[0x0/4] & 0x17) ? 128 : 64;

    /* If the card is something TNT based the calculation of the memory is different. */
    if(nv_info->device_id < 0x100)
    {
	if(nv_card.PFB[0x0/4] & 0x100)
	    nv_info->memory_size = ((nv_card.PFB[0x0/4] >> 12) & 0xf)*2+2;
	else
	{
	    switch(nv_card.PFB[0x0/4] & 0x3)
	    {
		case 0:
		    nv_info->memory_size = 32;
		    break;
		case 1:
		    nv_info->memory_size = 4;
		    break;
		case 2:
		    nv_info->memory_size = 8;
		    break;
		case 3:
		    nv_info->memory_size = 16;
		    break;
		default:
		    nv_info->memory_size = 16;
		    break;    	    
	    }
	}
    }
    /* Memory calculation for geforce cards or better. */
    else
    {
	nv_info->memory_size = (nv_card.PFB[0x20c/4] >> 20) & 0xff;
    }
    
    /* 
	Hack.
	Nvidia was so nice to ship support both DDR and SDR memory on some gf2mx and gf4mx cards :(
	Because of this the ranges of the memory speed can be different.
	Check if the card is a gf2mx/gf4mx using DDR and change the speed range.
    */ 
    if((nv_info->device_id == 0x110 || nv_info->device_id == 0x111 || nv_info->device_id == 0x172) && ((nv_card.PFB[0x200/4] & 0x01) == 1))
    {
	nv_info->memclk_max *= 2;
    }
    
    /* If the card is an AGP card */
    if(((nv_card.PEXTDEV[0x0/4]  >> 14) & 0x1) == 0x1)
    {
	/* AGP info */
	nv_info->bus_type = (char*)strdup("AGP");

	agp_status = nv_card.PMC[0x1848/4];
        agp_rate = nv_card.PMC[0x1848/4] & 0x7;
	nv_info->agp_support_rate = (char*)calloc(1, sizeof(char));

	/* If true, the user has AGP8x support */
	if(agp_status & 0x8)
	    agp_rate <<= 2;
	
	for(i=1; i <= 8; i*=2)
	{
	    if(agp_rate & i)
	    {
		char *temp = (char*)calloc(3, sizeof(char));
		sprintf(temp, "%dX ", i);
		nv_info->agp_support_rate = (char*)realloc(nv_info->agp_support_rate, strlen(nv_info->agp_support_rate)+3);
		nv_info->agp_support_rate = strncat(nv_info->agp_support_rate, temp, 3);
		free(temp);
	    }
	}


        nv_info->fw_support = (char*)strdup(((nv_card.PMC[0x1848/4] >> 4) & 0x1) ? "Supported" : "Unsupported");
	nv_info->sba_support = (char*)strdup(((nv_card.PMC[0x1848/4] >> 9) & 0x1) ? "Supported" : "Unsupported");
	nv_info->agp_status = (char*)strdup(((nv_card.PMC[0x184c/4] >> 8) & 0x1) ? "Enabled" : "Disabled");
	nv_info->agp_rate = nv_card.PMC[0x184c/4] & 0x7;
	nv_info->fw_status = (char*)strdup(((nv_card.PMC[0x184c/4] >> 4) & 0x1) ? "Enabled" : "Disabled");
	nv_info->sba_status = (char*)strdup(((nv_card.PMC[0x184c/4] >> 9) & 0x1) ? "Enabled" : "Disabled");
    }

    /* The card is a PCI card */
    else
    {
	/* PCI info */
	nv_info->bus_type = "PCI";
	nv_info->agp_support_rate = (char*)strdup("-");
        nv_info->fw_support = (char*)strdup("-");
	nv_info->sba_support = (char*)strdup("-");
	nv_info->agp_status = (char*)strdup("-");
	nv_info->agp_rate = 0;
	nv_info->fw_status = (char*)strdup("-");
	nv_info->sba_status = (char*)strdup("-");
    }

    return nv_info;
}
