#!/bin/bash 

# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available at
# <http://www.fsf.org/copyleft/gpl.html>, or the OpenIB.org BSD
# license, available in the LICENSE.TXT file accompanying this
# software.  These details are also available at
# <http://openib.org/license.html>.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# Copyright (c) 2004 Mellanox Technologies Ltd.  All rights reserved.
#
# Description: RPM build script
#


fullname() 
{
self=`basename $0`
if [ "$#" -gt 1 ]; then
  echo "$self: more than one argument supplied... forgot some quotation marks?"
elif [ -z "$1" ]; then
  echo "Syntax: `basename $0` [path]"
  echo "  Displays the full (expanded) name of the path given as argument."
else
  if [ -d "$1" ]; then
    ( cd $1; pwd )
  elif [ -e "$1" ]; then
    dir=`dirname "$1"`
    fnm=`basename "$1"`
    #fnm=`echo "$1" | sed "s%.*/%%"`;
    cd "$dir"; fdir=`pwd`;
    echo "$fdir/$fnm"
  else
    echo "$self: $1: no such file or directory" >&2
    exit 1
  fi
fi
}

usage()
{
cat << EOF

    `basename $0`  [ --install_root <Install_Root> ]
    		   [ --prefix <Stack_Prefix> ]
    		   [ --driver_home <Driver Path> ] 
		   [ --compfile(--comp) gcc|pgi|pathscale|intel|<filename.def> ] 
		   [ --csize small|medium|big ]

    --install_root <Install_Root>	Root directory where the MPI tree will be installed. 
    					Use ./. as an argument for the standard root. 
		                        Default: $install_root

    --prefix <Stack_Prefix>		Actual MPI installation path.
    					Default: $STACK_PREFIX

    --driver_home <Driver Path>		Full path to IB stack installation.
    					Default:
					$driver_home

    --compfile gcc|pgi|pathscale|intel  
    			or 
    --compfile <path>/filename.def	Select one of the supported compilers for the MPI build, 
    					or specify the full path to a custom compiler configuration
					file (file name format: <filename>.def).
				        Examples of custom *.def files you can be found in
					the ./compilers directory.
					Default: $compfile

    --csize small|medium|big		Define the cluster size configuration. This parameter can be 
    					tuned during MPI run time via a param file that specifies 
					a different configuration.
					Default: $cluster_size

    --topdir <rpmbuild topdir>      Override the default rpmbuild _topdir

    Usage example.
    build_rpm.sh --prefix /opt/ --driver_home /usr/local/openib \\
    		 --compfile intel --csize small
EOF
}


install_root=/var/tmp/IBED
STACK_PREFIX=/usr/local/ibgd
cluster_size=small
mpich_spec=mpich-osu.spec
base_dir=`dirname $0`
driver_home=$install_root/$STACK_PREFIX/driver/infinihost
update_driver_home=0
compfile=gcc
RELEASE=1
RPM_DIR=$(rpm --eval '%{_topdir}')

while [ $# -gt 0 ]
do
    case $1 in
            --prefix)
                shift
                STACK_PREFIX=$1
            ;;
            --install_root)
                shift
                install_root=$1
            ;;
	    --driver_home)
		shift
		update_driver_home=1
		driver_home=$1
            ;;
            --compfile|--comp)
                shift
		compfile=$1
	    ;;
	    --csize)
	        shift
		cluster_size=$1
            ;;
            --topdir)
                shift
                RPM_DIR=$1
            ;;
            *)
                usage
                exit 1
            ;;
    esac            
    shift
done

if [ $update_driver_home -eq 0 ]; then
	driver_home=$install_root/$STACK_PREFIX/driver/infinihost
fi

if [ ! -f ${base_dir}/BUILD_ID ]; then
    echo "BUILD_ID file does not exist"
    exit 1
fi
source ${base_dir}/BUILD_ID

if [ ! -d ${RPM_DIR} ]; then
    mkdir -p $RPM_DIR/{BUILD,RPMS,SOURCES,SPECS,SRPMS}
fi

cp -f $base_dir/$mpich_spec $RPM_DIR/SPECS


# Preparing the directories and the source files

echo source ${base_dir}/build_mvapich_env.sh --build_root ${RPM_DIR}/BUILD --prefix $STACK_PREFIX --install_root $install_root --driver_home $driver_home --compfile $compfile
source ${base_dir}/build_mvapich_env.sh --build_root ${RPM_DIR}/BUILD --prefix $STACK_PREFIX --install_root $install_root --driver_home $driver_home --compfile $compfile
if [ $? -ne 0 ]; then
	echo "Failed to run build_mvapich_env.sh"
	exit 1
fi

# Building RPM

BUILD_ENV="env STACK_PREFIX=$STACK_PREFIX install_root=$install_root DEF_FILE=$def_file compfile=$compfile compiler=$compiler comp_ver=$comp_ver mvapich_rev=$mvapich_rev build_id=$build_id arch_name=$arch_name MTHOME_INCLUDE=$MTHOME_INCLUDE MTHOME_LIB=$MTHOME_LIB cluster_size=$cluster_size patch_level=$patch_level"

cd ${0%*/*}

#SOURCE_DIR=`pwd`
KERNEL=`uname -r | sed -e "s/\-/\_/g"`

NAME=${rpm_name}_${compiler}${comp_ver}
VERSION=`echo ${mvapich_rev}| sed 's/-/_/g'`$patch_level

build_arch=`rpm --showrc | grep -w "build arch" | cut -d : -f 2 | tr -d " "`

if [ -f /etc/SuSE-release ]; then
   if [ `grep VERSION /etc/SuSE-release | cut -d = -f 2 | cut -d . -f 1 | tr -d [:space:]` -ge 9 ]; then
	$BUILD_ENV rpmbuild -bb --define "_topdir $RPM_DIR" $RPM_DIR/SPECS/$mpich_spec
   else	
	$BUILD_ENV rpm -bb --define "_topdir $RPM_DIR" $RPM_DIR/SPECS/$mpich_spec
   fi
elif [[ -f /etc/redhat-release || -f /etc/rocks-release ]]; then
	$BUILD_ENV rpmbuild -bb --define "_topdir $RPM_DIR" $RPM_DIR/SPECS/$mpich_spec
else
        echo Build rpm on this system is not supported...
        exit 1
fi

if [ $? != 0 ]; then
    echo RPM building failed...
    exit 1
fi 

rm -f $RPM_DIR/SPECS/$mpich_spec

echo "$RPM_DIR/RPMS/$build_arch/${NAME}-${VERSION}-${RELEASE}.${build_arch}.rpm"

if [ ! -f $RPM_DIR/RPMS/$build_arch/${NAME}-${VERSION}-${RELEASE}.${build_arch}.rpm ] ; then
	echo RPM building failed...
	exit 1
else
	echo RPM: $RPM_DIR/RPMS/$build_arch/${NAME}-${VERSION}-${RELEASE}.${build_arch}.rpm 
fi

exit 0
