/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_sa_class_port_info.c 6540 2006-04-20 16:55:11Z halr $
 */


/*
 * Abstract:
 *    Implementation of osm_cpi_rcv_t.
 * This object represents the ClassPortInfo Receiver object.
 * This object is part of the opensm family of objects.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.8 $
 */

/*
  Next available error code: 0x403
*/

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <iba/ib_types.h>
#include <complib/cl_memory.h>
#include <complib/cl_qmap.h>
#include <complib/cl_passivelock.h>
#include <complib/cl_debug.h>
#include <complib/cl_qlist.h>
#include <opensm/osm_sa_class_port_info.h>
#include <vendor/osm_vendor.h>
#include <vendor/osm_vendor_api.h>
#include <opensm/osm_helper.h>

#define MAX_MSECS_TO_RTV 24
/* Precalculated table in msec (index is related to encoded value) */
/* 4.096 usec * 2 ** n (where n = 8 - 31) */
static uint32_t __msecs_to_rtv_table[MAX_MSECS_TO_RTV] =
					{ 1, 2, 4, 8,
					  16, 33, 67, 134, 
					  268, 536, 1073, 2147,
					  4294, 8589, 17179, 34359,
					  68719, 137438, 274877, 549755,
					  1099511, 2199023, 4398046, 8796093 };

/**********************************************************************
 **********************************************************************/
void
osm_cpi_rcv_construct(
  IN osm_cpi_rcv_t* const p_rcv )
{
  cl_memclr( p_rcv, sizeof(*p_rcv) );
}

/**********************************************************************
 **********************************************************************/
void
osm_cpi_rcv_destroy(
  IN osm_cpi_rcv_t* const p_rcv )
{
  OSM_LOG_ENTER( p_rcv->p_log, osm_cpi_rcv_destroy );
  OSM_LOG_EXIT( p_rcv->p_log );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_cpi_rcv_init(
  IN osm_cpi_rcv_t*     const p_rcv,
  IN osm_sa_resp_t*     const p_resp,
  IN osm_mad_pool_t*    const p_mad_pool,
  IN osm_subn_t*        const p_subn,
  IN osm_log_t*         const p_log,
  IN cl_plock_t*        const p_lock )
{
  ib_api_status_t status = IB_SUCCESS;

  OSM_LOG_ENTER( p_log, osm_cpi_rcv_init );

  osm_cpi_rcv_construct( p_rcv );

  p_rcv->p_log = p_log;
  p_rcv->p_subn = p_subn;
  p_rcv->p_lock = p_lock;
  p_rcv->p_resp = p_resp;
  p_rcv->p_mad_pool = p_mad_pool;

  OSM_LOG_EXIT( p_rcv->p_log );
  return( status );
}

/**********************************************************************
 **********************************************************************/
static void
__osm_cpi_rcv_respond(
  IN osm_cpi_rcv_t*        const p_rcv,
  IN const osm_madw_t*     const p_madw)
{
  osm_madw_t*              p_resp_madw;
  const ib_sa_mad_t*    p_sa_mad;
  ib_sa_mad_t*          p_resp_sa_mad;
  ib_class_port_info_t    *p_resp_cpi;
  ib_api_status_t       status;
  ib_gid_t                zero_gid;
  uint8_t                   rtv;

  OSM_LOG_ENTER( p_rcv->p_log, __osm_cpi_rcv_respond );

  cl_memclr(&zero_gid, sizeof(ib_gid_t));

  /*
    Get a MAD to reply. Address of Mad is in the received mad_wrapper
  */
  p_resp_madw = osm_mad_pool_get( p_rcv->p_mad_pool, 
                                  p_madw->h_bind,
                                  MAD_BLOCK_SIZE,
                                  &p_madw->mad_addr );
  if( !p_resp_madw )
  {
    osm_log( p_rcv->p_log, OSM_LOG_ERROR,
             "__osm_cpi_rcv_respond: ERR 1408: "
             "Unable to allocate MAD\n" );
    goto Exit;
  }

  p_sa_mad = osm_madw_get_sa_mad_ptr( p_madw );
  p_resp_sa_mad = osm_madw_get_sa_mad_ptr( p_resp_madw );

  cl_memcpy( p_resp_sa_mad, p_sa_mad, IB_SA_MAD_HDR_SIZE );
  p_resp_sa_mad->method |= IB_MAD_METHOD_RESP_MASK;
  /* C15-0.1.5 - always return SM_Key = 0 (table 185 p 884) */
  p_resp_sa_mad->sm_key = 0;
  p_resp_sa_mad->paylen_newwin = 0;

  p_resp_cpi = (ib_class_port_info_t*)ib_sa_mad_get_payload_ptr( p_resp_sa_mad );

  /* finally do it (the job) man ! */
  p_resp_cpi->base_ver = 1;
  p_resp_cpi->class_ver = 2;
  /* Calculate encoded response time value */
  /* transaction timeout is in msec */
  if (p_rcv->p_subn->opt.transaction_timeout > __msecs_to_rtv_table[MAX_MSECS_TO_RTV])
    rtv = MAX_MSECS_TO_RTV - 1;
  else {
    for (rtv = 0; rtv < MAX_MSECS_TO_RTV; rtv++) {
      if (p_rcv->p_subn->opt.transaction_timeout <= __msecs_to_rtv_table[rtv])
         break;
    }
  }
  rtv += 8;
  p_resp_cpi->resp_time_val = rtv;
  p_resp_cpi->redir_gid = zero_gid;
  p_resp_cpi->redir_tc_sl_fl = 0;
  p_resp_cpi->redir_lid = 0;
  p_resp_cpi->redir_pkey = 0;
  p_resp_cpi->redir_qp = CL_NTOH32(1);
  p_resp_cpi->redir_qkey = IB_QP1_WELL_KNOWN_Q_KEY;
  p_resp_cpi->trap_gid = zero_gid;
  p_resp_cpi->trap_tc_sl_fl = 0;
  p_resp_cpi->trap_lid = 0;
  p_resp_cpi->trap_pkey = 0;
  p_resp_cpi->trap_hop_qp = 0;
  p_resp_cpi->trap_qkey = IB_QP1_WELL_KNOWN_Q_KEY;

  /* set specific capability mask bits */
  /* we do not support the optionals:
     OSM_CAP_IS_SUBN_OPT_RECS_SUP :
     SwitchInfoRecord,
     LinearForwardingTableRecord, (we do support it under the table)
     RandomForwardingTableRecord,
     MulticastForwardingTableRecord,
     SMInfoRecord, (we do support it under the table)
     InformInfoRecord,
     LinkRecord, (we do support it under the table)
     ServiceAssociationRecord

     OSM_CAP_IS_MULTIPATH_SUP:
     MultiPathRecord,
     TraceRecord

     OSM_CAP_IS_REINIT_SUP:
     For reinitialization functionality.

     So not sending traps, but supporting Get(Notice) and Set(Notice).
  */

  /* Note host notation replaced later */
  p_resp_cpi->cap_mask = OSM_CAP_IS_SUBN_GET_SET_NOTICE_SUP |
                         OSM_CAP_IS_PORT_INFO_CAPMASK_MATCH_SUPPORTED;
  if (p_rcv->p_subn->opt.no_multicast_option != TRUE)
    p_resp_cpi->cap_mask |= OSM_CAP_IS_UD_MCAST_SUP;
  p_resp_cpi->cap_mask = cl_hton16(p_resp_cpi->cap_mask);

  if( osm_log_is_active( p_rcv->p_log, OSM_LOG_FRAMES ) )
    osm_dump_sa_mad( p_rcv->p_log, p_resp_sa_mad, OSM_LOG_FRAMES );

  status = osm_vendor_send( p_resp_madw->h_bind, p_resp_madw,  FALSE );
  if( status != IB_SUCCESS )
  {
    osm_log( p_rcv->p_log, OSM_LOG_ERROR,
             "__osm_cpi_rcv_respond: ERR 1409: "
             "Unable to send MAD (%s)\n", ib_get_err_str( status ) );
    /*  osm_mad_pool_put( p_rcv->p_mad_pool, p_resp_madw ); */
    goto Exit;
  }

 Exit:
  OSM_LOG_EXIT( p_rcv->p_log );
}

/**********************************************************************
 * This code actually handles the call
 **********************************************************************/
void
osm_cpi_rcv_process(
  IN osm_cpi_rcv_t*        const p_rcv,
  IN const osm_madw_t*     const p_madw )
{
  const ib_path_rec_t*     p_pr;
  const ib_sa_mad_t*    p_sa_mad;

  OSM_LOG_ENTER( p_rcv->p_log, osm_cpi_rcv_process );

  CL_ASSERT( p_madw );

  p_sa_mad = osm_madw_get_sa_mad_ptr( p_madw );

  /* we only supports GET */
  if (p_sa_mad->method != IB_MAD_METHOD_GET)
  {
    osm_log( p_rcv->p_log, OSM_LOG_ERROR,
             "osm_cpi_rcv_process: ERR 1403: "
             "Unsupported Method (%s)\n",
             ib_get_sa_method_str( p_sa_mad->method ) );
    osm_sa_send_error( p_rcv->p_resp, p_madw, IB_SA_MAD_STATUS_REQ_INVALID);
    goto Exit;
  }

  p_pr = (ib_path_rec_t*)ib_sa_mad_get_payload_ptr( p_sa_mad );

  CL_ASSERT( p_sa_mad->attr_id == IB_MAD_ATTR_CLASS_PORT_INFO );

  /*
    CLASS PORT INFO does not really look on the SMDB - no lock required.
  */

  __osm_cpi_rcv_respond( p_rcv, p_madw);

 Exit:
  OSM_LOG_EXIT( p_rcv->p_log );
}
