/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_mtree.c 7353 2006-05-19 14:19:36Z halr $
 */


/*
 * Abstract:
 *    Implementation of osm_mtree_node_t.
 * This file implements the Multicast Tree object.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.5 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <complib/cl_debug.h>
#include <opensm/osm_mtree.h>

/**********************************************************************
 **********************************************************************/
void
osm_mtree_node_init(
  IN osm_mtree_node_t*     const p_mtn,
  IN const osm_switch_t*      const p_sw )
{
  uint32_t              i;

  CL_ASSERT( p_mtn );
  CL_ASSERT( p_sw );

  osm_mtree_node_construct( p_mtn );

  p_mtn->p_sw = (osm_switch_t*)p_sw;
  p_mtn->max_children = osm_switch_get_num_ports( p_sw );

  for( i = 0; i < p_mtn->max_children; i++ )
    p_mtn->child_array[i] = NULL;
}

/**********************************************************************
 **********************************************************************/
osm_mtree_node_t*
osm_mtree_node_new(
  IN const osm_switch_t* const p_sw )
{
  osm_mtree_node_t *p_mtn;

  p_mtn = cl_malloc( sizeof(osm_mtree_node_t) +
                     sizeof(void*) * (osm_switch_get_num_ports( p_sw ) - 1) );

  if( p_mtn != NULL )
    osm_mtree_node_init( p_mtn, p_sw );

  return( p_mtn );
}

/**********************************************************************
 **********************************************************************/
void
osm_mtree_destroy(
  IN osm_mtree_node_t *p_mtn )
{
  uint32_t i;

  if (p_mtn == NULL)
    return;

  if ( p_mtn->child_array != NULL ) 
    for (i = 0 ; i< p_mtn->max_children; i++ )
      if ( (p_mtn->child_array[i] != NULL) && 
           (p_mtn->child_array[i] != OSM_MTREE_LEAF) )
        osm_mtree_destroy(p_mtn->child_array[i]);
  
  cl_free( p_mtn );
}

/**********************************************************************
 **********************************************************************/
void
__osm_mtree_dump(
  IN osm_mtree_node_t *p_mtn )
{
  uint32_t i;

  if (p_mtn == NULL)
    return;

  printf("GUID:0x%016" PRIx64 " max_children:%d\n", 
         p_mtn->p_sw->p_node->node_info.node_guid,
         p_mtn->max_children );
  if ( p_mtn->child_array != NULL ) 
  {
    for (i = 0 ; i< p_mtn->max_children; i++ )
    {
      printf("i=%d\n", i);
      if ( (p_mtn->child_array[i] != NULL) && (p_mtn->child_array[i] != OSM_MTREE_LEAF) )
        __osm_mtree_dump(p_mtn->child_array[i]);
    }
  }
  
}
