/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_matrix.c 4246 2005-11-30 15:42:42Z halr $
 */


/*
 * Abstract:
 *    Implementation of osm_lid_matrix_t.
 * This file implements the LID Matrix object.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.7 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <opensm/osm_matrix.h>


/**********************************************************************
 **********************************************************************/
void
osm_lid_matrix_destroy(
  IN osm_lid_matrix_t* const p_lmx )
{
  cl_vector_destroy( &p_lmx->lid_vec );
}

/**********************************************************************
 Initializer function called by cl_vector
**********************************************************************/
cl_status_t
__osm_lid_matrix_vec_init(
  IN void* const p_elem,
  IN void* context )
{
  osm_lid_matrix_t* const p_lmx = (osm_lid_matrix_t*)context;

  cl_memset( p_elem, OSM_NO_PATH, p_lmx->num_ports + 1);
  return( CL_SUCCESS );
}

/**********************************************************************
 Initializer function called by cl_vector
**********************************************************************/
void
__osm_lid_matrix_vec_clear(
  IN const size_t index,
  IN void* const p_elem,
  IN void* context )
{
  osm_lid_matrix_t* const p_lmx = (osm_lid_matrix_t*)context;

  UNUSED_PARAM( index );
  cl_memset( p_elem, OSM_NO_PATH, p_lmx->num_ports + 1);
}

/**********************************************************************
 **********************************************************************/
void
osm_lid_matrix_clear(
  IN osm_lid_matrix_t* const p_lmx )
{
  cl_vector_apply_func( &p_lmx->lid_vec,
                        __osm_lid_matrix_vec_clear, p_lmx );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_lid_matrix_init(
  IN osm_lid_matrix_t* const p_lmx,
  IN const uint8_t num_ports )
{
  cl_vector_t *p_vec;
  cl_status_t status;

  CL_ASSERT( p_lmx );
  CL_ASSERT( num_ports );

  p_lmx->num_ports = num_ports;

  p_vec = &p_lmx->lid_vec;
  /*
    Initialize the vector for the number of ports plus an
    extra entry to hold the "least-hops" count for that LID.
  */
  status = cl_vector_init( p_vec,
                           0,             /* min_size, */
                           1,             /* grow_size */
                           sizeof(uint8_t)*(num_ports + 1), /*  element size */
                           __osm_lid_matrix_vec_init, /*  init function */
                           NULL,           /*  destory func */
                           p_lmx          /*  context */
                           );

  return( status );
}

/**********************************************************************
 **********************************************************************/
cl_status_t
osm_lid_matrix_set(
  IN osm_lid_matrix_t* const p_lmx,
  IN const uint16_t lid_ho,
  IN const uint8_t port_num,
  IN const uint8_t val )
{
  uint8_t *p_port_array;
  cl_status_t status;

  CL_ASSERT( port_num < p_lmx->num_ports );
  status = cl_vector_set_min_size( &p_lmx->lid_vec, lid_ho + 1 );
  if( status == CL_SUCCESS )
  {
    p_port_array = (uint8_t *)cl_vector_get_ptr( &p_lmx->lid_vec, lid_ho );
    p_port_array[port_num] = val;
    if( p_port_array[p_lmx->num_ports] > val )
      p_port_array[p_lmx->num_ports] = val;
  }
  return( status );
}
