/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2006 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_log.c 7617 2006-06-01 13:00:12Z halr $
 */


/*
 * Abstract:
 *    Implementaion of osm_log_t.
 * This object represents the log file.
 * This object is part of the opensm family of objects.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.8 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <opensm/osm_log.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifndef WIN32
#include <sys/time.h>
#include <unistd.h>
#include <complib/cl_timer.h>

static char *month_str[] = {
  "Jan",
  "Feb",
  "Mar",
  "Apr",
  "May",
  "Jun",
  "Jul",
  "Aug",
  "Sep",
  "Oct",
  "Nov",
  "Dec"
};
#endif /* ndef WIN32 */

void
osm_log(
  IN osm_log_t* const p_log,
  IN const osm_log_level_t verbosity,
  IN const char *p_str, ... )
{
  char      buffer[LOG_ENTRY_SIZE_MAX];
  va_list   args;
  int       ret;

#ifdef WIN32
  SYSTEMTIME st;
  uint32_t pid = GetCurrentThreadId();
#else
  pid_t pid = 0;
  time_t tim;
  struct tm result;
  uint64_t time_usecs;
  uint32_t usecs;

  time_usecs = cl_get_time_stamp();
  tim = time_usecs/1000000;
  usecs = time_usecs % 1000000;
  localtime_r(&tim, &result);
#endif /* WIN32 */

  /* If this is a call to syslog - always print it */
  if ( verbosity & OSM_LOG_SYS )
  {
    /* this is a call to the syslog */
    va_start( args, p_str );
    vsprintf( buffer, p_str, args );
    va_end(args);
    cl_log_event("OpenSM", LOG_INFO, buffer , NULL, 0);

    /* SYSLOG should go to stdout too */
    if (p_log->out_port != stdout)
    {
      printf("%s\n", buffer);
      fflush( stdout );
    }

    /* send it also to the log file */
#ifdef WIN32
    GetLocalTime(&st);
    fprintf( p_log->out_port, "[%02d:%02d:%02d:%03d][%04X] -> %s",
                     st.wHour, st.wMinute, st.wSecond, st.wMilliseconds,
                     pid, buffer);
#else
    fprintf( p_log->out_port, "%s %02d %02d:%02d:%02d %06d [%04X] -> %s\n",
             (result.tm_mon < 12 ? month_str[result.tm_mon] : "???"),
             result.tm_mday, result.tm_hour,
             result.tm_min, result.tm_sec,
             usecs, pid, buffer);
    fflush( p_log->out_port );
#endif    
  }

  /* SYS messages go to the log anyways */
  if (p_log->level & verbosity)
  {
#ifdef _MEM_DEBUG_MODE_
    /* If we are running in MEM_DEBUG_MODE then 
       the cl_mem_check will be called on every run */
    if (cl_mem_check() == FALSE) 
    {
      fprintf( p_log->out_port, "*** MEMORY ERROR!!! ***\n" );
      CL_ASSERT(0);
    }
#endif
    
    va_start( args, p_str );
    vsprintf( buffer, p_str, args );
    va_end(args);
    
    /* regular log to default out_port */
    cl_spinlock_acquire( &p_log->lock );
#ifdef WIN32
    GetLocalTime(&st);
    ret = fprintf(   p_log->out_port, "[%02d:%02d:%02d:%03d][%04X] -> %s",
                     st.wHour, st.wMinute, st.wSecond, st.wMilliseconds,
                     pid, buffer);
    
#else
    pid = pthread_self();
    tim = time(NULL);
    ret = fprintf( p_log->out_port, "%s %02d %02d:%02d:%02d %06d [%04X] -> %s",
                   ((result.tm_mon < 12) && (result.tm_mon >= 0) ? 
                    month_str[result.tm_mon] : "???"),
                   result.tm_mday, result.tm_hour,
                   result.tm_min, result.tm_sec,
                   usecs, pid, buffer);
#endif /*  WIN32 */
    
    /*
      Flush log on errors too.
    */
    if( p_log->flush || (verbosity & OSM_LOG_ERROR) )
      fflush( p_log->out_port );
    
    cl_spinlock_release( &p_log->lock );
    
    if (ret < 0)
    {
      fprintf(stderr, "OSM LOG FAILURE! Probably quota exceeded\n");
      exit(1);
    }
  }
}

void
osm_log_raw(
  IN osm_log_t* const p_log,
  IN const osm_log_level_t verbosity,
  IN const char *p_buf )
{
  if( p_log->level & verbosity )
  {
    cl_spinlock_acquire( &p_log->lock );
    printf( "%s", p_buf );
    cl_spinlock_release( &p_log->lock );

    /*
      Flush log on errors too.
    */
    if( p_log->flush || (verbosity & OSM_LOG_ERROR) )
      fflush( stdout );
  }
}

boolean_t
osm_is_debug(void)
{
#if defined( _DEBUG_ )
  return TRUE;
#else
  return FALSE;
#endif /* defined( _DEBUG_ ) */
}
