/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_helper.c 6579 2006-04-24 15:30:05Z halr $
 */


/*
 * Abstract:
 *    Implementation of opensm helper functions.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.19 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdlib.h>
#include <stdio.h>
#include <complib/cl_memory.h>
#include <complib/cl_debug.h>
#include <iba/ib_types.h>
#include <opensm/osm_helper.h>
#include <opensm/osm_log.h>

#define LINE_LENGTH 256

/* we use two tables - one for queries and one for responses */
const char* const __ib_sa_method_str[] =
{
  "RESERVED",                 /* 0 */
  "SubnAdmGet",             /* 1 */
  "SubnAdmSet",             /* 2 */
  "RESERVED",                  /* 3 */
  "RESERVED",                  /* 4 */
  "RESERVED",                  /* 5 */
  "SubnAdmReport",          /* 6 */
  "RESERVED",                  /* 7 */
  "RESERVED",                  /* 8 */
  "RESERVED",                  /* 9 */
  "RESERVED",                  /* A */
  "RESERVED",                  /* B */
  "RESERVED",                  /* C */
  "RESERVED",                  /* D */
  "RESERVED",                  /* E */
  "RESERVED",                  /* F */
  "RESERVED",                  /* 10 */
  "RESERVED",                  /* 11 */
  "SubnAdmGetTable",           /* 12 */
  "SubnAdmGetTraceTable",      /* 13 */
  "SubnAdmGetMulti",           /* 14 */
  "SubnAdmDelete",          /* 15 */
  "UNKNOWN"                   /* 16 */
};

const char* const __ib_sa_resp_method_str[] =
{
  "RESERVED",                 /* 80 */
  "SubnAdmGetResp",            /* 81 */
  "RESERVED (SetResp?)",    /* 82 */
  "RESERVED",                  /* 83 */
  "RESERVED",                  /* 84 */
  "RESERVED",                  /* 85 */
  "SubnAdmReportResp",         /* 86 */
  "RESERVED",                  /* 87 */
  "RESERVED",                  /* 88 */
  "RESERVED",                  /* 89 */
  "RESERVED",                  /* 8A */
  "RESERVED",                  /* 8B */
  "RESERVED",                  /* 8C */
  "RESERVED",                  /* 8D */
  "RESERVED",                  /* 8E */
  "RESERVED",                  /* 8F */
  "RESERVED",                  /* 90 */
  "RESERVED",                  /* 91 */
  "SubnAdmGetTableResp",       /* 92 */
  "RESERVED",                  /* 93 */
  "SubnAdmGetMultiResp",    /* 94 */
  "SubnAdmDeleteResp",         /* 95 */
  "UNKNOWN"
};

#define OSM_SA_METHOD_STR_UNKNOWN_VAL 0x16

const char* const __ib_sm_method_str[] =
{
  "RESERVED0",              /* 0 */
  "SubnGet",              /* 1 */
  "SubnSet",              /* 2 */
  "RESERVED3",               /* 3 */
  "RESERVED4",               /* 4 */
  "SubnTrap",                /* 5 */
  "RESERVED6",               /* 6 */
  "SubnTrapRepress",         /* 7 */
  "RESERVED8",               /* 8 */
  "RESERVED9",               /* 9 */
  "RESERVEDA",               /* A */
  "RESERVEDB",               /* B */
  "RESERVEDC",               /* C */
  "RESERVEDD",               /* D */
  "RESERVEDE",               /* E */
  "RESERVEDF",               /* F */
  "RESERVED10",              /* 10 */
  "SubnGetResp",             /* 11 */
  "RESERVED12",           /* 12 */
  "RESERVED13",           /* 13 */
  "RESERVED14",           /* 14 */
  "RESERVED15",           /* 15 */
  "RESERVED16",              /* 16 */
  "RESERVED17",           /* 17 */
  "RESERVED18",           /* 18 */
  "RESERVED19",           /* 19 */
  "RESERVED1A",           /* 1A */
  "RESERVED1B",           /* 1B */
  "RESERVED1C",           /* 1C */
  "RESERVED1D",           /* 1D */
  "RESERVED1E",             /* 1E */
  "RESERVED1F",             /* 1F */
  "UNKNOWN"                  /* 20 */
};

#define OSM_SM_METHOD_STR_UNKNOWN_VAL 0x21

const char* const __ib_sm_attr_str[] =
{
  "RESERVED",                     /* 0 */
  "ClassPortInfo",             /* 1 */
  "Notice",                    /* 2 */
  "InformInfo",                /* 3 */
  "RESERVED",                     /* 4 */
  "RESERVED",                     /* 5 */
  "RESERVED",                     /* 6 */
  "RESERVED",                     /* 7 */
  "RESERVED",                     /* 8 */
  "RESERVED",                     /* 9 */
  "RESERVED",                     /* A */
  "RESERVED",                     /* B */
  "RESERVED",                     /* C */
  "RESERVED",                     /* D */
  "RESERVED",                     /* E */
  "RESERVED",                     /* F */
  "NodeDescription",           /* 10 */
  "NodeInfo",                     /* 11 */
  "SwitchInfo",                /* 12 */
  "UNKNOWN",                   /* 13 */
  "GUIDInfo",                     /* 14 */
  "PortInfo",                     /* 15 */
  "P_KeyTable",                /* 16 */
  "SLtoVLMappingTable",           /* 17 */
  "VLArbitrationTable",           /* 18 */
  "LinearForwardingTable",        /* 19 */
  "RandomForwardingTable",        /* 1A */
  "MulticastForwardingTable",        /* 1B */
  "UNKNOWN",                   /* 1C */
  "UNKNOWN",                   /* 1D */
  "UNKNOWN",                   /* 1E */
  "UNKNOWN",                   /* 1F */
  "SMInfo",                    /* 20 */
  "UNKNOWN"                    /* 21 - always highest value*/
};

#define OSM_SM_ATTR_STR_UNKNOWN_VAL 0x21

const char* const __ib_sa_attr_str[] =
{
  "RESERVED",                     /* 0 */
  "ClassPortInfo",             /* 1 */
  "Notice",                    /* 2 */
  "InformInfo",                /* 3 */
  "RESERVED",                     /* 4 */
  "RESERVED",                     /* 5 */
  "RESERVED",                     /* 6 */
  "RESERVED",                     /* 7 */
  "RESERVED",                     /* 8 */
  "RESERVED",                     /* 9 */
  "RESERVED",                     /* A */
  "RESERVED",                     /* B */
  "RESERVED",                     /* C */
  "RESERVED",                     /* D */
  "RESERVED",                     /* E */
  "RESERVED",                     /* F */
  "RESERVED",                     /* 10 */
  "NodeRecord",                /* 11 */
  "PortInfoRecord",               /* 12 */
  "SLtoVLMappingTableRecord",        /* 13 */
  "SwitchInfoRecord",             /* 14 */
  "LinearForwardingTableRecord",     /* 15 */
  "RandomForwardingTableRecord",     /* 16 */
  "MulticastForwardingTableRecord",  /* 17 */
  "SMInfoRecord",                 /* 18 */
  "RESERVED",                     /* 19 */
  "RandomForwardingTable",        /* 1A */
  "MulticastForwardingTable",        /* 1B */
  "UNKNOWN",                   /* 1C */
  "UNKNOWN",                   /* 1D */
  "UNKNOWN",                   /* 1E */
  "UNKNOWN",                   /* 1F */
  "LinkRecord",                /* 20 */
  "UNKNOWN",                   /* 21 */
  "UNKNOWN",                   /* 22 */
  "UNKNOWN",                   /* 23 */
  "UNKNOWN",                   /* 24 */
  "UNKNOWN",                   /* 25 */
  "UNKNOWN",                   /* 26 */
  "UNKNOWN",                   /* 27 */
  "UNKNOWN",                   /* 28 */
  "UNKNOWN",                   /* 29 */
  "UNKNOWN",                   /* 2A */
  "UNKNOWN",                   /* 2B */
  "UNKNOWN",                   /* 2C */
  "UNKNOWN",                   /* 2D */
  "UNKNOWN",                   /* 2E */
  "UNKNOWN",                   /* 2F */
  "GuidInfoRecord",               /* 30 */
  "ServiceRecord",             /* 31 */
  "UNKNOWN",                   /* 32 */
  "P_KeyTableRecord",             /* 33 */
  "UNKNOWN",                   /* 34 */
  "PathRecord",                /* 35 */
  "VLArbitrationTableRecord",        /* 36 */
  "UNKNOWN",                   /* 37 */
  "MCMemberRecord",               /* 38 */
  "TraceRecord",                  /* 39 */
  "MultiPathRecord",              /* 3A */
  "ServiceAssociationRecord",        /* 3B */
  "UNKNOWN"                    /* 3C - always highest value*/
};

#define OSM_SA_ATTR_STR_UNKNOWN_VAL 0x3C


/**********************************************************************
 **********************************************************************/
const char*
ib_get_sa_method_str(
  IN uint8_t method )
{
  if (method & 0x80)
  {
    method = method & 0x7f;
    if( method >= OSM_SA_METHOD_STR_UNKNOWN_VAL  )
      method = OSM_SA_METHOD_STR_UNKNOWN_VAL;
    /* it is a response - use the response table */
    return( __ib_sa_resp_method_str[method] );
  }
  else
  {
    if( method >= OSM_SA_METHOD_STR_UNKNOWN_VAL  )
      method = OSM_SA_METHOD_STR_UNKNOWN_VAL;
    return( __ib_sa_method_str[method] );
  }
}

/**********************************************************************
 **********************************************************************/
const char*
ib_get_sm_method_str(
  IN uint8_t method )
{
  if (method & 0x80) method = (method & 0x0F) | 0x10;
  if( method >= OSM_SM_METHOD_STR_UNKNOWN_VAL  )
    method = OSM_SM_METHOD_STR_UNKNOWN_VAL;
  return( __ib_sm_method_str[method] );
}

/**********************************************************************
 **********************************************************************/
const char*
ib_get_sm_attr_str(
  IN ib_net16_t attr )
{
  uint16_t host_attr;
  host_attr = cl_ntoh16( attr );

  if( host_attr >= OSM_SM_ATTR_STR_UNKNOWN_VAL  )
    host_attr = OSM_SM_ATTR_STR_UNKNOWN_VAL;

  return( __ib_sm_attr_str[host_attr] );
}

/**********************************************************************
 **********************************************************************/
const char*
ib_get_sa_attr_str(
  IN ib_net16_t attr )
{
  uint16_t host_attr;
  host_attr = cl_ntoh16( attr );

  if( host_attr >= OSM_SA_ATTR_STR_UNKNOWN_VAL  )
    host_attr = OSM_SA_ATTR_STR_UNKNOWN_VAL;

  return( __ib_sa_attr_str[host_attr] );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_dbg_do_line(
  IN char** pp_local,
  IN const uint32_t buf_size,
  IN const char* const p_prefix_str,
  IN const char* const p_new_str,
  IN uint32_t* const p_total_len )
{
  char line[LINE_LENGTH];
  uint32_t len;

  sprintf( line, "%s%s", p_prefix_str, p_new_str );
  len = (uint32_t) strlen( line );
  *p_total_len += len;
  if( *p_total_len + sizeof('\0') > buf_size )
    return( IB_INSUFFICIENT_MEMORY );

  strcpy( *pp_local, line );
  *pp_local += len;
  return( IB_SUCCESS );
}

/**********************************************************************
 **********************************************************************/
void
osm_dbg_get_capabilities_str(
  IN char* p_buf,
  IN const uint32_t buf_size,
  IN const char* const p_prefix_str,
  IN const ib_port_info_t* const p_pi )
{
  uint32_t total_len = 0;
  char *p_local = p_buf;

  strcpy( p_local, "Capabilities Mask:\n" );
  p_local += strlen( p_local );

  if( p_pi->capability_mask & IB_PORT_CAP_RESV0 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV0\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_IS_SM )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_IS_SM\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_NOTICE )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_NOTICE\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_TRAP )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_TRAP\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_IPD )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_IPD\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_AUTO_MIG )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_AUTO_MIG\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_SL_MAP )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_SL_MAP\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_NV_MKEY )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_NV_MKEY\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_NV_PKEY )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_NV_PKEY\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_LED_INFO )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_LED_INFO\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_SM_DISAB )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_SM_DISAB\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_SYS_IMG_GUID )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_SYS_IMG_GUID\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_PKEY_SW_EXT_PORT_TRAP )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_PKEY_SW_EXT_PORT_TRAP\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV13 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV13\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV14 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV14\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV15 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV15\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_COM_MGT )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_COM_MGT\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_SNMP )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_SNMP\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_REINIT )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_REINIT\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_DEV_MGT )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_DEV_MGT\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_VEND_CLS )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_VEND_CLS\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_DR_NTC )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_DR_NTC\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_CAP_NTC )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_CAP_NTC\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_BM )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_BM\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_LINK_RT_LATENCY )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_LINK_RT_LATENCY\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_HAS_CLIENT_REREG )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_HAS_CLIENT_REREG\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV26 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV26\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV27 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV27\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV28)
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV28\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV29 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV29\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV30 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV30\n", &total_len ) != IB_SUCCESS )
      return;
  }
  if( p_pi->capability_mask & IB_PORT_CAP_RESV31 )
  {
    if( osm_dbg_do_line( &p_local, buf_size, p_prefix_str,
                         "IB_PORT_CAP_RESV31\n", &total_len ) != IB_SUCCESS )
      return;
  }

  return;
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_port_info(
  IN osm_log_t* const p_log,
  IN const ib_net64_t node_guid,
  IN const ib_net64_t port_guid,
  IN const uint8_t port_num,
  IN const ib_port_info_t* const p_pi,
  IN const osm_log_level_t log_level )
{
  char buf[BUF_SIZE];

  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, log_level,
             "PortInfo dump:\n"
             "\t\t\t\tport number.............0x%X\n"
             "\t\t\t\tnode_guid...............0x%016" PRIx64 "\n"
             "\t\t\t\tport_guid...............0x%016" PRIx64 "\n"
             "\t\t\t\tm_key...................0x%016" PRIx64 "\n"
             "\t\t\t\tsubnet_prefix...........0x%016" PRIx64 "\n"
             "\t\t\t\tbase_lid................0x%X\n"
             "\t\t\t\tmaster_sm_base_lid......0x%X\n"
             "\t\t\t\tcapability_mask.........0x%X\n"
             "\t\t\t\tdiag_code...............0x%X\n"
             "\t\t\t\tm_key_lease_period......0x%X\n"
             "\t\t\t\tlocal_port_num..........0x%X\n"
             "\t\t\t\tlink_width_enabled......0x%X\n"
             "\t\t\t\tlink_width_supported....0x%X\n"
             "\t\t\t\tlink_width_active.......0x%X\n"
             "\t\t\t\tlink_speed_supported....0x%X\n"
             "\t\t\t\tport_state..............%s\n"
             "\t\t\t\tstate_info2.............0x%X\n"
             "\t\t\t\tm_key_protect_bits......0x%X\n"
             "\t\t\t\tlmc.....................0x%X\n"
             "\t\t\t\tlink_speed..............0x%X\n"
             "\t\t\t\tmtu_smsl................0x%X\n"
             "\t\t\t\tvl_cap_init_type........0x%X\n"
             "\t\t\t\tvl_high_limit...........0x%X\n"
             "\t\t\t\tvl_arb_high_cap.........0x%X\n"
             "\t\t\t\tvl_arb_low_cap..........0x%X\n"
             "\t\t\t\tinit_rep_mtu_cap........0x%X\n"
             "\t\t\t\tvl_stall_life...........0x%X\n"
             "\t\t\t\tvl_enforce..............0x%X\n"
             "\t\t\t\tm_key_violations........0x%X\n"
             "\t\t\t\tp_key_violations........0x%X\n"
             "\t\t\t\tq_key_violations........0x%X\n"
             "\t\t\t\tguid_cap................0x%X\n"
             "\t\t\t\tclient_reregister.......0x%X\n"
             "\t\t\t\tsubnet_timeout..........0x%X\n"
             "\t\t\t\tresp_time_value.........0x%X\n"
             "\t\t\t\terror_threshold.........0x%X\n"
             "",
             port_num,
             cl_ntoh64( node_guid ),
             cl_ntoh64( port_guid ),
             cl_ntoh64( p_pi->m_key ),
             cl_ntoh64( p_pi->subnet_prefix ),
             cl_ntoh16( p_pi->base_lid ),
             cl_ntoh16( p_pi->master_sm_base_lid ),
             cl_ntoh32( p_pi->capability_mask ),
             cl_ntoh16( p_pi->diag_code ),
             cl_ntoh16( p_pi->m_key_lease_period ),
             p_pi->local_port_num,
             p_pi->link_width_enabled,
             p_pi->link_width_supported,
             p_pi->link_width_active,
             ib_port_info_get_link_speed_sup( p_pi ),
             ib_get_port_state_str( ib_port_info_get_port_state( p_pi ) ),
             p_pi->state_info2,
             ib_port_info_get_mpb( p_pi ),
             ib_port_info_get_lmc( p_pi ),
             p_pi->link_speed,
             p_pi->mtu_smsl,
             p_pi->vl_cap,
             p_pi->vl_high_limit,
             p_pi->vl_arb_high_cap,
             p_pi->vl_arb_low_cap,
             p_pi->mtu_cap,
             p_pi->vl_stall_life,
             p_pi->vl_enforce,
             cl_ntoh16( p_pi->m_key_violations ),
             cl_ntoh16( p_pi->p_key_violations ),
             cl_ntoh16( p_pi->q_key_violations ),
             p_pi->guid_cap,
             ib_port_info_get_client_rereg( p_pi ),
             ib_port_info_get_timeout( p_pi ),
             p_pi->resp_time_value,
             p_pi->error_threshold
             );

    /*  show the capabilities mask */
    osm_dbg_get_capabilities_str( buf, BUF_SIZE, "\t\t\t\t", p_pi );

    osm_log( p_log, log_level, "%s", buf );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_portinfo_record(
  IN osm_log_t* const p_log,
  IN const ib_portinfo_record_t* const p_pir,
  IN const osm_log_level_t log_level )
{
  char buf[BUF_SIZE];
  const ib_port_info_t * const p_pi = &p_pir->port_info;

  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, log_level,
             "PortInfo Record dump:\n"
             "\t\t\t\tRID\n"
             "\t\t\t\tEndPortLid..............0x%X\n"
             "\t\t\t\tPortNum.................0x%X\n"
             "\t\t\t\tReserved................0x%X\n"
             "\t\t\t\tPortInfo dump\n"
             "\t\t\t\tm_key...................0x%016" PRIx64 "\n"
             "\t\t\t\tsubnet_prefix...........0x%016" PRIx64 "\n"
             "\t\t\t\tbase_lid................0x%X\n"
             "\t\t\t\tmaster_sm_base_lid......0x%X\n"
             "\t\t\t\tcapability_mask.........0x%X\n"
             "\t\t\t\tdiag_code...............0x%X\n"
             "\t\t\t\tm_key_lease_period......0x%X\n"
             "\t\t\t\tlocal_port_num..........0x%X\n"
             "\t\t\t\tlink_width_enabled......0x%X\n"
             "\t\t\t\tlink_width_supported....0x%X\n"
             "\t\t\t\tlink_width_active.......0x%X\n"
             "\t\t\t\tlink_speed_supported....0x%X\n"
             "\t\t\t\tport_state..............%s\n"
             "\t\t\t\tstate_info2.............0x%X\n"
             "\t\t\t\tm_key_protect_bits......0x%X\n"
             "\t\t\t\tlmc.....................0x%X\n"
             "\t\t\t\tlink_speed..............0x%X\n"
             "\t\t\t\tmtu_smsl................0x%X\n"
             "\t\t\t\tvl_cap_init_type........0x%X\n"
             "\t\t\t\tvl_high_limit...........0x%X\n"
             "\t\t\t\tvl_arb_high_cap.........0x%X\n"
             "\t\t\t\tvl_arb_low_cap..........0x%X\n"
             "\t\t\t\tinit_rep_mtu_cap........0x%X\n"
             "\t\t\t\tvl_stall_life...........0x%X\n"
             "\t\t\t\tvl_enforce..............0x%X\n"
             "\t\t\t\tm_key_violations........0x%X\n"
             "\t\t\t\tp_key_violations........0x%X\n"
             "\t\t\t\tq_key_violations........0x%X\n"
             "\t\t\t\tguid_cap................0x%X\n"
             "\t\t\t\tsubnet_timeout..........0x%X\n"
             "\t\t\t\tresp_time_value.........0x%X\n"
             "\t\t\t\terror_threshold.........0x%X\n"
             "",
             cl_ntoh16(p_pir->lid),
             p_pir->port_num,
             p_pir->resv,
             cl_ntoh64( p_pi->m_key ),
             cl_ntoh64( p_pi->subnet_prefix ),
             cl_ntoh16( p_pi->base_lid ),
             cl_ntoh16( p_pi->master_sm_base_lid ),
             cl_ntoh32( p_pi->capability_mask ),
             cl_ntoh16( p_pi->diag_code ),
             cl_ntoh16( p_pi->m_key_lease_period ),
             p_pi->local_port_num,
             p_pi->link_width_enabled,
             p_pi->link_width_supported,
             p_pi->link_width_active,
             ib_port_info_get_link_speed_sup( p_pi ),
             ib_get_port_state_str( ib_port_info_get_port_state( p_pi ) ),
             p_pi->state_info2,
             ib_port_info_get_mpb( p_pi ),
             ib_port_info_get_lmc( p_pi ),
             p_pi->link_speed,
             p_pi->mtu_smsl,
             p_pi->vl_cap,
             p_pi->vl_high_limit,
             p_pi->vl_arb_high_cap,
             p_pi->vl_arb_low_cap,
             p_pi->mtu_cap,
             p_pi->vl_stall_life,
             p_pi->vl_enforce,
             cl_ntoh16( p_pi->m_key_violations ),
             cl_ntoh16( p_pi->p_key_violations ),
             cl_ntoh16( p_pi->q_key_violations ),
             p_pi->guid_cap,
             ib_port_info_get_timeout( p_pi ),
             p_pi->resp_time_value,
             p_pi->error_threshold
             );

    /*  show the capabilities mask */
    osm_dbg_get_capabilities_str( buf, BUF_SIZE, "\t\t\t\t", p_pi );

    osm_log( p_log, log_level, "%s", buf );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_guidinfo_record(
  IN osm_log_t* const p_log,
  IN const ib_guidinfo_record_t* const p_gir,
  IN const osm_log_level_t log_level )
{
  const ib_guid_info_t * const p_gi = &p_gir->guid_info;

  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, log_level,
             "GUIDInfo Record dump:\n"
             "\t\t\t\tRID\n"
             "\t\t\t\tLid.....................0x%X\n"
             "\t\t\t\tBlockNum................0x%X\n"
             "\t\t\t\tReserved................0x%X\n"
             "\t\t\t\tGUIDInfo dump\n"
             "\t\t\t\tReserved................0x%X\n"
             "\t\t\t\tGUID 0..................0x%016" PRIx64 "\n"
             "\t\t\t\tGUID 1..................0x%016" PRIx64 "\n"
             "\t\t\t\tGUID 2..................0x%016" PRIx64 "\n"
             "\t\t\t\tGUID 3..................0x%016" PRIx64 "\n"
             "\t\t\t\tGUID 4..................0x%016" PRIx64 "\n"
             "\t\t\t\tGUID 5..................0x%016" PRIx64 "\n"
             "\t\t\t\tGUID 6..................0x%016" PRIx64 "\n"
             "\t\t\t\tGUID 7..................0x%016" PRIx64 "\n",
             cl_ntoh16(p_gir->lid),
             p_gir->block_num,
             p_gir->resv,
             cl_ntoh32(p_gir->reserved),
             cl_ntoh64(p_gi->guid[0]),
             cl_ntoh64(p_gi->guid[1]),
             cl_ntoh64(p_gi->guid[2]),
             cl_ntoh64(p_gi->guid[3]),
             cl_ntoh64(p_gi->guid[4]),
             cl_ntoh64(p_gi->guid[5]),
             cl_ntoh64(p_gi->guid[6]),
             cl_ntoh64(p_gi->guid[7])
           );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_node_info(
  IN osm_log_t* const p_log,
  IN const ib_node_info_t* const p_ni,
  IN const osm_log_level_t log_level )
{
  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, log_level,
             "NodeInfo dump:\n"
             "\t\t\t\tbase_version............0x%X\n"
             "\t\t\t\tclass_version...........0x%X\n"
             "\t\t\t\tnode_type...............%s\n"
             "\t\t\t\tnum_ports...............0x%X\n"
             "\t\t\t\tsys_guid................0x%016" PRIx64 "\n"
             "\t\t\t\tnode_guid...............0x%016" PRIx64 "\n"
             "\t\t\t\tport_guid...............0x%016" PRIx64 "\n"
             "\t\t\t\tpartition_cap...........0x%X\n"
             "\t\t\t\tdevice_id...............0x%X\n"
             "\t\t\t\trevision................0x%X\n"
             "\t\t\t\tport_num................0x%X\n"
             "\t\t\t\tvendor_id...............0x%X\n"
             "",
             p_ni->base_version,
             p_ni->class_version,
             ib_get_node_type_str( p_ni->node_type ),
             p_ni->num_ports,
             cl_ntoh64( p_ni->sys_guid ),
             cl_ntoh64( p_ni->node_guid ),
             cl_ntoh64( p_ni->port_guid ),
             cl_ntoh16( p_ni->partition_cap ),
             cl_ntoh16( p_ni->device_id ),
             cl_ntoh32( p_ni->revision ),
             ib_node_info_get_local_port_num( p_ni ),
             cl_ntoh32( ib_node_info_get_vendor_id( p_ni ) )
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_node_record(
  IN osm_log_t* const p_log,
  IN const ib_node_record_t* const p_nr,
  IN const osm_log_level_t log_level )
{
  const ib_node_info_t * const p_ni = &p_nr->node_info;

  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, log_level,
             "NodeRecord dump:\n"
             "\t\t\t\tRID\n"
             "\t\t\t\tLid.....................0x%X\n"
             "\t\t\t\tReserved................0x%X\n"
             "\t\t\t\tNodeInfoDump\n"
             "\t\t\t\tbase_version............0x%X\n"
             "\t\t\t\tclass_version...........0x%X\n"
             "\t\t\t\tnode_type...............%s\n"
             "\t\t\t\tnum_ports...............0x%X\n"
             "\t\t\t\tsys_guid................0x%016" PRIx64 "\n"
             "\t\t\t\tnode_guid...............0x%016" PRIx64 "\n"
             "\t\t\t\tport_guid...............0x%016" PRIx64 "\n"
             "\t\t\t\tpartition_cap...........0x%X\n"
             "\t\t\t\tdevice_id...............0x%X\n"
             "\t\t\t\trevision................0x%X\n"
             "\t\t\t\tport_num................0x%X\n"
             "\t\t\t\tvendor_id...............0x%X\n"
             "\t\t\t\tNodeDescription\n"
             "\t\t\t\t%s\n"
             "",
             cl_ntoh16(p_nr->lid),
             cl_ntoh16(p_nr->resv),
             p_ni->base_version,
             p_ni->class_version,
             ib_get_node_type_str( p_ni->node_type ),
             p_ni->num_ports,
             cl_ntoh64( p_ni->sys_guid ),
             cl_ntoh64( p_ni->node_guid ),
             cl_ntoh64( p_ni->port_guid ),
             cl_ntoh16( p_ni->partition_cap ),
             cl_ntoh16( p_ni->device_id ),
             cl_ntoh32( p_ni->revision ),
             ib_node_info_get_local_port_num( p_ni ),
             cl_ntoh32( ib_node_info_get_vendor_id( p_ni )),
             p_nr->node_desc.description
             );

  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_path_record(
  IN osm_log_t* const p_log,
  IN const ib_path_rec_t* const p_pr,
  IN const osm_log_level_t log_level )
{
  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, log_level,
             "PathRecord dump:\n"
             "\t\t\t\tresv0...................0x%016" PRIx64 "\n"
             "\t\t\t\tdgid....................0x%016" PRIx64 " : "
             "0x%016" PRIx64 "\n"
             "\t\t\t\tsgid....................0x%016" PRIx64 " : "
             "0x%016" PRIx64 "\n"
             "\t\t\t\tdlid....................0x%X\n"
             "\t\t\t\tslid....................0x%X\n"
             "\t\t\t\thop_flow_raw............0x%X\n"
             "\t\t\t\ttclass..................0x%X\n"
             "\t\t\t\tnum_path_revers.........0x%X\n"
             "\t\t\t\tpkey....................0x%X\n"
             "\t\t\t\tsl......................0x%X\n"
             "\t\t\t\tmtu.....................0x%X\n"
             "\t\t\t\trate....................0x%X\n"
             "\t\t\t\tpkt_life................0x%X\n"
             "\t\t\t\tpreference..............0x%X\n"
             "\t\t\t\tresv2...................0x%X\n"
             "\t\t\t\tresv3...................0x%X\n"
             "",
             *(uint64_t*)p_pr->resv0,
             cl_ntoh64( p_pr->dgid.unicast.prefix ),
             cl_ntoh64( p_pr->dgid.unicast.interface_id ),
             cl_ntoh64( p_pr->sgid.unicast.prefix ),
             cl_ntoh64( p_pr->sgid.unicast.interface_id ),
             cl_ntoh16( p_pr->dlid ),
             cl_ntoh16( p_pr->slid ),
             cl_ntoh32( p_pr->hop_flow_raw ),
             p_pr->tclass,
             p_pr->num_path,
             cl_ntoh16( p_pr->pkey ),
             cl_ntoh16( p_pr->sl ),
             p_pr->mtu,
             p_pr->rate,
             p_pr->pkt_life,
             p_pr->preference,
             *(uint32_t*)&p_pr->resv2,
             *((uint16_t*)&p_pr->resv2 + 2)
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_mc_record(
  IN osm_log_t* const p_log,
  IN const ib_member_rec_t* const p_mcmr,
  IN const osm_log_level_t log_level )
{

  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, log_level,
             "MCMember Record dump:\n"
             "\t\t\t\tMGID....................0x%016" PRIx64 " : "
             "0x%016" PRIx64 "\n"
             "\t\t\t\tPortGid.................0x%016" PRIx64 " : "
             "0x%016" PRIx64 "\n"
             "\t\t\t\tqkey....................0x%X\n"
             "\t\t\t\tMlid....................0x%X\n"
             "\t\t\t\tMtu.....................0x%X\n"
             "\t\t\t\tTClass..................0x%X\n"
             "\t\t\t\tpkey....................0x%X\n"
             "\t\t\t\tRate....................0x%X\n"
             "\t\t\t\tPacketLife..............0x%X\n"
             "\t\t\t\tSLFlowLabelHopLimit.....0x%X\n"
             "\t\t\t\tScopeState..............0x%X\n"
             "\t\t\t\tProxyJoin...............0x%X\n"
             "",
             cl_ntoh64( p_mcmr->mgid.unicast.prefix ),
             cl_ntoh64( p_mcmr->mgid.unicast.interface_id ),
             cl_ntoh64( p_mcmr->port_gid.unicast.prefix ),
             cl_ntoh64( p_mcmr->port_gid.unicast.interface_id ),
             cl_ntoh32( p_mcmr->qkey ),
             cl_ntoh16( p_mcmr->mlid ),
             p_mcmr->mtu,
             p_mcmr->tclass,
             cl_ntoh16( p_mcmr->pkey ),
             p_mcmr->rate,
             p_mcmr->pkt_life,
             cl_ntoh32( p_mcmr->sl_flow_hop ),
             p_mcmr->scope_state,
             p_mcmr->proxy_join
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_service_record(
  IN osm_log_t* const p_log,
  IN const ib_service_record_t* const p_sr,
  IN const osm_log_level_t log_level )
{
  char buf_service_key[33];
  char buf_service_name[65];

  if( osm_log_is_active( p_log, log_level ) )
  {
    sprintf(buf_service_key,
            "0x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x",
            p_sr->service_key[0],
            p_sr->service_key[1],
            p_sr->service_key[2],
            p_sr->service_key[3],
            p_sr->service_key[4],
            p_sr->service_key[5],
            p_sr->service_key[6],
            p_sr->service_key[7],
            p_sr->service_key[8],
            p_sr->service_key[9],
            p_sr->service_key[10],
            p_sr->service_key[11],
            p_sr->service_key[12],
            p_sr->service_key[13],
            p_sr->service_key[14],
            p_sr->service_key[15]);
    strncpy(buf_service_name, (char *)p_sr->service_name, 64);
    buf_service_name[64] = '\0';

    osm_log( p_log, log_level,
             "ServiceRecord dump:\n"
             "\t\t\t\tServiceID...............0x%016" PRIx64 "\n"
             "\t\t\t\tServiceGID..............0x%016" PRIx64 " : "
             "0x%016" PRIx64 "\n"
             "\t\t\t\tServiceP_Key............0x%X\n"
             "\t\t\t\tServiceLease............0x%X\n"
             "\t\t\t\tServiceKey..............%s\n"
             "\t\t\t\tServiceName.............%s\n"
             "\t\t\t\tServiceData8.1..........0x%X\n"
             "\t\t\t\tServiceData8.2..........0x%X\n"
             "\t\t\t\tServiceData8.3..........0x%X\n"
             "\t\t\t\tServiceData8.4..........0x%X\n"
             "\t\t\t\tServiceData8.5..........0x%X\n"
             "\t\t\t\tServiceData8.6..........0x%X\n"
             "\t\t\t\tServiceData8.7..........0x%X\n"
             "\t\t\t\tServiceData8.8..........0x%X\n"
             "\t\t\t\tServiceData8.9..........0x%X\n"
             "\t\t\t\tServiceData8.10.........0x%X\n"
             "\t\t\t\tServiceData8.11.........0x%X\n"
             "\t\t\t\tServiceData8.12.........0x%X\n"
             "\t\t\t\tServiceData8.13.........0x%X\n"
             "\t\t\t\tServiceData8.14.........0x%X\n"
             "\t\t\t\tServiceData8.15.........0x%X\n"
             "\t\t\t\tServiceData8.16.........0x%X\n"
             "\t\t\t\tServiceData16.1.........0x%X\n"
             "\t\t\t\tServiceData16.2.........0x%X\n"
             "\t\t\t\tServiceData16.3.........0x%X\n"
             "\t\t\t\tServiceData16.4.........0x%X\n"
             "\t\t\t\tServiceData16.5.........0x%X\n"
             "\t\t\t\tServiceData16.6.........0x%X\n"
             "\t\t\t\tServiceData16.7.........0x%X\n"
             "\t\t\t\tServiceData16.8.........0x%X\n"
             "\t\t\t\tServiceData32.1.........0x%X\n"
             "\t\t\t\tServiceData32.2.........0x%X\n"
             "\t\t\t\tServiceData32.3.........0x%X\n"
             "\t\t\t\tServiceData32.4.........0x%X\n"
             "\t\t\t\tServiceData64.1.........0x%016" PRIx64 "\n"
             "\t\t\t\tServiceData64.2.........0x%016" PRIx64 "\n"
             "",
             cl_ntoh64( p_sr->service_id ),
             cl_ntoh64( p_sr->service_gid.unicast.prefix ),
             cl_ntoh64( p_sr->service_gid.unicast.interface_id ),
             cl_ntoh16( p_sr->service_pkey ),
             cl_ntoh32( p_sr->service_lease ),
             buf_service_key,
             buf_service_name,
             p_sr->service_data8[0], p_sr->service_data8[1],
             p_sr->service_data8[2], p_sr->service_data8[3],
             p_sr->service_data8[4], p_sr->service_data8[5],
             p_sr->service_data8[6], p_sr->service_data8[7],
             p_sr->service_data8[8], p_sr->service_data8[9],
             p_sr->service_data8[10], p_sr->service_data8[11],
             p_sr->service_data8[12], p_sr->service_data8[13],
             p_sr->service_data8[14], p_sr->service_data8[15],
             cl_ntoh16(p_sr->service_data16[0]),
             cl_ntoh16(p_sr->service_data16[1]),
             cl_ntoh16(p_sr->service_data16[2]),
             cl_ntoh16(p_sr->service_data16[3]),
             cl_ntoh16(p_sr->service_data16[4]),
             cl_ntoh16(p_sr->service_data16[5]),
             cl_ntoh16(p_sr->service_data16[6]),
             cl_ntoh16(p_sr->service_data16[7]),
             cl_ntoh32(p_sr->service_data32[0]),
             cl_ntoh32(p_sr->service_data32[1]),
             cl_ntoh32(p_sr->service_data32[2]),
             cl_ntoh32(p_sr->service_data32[3]),
             cl_ntoh64(p_sr->service_data64[0]),
             cl_ntoh64(p_sr->service_data64[1])
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_inform_info(
  IN osm_log_t* const p_log,
  IN const ib_inform_info_t* const p_ii,
  IN const osm_log_level_t log_level )
{
  uint32_t qpn;
  uint8_t  resp_time_val;

  ib_inform_info_get_qpn_resp_time(p_ii->g_or_v.generic.qpn_resp_time_val,
                                   &qpn, &resp_time_val);

  if( osm_log_is_active( p_log, log_level ) )
  {
    if (p_ii->is_generic)
    {
      osm_log( p_log, log_level,
               "InformInfo dump:\n"
               "\t\t\t\tgid.....................0x%016" PRIx64 ",%016" PRIx64 "\n"
               "\t\t\t\tlid_range_begin.........0x%X\n"
               "\t\t\t\tlid_range_end...........0x%X\n"
               "\t\t\t\tis_generic..............0x%X\n"
               "\t\t\t\tsubscribe...............0x%X\n"
               "\t\t\t\ttrap_type...............0x%X\n"
               "\t\t\t\ttrap_num................%u\n"
               "\t\t\t\tqpn.....................0x%06X\n"
               "\t\t\t\tresp_time_val...........0x%X\n"
               "\t\t\t\tnode_type...............0x%06X\n"
               "",
               cl_ntoh64( p_ii->gid.unicast.prefix ),
               cl_ntoh64( p_ii->gid.unicast.interface_id ),
               cl_ntoh16( p_ii->lid_range_begin ),
               cl_ntoh16( p_ii->lid_range_end ),
               p_ii->is_generic,
               p_ii->subscribe,
               cl_ntoh16( p_ii->trap_type ),
               cl_ntoh16( p_ii->g_or_v.generic.trap_num ),
               cl_ntoh32(qpn),
               resp_time_val,
               cl_ntoh32(ib_inform_info_get_node_type( p_ii ))
               );
    }
    else
    {
      osm_log( p_log, log_level,
               "InformInfo dump:\n"
               "\t\t\t\tgid.....................0x%016" PRIx64 ",%016" PRIx64 "\n"
               "\t\t\t\tlid_range_begin.........0x%X\n"
               "\t\t\t\tlid_range_end...........0x%X\n"
               "\t\t\t\tis_generic..............0x%X\n"
               "\t\t\t\tsubscribe...............0x%X\n"
               "\t\t\t\ttrap_type...............0x%X\n"
               "\t\t\t\tdev_id..................0x%X\n"
               "\t\t\t\tqpn.....................0x%06X\n"
               "\t\t\t\tresp_time_val...........0x%X\n"
               "\t\t\t\tvendor_id...............0x%06X\n"
               "",
               cl_ntoh64( p_ii->gid.unicast.prefix ),
               cl_ntoh64( p_ii->gid.unicast.interface_id ),
               cl_ntoh16( p_ii->lid_range_begin ),
               cl_ntoh16( p_ii->lid_range_end ),
               p_ii->is_generic,
               p_ii->subscribe,
               cl_ntoh16( p_ii->trap_type ),
               cl_ntoh16( p_ii->g_or_v.vend.dev_id ),
               cl_ntoh32(qpn),
               resp_time_val,
               cl_ntoh32(ib_inform_info_get_node_type( p_ii ))
               );
    }
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_link_record(
  IN osm_log_t*            const p_log,
  IN const ib_link_record_t*  const p_lr,
  IN const osm_log_level_t log_level )
{
  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, log_level,
             "LinkRecord dump:\n"
             "\t\t\t\tfrom_lid................0x%X\n"
             "\t\t\t\tfrom_port_num...........0x%X\n"
             "\t\t\t\tto_port_num.............0x%X\n"
             "\t\t\t\tto_lid..................0x%X\n"
             "",
             cl_ntoh16( p_lr->from_lid ),
             p_lr->from_port_num,
             p_lr->to_port_num,
             cl_ntoh16( p_lr->to_lid )
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_switch_info(
  IN osm_log_t* const p_log,
  IN const ib_switch_info_t* const p_si,
  IN const osm_log_level_t log_level )
{
  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, OSM_LOG_VERBOSE,
             "SwitchInfo dump:\n"
             "\t\t\t\tlin_cap.................0x%X\n"
             "\t\t\t\trand_cap................0x%X\n"
             "\t\t\t\tmcast_cap...............0x%X\n"
             "\t\t\t\tlin_top.................0x%X\n"
             "\t\t\t\tdef_port................0x%X\n"
             "\t\t\t\tdef_mcast_pri_port......0x%X\n"
             "\t\t\t\tdef_mcast_not_port......0x%X\n"
             "\t\t\t\tlife_state..............0x%X\n"
             "\t\t\t\tlids_per_port...........0x%X\n"
             "\t\t\t\tpartition_enf_cap.......0x%X\n"
             "\t\t\t\tflags...................0x%X\n"
             "",
             cl_ntoh16( p_si->lin_cap ),
             cl_ntoh16( p_si->rand_cap ),
             cl_ntoh16( p_si->mcast_cap ),
             cl_ntoh16( p_si->lin_top ),
             p_si->def_port,
             p_si->def_mcast_pri_port,
             p_si->def_mcast_not_port,
             p_si->life_state,
             cl_ntoh16( p_si->lids_per_port ),
             cl_ntoh16( p_si->enforce_cap ),
             p_si->flags
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_pkey_block(
  IN osm_log_t* const p_log,
  IN uint64_t port_guid,
  IN uint16_t block_num,
  IN uint8_t port_num,
  IN const ib_pkey_table_t* const p_pkey_tbl,
  IN const osm_log_level_t log_level )
{
  int i;
  char buf_line[1024];

  if( osm_log_is_active( p_log, log_level ) )
  {
    buf_line[0] = '\0';
    for (i = 0; i < 32; i++)
      sprintf( buf_line,"%s 0x%04x |",
               buf_line, cl_ntoh16(p_pkey_tbl->pkey_entry[i]));

    osm_log( p_log, log_level,
             "P_Key table dump:\n"
             "\t\t\tport_guid...........0x%016" PRIx64 "\n"
             "\t\t\tblock_num...........0x%X\n"
             "\t\t\tport_num............0x%X\n\tP_Key Table: %s\n",
             cl_ntoh64( port_guid ),
             block_num,
             port_num,
             buf_line
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_slvl_map_table(
  IN osm_log_t* const p_log,
  IN uint64_t port_guid,
  IN uint8_t in_port_num,
  IN uint8_t out_port_num,
  IN const ib_slvl_table_t* const p_slvl_tbl,
  IN const osm_log_level_t log_level )
{
  uint8_t i;
  char buf_line1[1024];
  char buf_line2[1024];

  if( osm_log_is_active( p_log, log_level ) )
  {
    buf_line1[0] = '\0';
    buf_line2[0] = '\0';
    for (i = 0; i < 16; i++)
      sprintf( buf_line1,"%s %-2u |", buf_line1, i);
    for (i = 0; i < 16; i++)
      sprintf( buf_line2,"%s0x%01X |",
               buf_line2, ib_slvl_table_get(p_slvl_tbl, i));
    osm_log( p_log, log_level,
             "SLtoVL dump:\n"
             "\t\t\tport_guid............0x%016" PRIx64 "\n"
             "\t\t\tin_port_num..........0x%X\n"
             "\t\t\tout_port_num.........0x%X\n\tSL: | %s\n\tVL: | %s\n",
             cl_ntoh64( port_guid ),
             in_port_num,
             out_port_num,
             buf_line1, buf_line2
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_vl_arb_table(
  IN osm_log_t* const p_log,
  IN uint64_t port_guid,
  IN uint8_t block_num,
  IN uint8_t port_num,
  IN const ib_vl_arb_table_t* const p_vla_tbl,
  IN const osm_log_level_t log_level )
{
  int i;
  char buf_line1[1024];
  char buf_line2[1024];

  if( osm_log_is_active( p_log, log_level ) )
  {
    buf_line1[0] = '\0';
    buf_line2[0] = '\0';
    for (i = 0; i < 32; i++)
      sprintf( buf_line1,"%s 0x%01X |",
               buf_line1, p_vla_tbl->vl_entry[i].vl);
    for (i = 0; i < 32; i++)
      sprintf( buf_line2,"%s 0x%01X |",
               buf_line2, p_vla_tbl->vl_entry[i].weight);
    osm_log( p_log, log_level,
             "VlArb dump:\n"
             "\t\t\tport_guid...........0x%016" PRIx64 "\n"
             "\t\t\tblock_num...........0x%X\n"
             "\t\t\tport_num............0x%X\n\tVL    : | %s\n\tWEIGHT:| %s\n",
             cl_ntoh64( port_guid ),
             block_num,
             port_num,
             buf_line1, buf_line2
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_sm_info(
  IN osm_log_t* const p_log,
  IN const ib_sm_info_t* const p_smi,
  IN const osm_log_level_t log_level )
{
  if( osm_log_is_active( p_log, log_level ) )
  {
    osm_log( p_log, OSM_LOG_DEBUG,
             "SMInfo dump:\n"
             "\t\t\t\tguid....................0x%016" PRIx64 "\n"
             "\t\t\t\tsm_key..................0x%016" PRIx64 "\n"
             "\t\t\t\tact_count...............%u\n"
             "\t\t\t\tpriority................%u\n"
             "\t\t\t\tsm_state................%u\n"
             "",
             cl_ntoh64( p_smi->guid ),
             cl_ntoh64( p_smi->sm_key ),
             cl_ntoh32( p_smi->act_count ),
             ib_sminfo_get_priority( p_smi ),
             ib_sminfo_get_state( p_smi )
             );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_notice(
  IN osm_log_t* const p_log,
  IN const ib_mad_notice_attr_t *p_ntci,
  IN const osm_log_level_t log_level )
{
  char buff[1024];
  buff[0] = '\0';

  if( osm_log_is_active( p_log, log_level ) )
  {
    if (ib_notice_is_generic(p_ntci))
    {
      /* immediate data based on the trap */
      switch (cl_ntoh16(p_ntci->g_or_v.generic.trap_num)) {
      case 64:
      case 65:
      case 66:
      case 67:
        sprintf(buff,
                "\t\t\t\tsrc_gid..................0x%016" PRIx64
                ":0x%016" PRIx64 "\n",
                cl_ntoh64(p_ntci->data_details.ntc_64_67.gid.unicast.prefix),
                cl_ntoh64(p_ntci->data_details.ntc_64_67.gid.unicast.interface_id));
        break;
      case 128:
        sprintf(buff,
                "\t\t\t\tsw_lid...................0x%04X\n",
                cl_ntoh16(p_ntci->data_details.ntc_128.sw_lid));
        break;
      case 129:
      case 130:
      case 131:
        sprintf(buff,
                "\t\t\t\tlid......................0x%04X\n"
                "\t\t\t\tport_num.................%u\n",
                cl_ntoh16(p_ntci->data_details.ntc_129_131.lid),
                p_ntci->data_details.ntc_129_131.port_num);
        break;
      case 144:
        sprintf(buff,
                "\t\t\t\tlid......................0x%04x\n"
                "\t\t\t\tnew_cap_mask.............0x%08x\n",
                cl_ntoh16(p_ntci->data_details.ntc_144.lid),
                cl_ntoh32(p_ntci->data_details.ntc_144.new_cap_mask));
        break;
      case 145:
        sprintf(buff,
                "\t\t\t\tlid......................0x%04X\n"
                "\t\t\t\tnew_sys_guid.............0x%016" PRIx64 "\n",
                cl_ntoh16(p_ntci->data_details.ntc_145.lid),
                cl_ntoh64(p_ntci->data_details.ntc_145.new_sys_guid));
        break;
      }

      osm_log( p_log, log_level,
               "Generic Notice dump:\n"
               "\t\t\t\ttype.....................0x%02X\n"
               "\t\t\t\tprod_type................%u\n"
               "\t\t\t\ttrap_num.................%u\n%s"
               "",
               ib_notice_get_type(p_ntci),
               cl_ntoh32(ib_notice_get_prod_type(p_ntci)),
               cl_ntoh16(p_ntci->g_or_v.generic.trap_num),
               buff
               );
    }
    else
    {
      osm_log( p_log, log_level,
               "Vendor Notice dump:\n"
               "\t\t\t\ttype.....................0x%04x\n"
               "\t\t\t\tvendor...................%u\n"
               "\t\t\t\tdevice_id................%u\n"
               "",
               cl_ntoh16(ib_notice_get_type(p_ntci)),
               cl_ntoh32(ib_notice_get_vend_id(p_ntci)),
               cl_ntoh16(p_ntci->g_or_v.vend.dev_id)
               );
    }
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_dr_smp(
  IN osm_log_t* const p_log,
  IN const ib_smp_t* const p_smp,
  IN const osm_log_level_t log_level )
{
  uint32_t i;
  char buf[BUF_SIZE];
  char line[BUF_SIZE];

  if( osm_log_is_active( p_log, log_level ) )
  {
    sprintf( buf,
             "SMP dump:\n"
             "\t\t\t\tbase_ver................0x%X\n"
             "\t\t\t\tmgmt_class..............0x%X\n"
             "\t\t\t\tclass_ver...............0x%X\n"
             "\t\t\t\tmethod..................0x%X (%s)\n",
             p_smp->base_ver,
             p_smp->mgmt_class,
             p_smp->class_ver,
             p_smp->method, ib_get_sm_method_str(p_smp->method));

    if (p_smp->mgmt_class == IB_MCLASS_SUBN_DIR)
    {
      sprintf( line,
               "\t\t\t\tD bit...................0x%X\n"
               "\t\t\t\tstatus..................0x%X\n",
               ib_smp_is_d(p_smp),
               ib_smp_get_status(p_smp));
    }
    else
    {
      sprintf( line,
               "\t\t\t\tstatus..................0x%X\n",
               cl_ntoh16(p_smp->status));
    }
    strcat( buf, line );

    sprintf( line,
             "\t\t\t\thop_ptr.................0x%X\n"
             "\t\t\t\thop_count...............0x%X\n"
             "\t\t\t\ttrans_id................0x%" PRIx64 "\n"
             "\t\t\t\tattr_id.................0x%X (%s)\n"
             "\t\t\t\tresv....................0x%X\n"
             "\t\t\t\tattr_mod................0x%X\n"
             "\t\t\t\tm_key...................0x%016" PRIx64 "\n",
             p_smp->hop_ptr,
             p_smp->hop_count,
             cl_ntoh64(p_smp->trans_id),
             cl_ntoh16(p_smp->attr_id),
             ib_get_sm_attr_str( p_smp->attr_id ),
             cl_ntoh16(p_smp->resv),
             cl_ntoh32(p_smp->attr_mod),
             cl_ntoh64(p_smp->m_key)
             );
    strcat( buf, line );

    if (p_smp->mgmt_class == IB_MCLASS_SUBN_DIR)
    {
      sprintf( line,
               "\t\t\t\tdr_slid.................0x%X\n"
               "\t\t\t\tdr_dlid.................0x%X\n",
               cl_ntoh16(p_smp->dr_slid),
               cl_ntoh16(p_smp->dr_dlid)
               );
      strcat( buf, line );

      strcat( buf, "\n\t\t\t\tInitial path: " );

      for( i = 0; i <= p_smp->hop_count; i++ )
      {
        sprintf( line, "[%X]", p_smp->initial_path[i] );
        strcat( buf, line );
      }

      strcat( buf, "\n\t\t\t\tReturn path:  " );

      for( i = 0; i <= p_smp->hop_count; i++ )
      {
        sprintf( line, "[%X]", p_smp->return_path[i] );
        strcat( buf, line );
      }

      strcat( buf, "\n\t\t\t\tReserved:     " );

      for( i = 0; i < 7; i++ )
      {
        sprintf( line, "[%0X]", p_smp->resv1[i] );
        strcat( buf, line );
      }

      strcat( buf, "\n" );

      for( i = 0; i < 64; i += 16 )
      {
        sprintf( line, "\n\t\t\t\t%02X %02X %02X %02X "
                 "%02X %02X %02X %02X"
                 "   %02X %02X %02X %02X %02X %02X %02X %02X\n",
                 p_smp->data[i],
                 p_smp->data[i+1],
                 p_smp->data[i+2],
                 p_smp->data[i+3],
                 p_smp->data[i+4],
                 p_smp->data[i+5],
                 p_smp->data[i+6],
                 p_smp->data[i+7],
                 p_smp->data[i+8],
                 p_smp->data[i+9],
                 p_smp->data[i+10],
                 p_smp->data[i+11],
                 p_smp->data[i+12],
                 p_smp->data[i+13],
                 p_smp->data[i+14],
                 p_smp->data[i+15] );

        strcat( buf, line );
      }
    }
    else
    {
      /* not a Direct Route so provide source and destination lids */
      strcat(buf, "\t\t\t\tMAD IS LID ROUTED\n");
    }

    osm_log( p_log, log_level,
             "%s\n", buf );

  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_sa_mad(
  IN osm_log_t* const p_log,
  IN const ib_sa_mad_t* const p_mad,
  IN const osm_log_level_t log_level )
{
  char buf[BUF_SIZE];

  /* make sure the mad is valid */
  if (p_mad == NULL)
  {
    osm_log( p_log, log_level,
             "NULL MAD POINTER\n");
    return;
  }

  if( osm_log_is_active( p_log, log_level ) )
  {
    sprintf( buf,
             "SA MAD dump:\n"
             "\t\t\t\tbase_ver................0x%X\n"
             "\t\t\t\tmgmt_class..............0x%X\n"
             "\t\t\t\tclass_ver...............0x%X\n"
             "\t\t\t\tmethod..................0x%X (%s)\n"
             "\t\t\t\tstatus..................0x%X\n"
             "\t\t\t\tresv....................0x%X\n"
             "\t\t\t\ttrans_id................0x%" PRIx64 "\n"
             "\t\t\t\tattr_id.................0x%X (%s)\n"
             "\t\t\t\tresv1...................0x%X\n"
             "\t\t\t\tattr_mod................0x%X\n"
             "\t\t\t\trmpp_version............0x%X\n"
             "\t\t\t\trmpp_type...............0x%X\n"
             "\t\t\t\trmpp_flags..............0x%X\n"
             "\t\t\t\trmpp_status.............0x%X\n"
             "\t\t\t\tseg_num.................0x%X\n"
             "\t\t\t\tpayload_len/new_win.....0x%X\n"
             "\t\t\t\tsm_key..................0x%016" PRIx64 "\n"
             "\t\t\t\tattr_offset.............0x%X\n"
             "\t\t\t\tresv2...................0x%X\n"
             "\t\t\t\tcomp_mask...............0x%016" PRIx64 "\n",
             p_mad->base_ver,
             p_mad->mgmt_class,
             p_mad->class_ver,
             p_mad->method, ib_get_sa_method_str(p_mad->method),
             cl_ntoh16(p_mad->status),
             cl_ntoh16(p_mad->resv),
             cl_ntoh64(p_mad->trans_id),
             cl_ntoh16(p_mad->attr_id),
             ib_get_sa_attr_str( p_mad->attr_id ),
             cl_ntoh16(p_mad->resv1),
             cl_ntoh32(p_mad->attr_mod),
             p_mad->rmpp_version,
             p_mad->rmpp_type,
             p_mad->rmpp_flags,
             p_mad->rmpp_status,
             cl_ntoh32(p_mad->seg_num),
             cl_ntoh32(p_mad->paylen_newwin),
             cl_ntoh64(p_mad->sm_key),
             cl_ntoh16(p_mad->attr_offset),
             cl_ntoh16(p_mad->resv3),
             cl_ntoh64(p_mad->comp_mask)
             );

    strcat( buf, "\n" );

    osm_log( p_log, log_level,
             "%s\n", buf );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_dr_path(
  IN osm_log_t* const p_log,
  IN const osm_dr_path_t* const p_path,
  IN const osm_log_level_t log_level)
{
  uint32_t i;
  char buf[BUF_SIZE];
  char line[BUF_SIZE];

  if( osm_log_is_active( p_log, log_level) )
  {
    sprintf( buf, "Directed Path Dump of %u hop path:"
             "\n\t\t\t\tPath = ", p_path->hop_count );

    for( i = 0; i <= p_path->hop_count; i++ )
    {
      sprintf( line, "[%X]", p_path->path[i] );
      strcat( buf, line );
    }
    osm_log( p_log, log_level,
             "%s\n", buf );
  }
}

/**********************************************************************
 **********************************************************************/
void
osm_dump_smp_dr_path(
  IN osm_log_t* const p_log,
  IN const ib_smp_t* const p_smp,
  IN const osm_log_level_t log_level
  )
{
  uint32_t i;
  char buf[BUF_SIZE];
  char line[BUF_SIZE];

  if( osm_log_is_active( p_log, log_level) )
  {
    sprintf( buf, "Received SMP on a %u hop path:"
             "\n\t\t\t\tInitial path = ", p_smp->hop_count );

    for( i = 0; i <= p_smp->hop_count; i++ )
    {
      sprintf( line, "[%X]", p_smp->initial_path[i] );
      strcat( buf, line );
    }

    strcat( buf, "\n\t\t\t\tReturn path  = " );

    for( i = 0; i <= p_smp->hop_count; i++ )
    {
      sprintf( line, "[%X]", p_smp->return_path[i] );
      strcat( buf, line );
    }

    osm_log( p_log, log_level,
             "%s\n", buf );
  }
}

const char* const __osm_sm_state_str[] =
{
  "OSM_SM_STATE_NO_STATE",                 /* 0 */
  "OSM_SM_STATE_INIT",                     /* 1 */
  "OSM_SM_STATE_IDLE",                     /* 2 */
  "OSM_SM_STATE_SWEEP_LIGHT",                 /* 3 */
  "OSM_SM_STATE_SWEEP_LIGHT_WAIT",            /* 4 */
  "OSM_SM_STATE_SWEEP_HEAVY_SELF",            /* 5 */
  "OSM_SM_STATE_SWEEP_HEAVY_SUBNET",          /* 6 */
  "OSM_SM_STATE_SET_SM_UCAST_LID",            /* 7 */
  "OSM_SM_STATE_SET_SM_UCAST_LID_WAIT",       /* 8 */
  "OSM_SM_STATE_SET_SM_UCAST_LID_DONE",       /* 9 */
  "OSM_SM_STATE_SET_SUBNET_UCAST_LIDS",       /* 10 */
  "OSM_SM_STATE_SET_SUBNET_UCAST_LIDS_WAIT",     /* 11 */
  "OSM_SM_STATE_SET_SUBNET_UCAST_LIDS_DONE",     /* 12 */
  "OSM_SM_STATE_SET_UCAST_TABLES",            /* 13 */
  "OSM_SM_STATE_SET_UCAST_TABLES_WAIT",       /* 14 */
  "OSM_SM_STATE_SET_UCAST_TABLES_DONE",       /* 15 */
  "OSM_SM_STATE_SET_MCAST_TABLES",            /* 16 */
  "OSM_SM_STATE_SET_MCAST_TABLES_WAIT",       /* 17 */
  "OSM_SM_STATE_SET_MCAST_TABLES_DONE",       /* 18 */
  "OSM_SM_STATE_SET_LINK_PORTS",                 /* 19 */
  "OSM_SM_STATE_SET_LINK_PORTS_WAIT",               /* 20 */
  "OSM_SM_STATE_SET_LINK_PORTS_DONE",               /* 21 */
  "OSM_SM_STATE_SET_ARMED",                   /* 22 */
  "OSM_SM_STATE_SET_ARMED_WAIT",              /* 23 */
  "OSM_SM_STATE_SET_ARMED_DONE",              /* 24 */
  "OSM_SM_STATE_SET_ACTIVE",                  /* 25 */
  "OSM_SM_STATE_SET_ACTIVE_WAIT",             /* 26 */
  "OSM_SM_STATE_LOST_NEGOTIATION",            /* 27 */
  "OSM_SM_STATE_STANDBY",                     /* 28 */
  "OSM_SM_STATE_SUBNET_UP",                   /* 29 */
  "OSM_SM_STATE_PROCESS_REQUEST",             /* 30 */
  "OSM_SM_STATE_PROCESS_REQUEST_WAIT",        /* 31 */
  "OSM_SM_STATE_PROCESS_REQUEST_DONE",        /* 32 */
  "OSM_SM_STATE_MASTER_OR_HIGHER_SM_DETECTED",/* 33 */
  "OSM_SM_STATE_SET_PKEY",                    /* 34 */
  "OSM_SM_STATE_SET_PKEY_WAIT",               /* 35 */
  "OSM_SM_STATE_SET_PKEY_DONE",               /* 36 */
  "UNKNOWN STATE!!"                           /* 37 */
};

const char* const __osm_sm_signal_str[] =
{
  "OSM_SIGNAL_NONE",                       /* 0 */
  "OSM_SIGNAL_SWEEP",                      /* 1 */
  "OSM_SIGNAL_CHANGE_DETECTED",               /* 2 */
  "OSM_SIGNAL_NO_PENDING_TRANSACTIONS",       /* 3 */
  "OSM_SIGNAL_DONE",                       /* 4 */
  "OSM_SIGNAL_DONE_PENDING",                  /* 5 */
  "OSM_SIGNAL_LOST_SM_NEGOTIATION",           /* 6 */
  "OSM_SIGNAL_LIGHT_SWEEP_FAIL",              /* 7 */
  "OSM_SIGNAL_IDLE_TIME_PROCESS",             /* 8 */
  "OSM_SIGNAL_IDLE_TIME_PROCESS_REQUEST",        /* 9 */
  "OSM_SIGNAL_MASTER_OR_HIGHER_SM_DETECTED",  /* 10 */
  "OSM_SIGNAL_EXIT_STBY",                /* 11 */
  "UNKNOWN SIGNAL!!"                       /* 12 */
};

/**********************************************************************
 **********************************************************************/
const char*
osm_get_sm_state_str(
  IN osm_sm_state_t state )
{
  if( state > OSM_SM_STATE_MAX )
    state = OSM_SM_STATE_MAX;
  return( __osm_sm_state_str[state] );
}

/**********************************************************************
 **********************************************************************/
const char*
osm_get_sm_signal_str(
  IN osm_signal_t signal )
{
  if( signal > OSM_SIGNAL_MAX )
    signal = OSM_SIGNAL_MAX;
  return( __osm_sm_signal_str[signal] );
}

/**********************************************************************
 **********************************************************************/

static const char* const __osm_disp_msg_str[] =
{
  "OSM_MSG_REQ",
  "OSM_MSG_MAD_NODE_INFO",
  "OSM_MSG_MAD_PORT_INFO,",
  "OSM_MSG_MAD_SWITCH_INFO",
  "OSM_MSG_MAD_NODE_DESC",
  "OSM_MSG_NO_SMPS_OUTSTANDING",
  "OSM_MSG_MAD_NODE_RECORD",
  "OSM_MSG_MAD_PORTINFO_RECORD",
  "OSM_MSG_MAD_SERVICE_RECORD",
  "OSM_MSG_MAD_PATH_RECORD",
  "OSM_MSG_MAD_MCMEMBER_RECORD",
  "OSM_MSG_MAD_LINK_RECORD",
  "OSM_MSG_MAD_SMINFO_RECORD",
  "OSM_MSG_MAD_CLASS_PORT_INFO",
  "OSM_MSG_MAD_INFORM_INFO",
  "OSM_MSG_MAD_LFT_RECORD",
  "OSM_MSG_MAD_LFT",
  "OSM_MSG_MAD_SM_INFO",
  "OSM_MSG_MAD_NOTICE",
  "OSM_MSG_LIGHT_SWEEP_FAIL",
  "OSM_MSG_MAD_MFT",
  "OSM_MSG_MAD_PKEY_TBL_RECORD",
  "OSM_MSG_MAD_VL_ARB_RECORD",
  "OSM_MSG_MAD_SLVL_TBL_RECORD",
  "OSM_MSG_MAD_PKEY",
  "OSM_MSG_MAD_VL_ARB",
  "OSM_MSG_MAD_SLVL",
  "UNKNOWN!!"
};

/**********************************************************************
 **********************************************************************/
const char*
osm_get_disp_msg_str(
  IN cl_disp_msgid_t msg )
{
  if( msg > OSM_MSG_MAX )
    msg = OSM_MSG_MAX;
  return( __osm_disp_msg_str[msg] );
}

static const char* const __osm_port_state_str_fixed_width[] =
{
  "NOC",
  "DWN",
  "INI",
  "ARM",
  "ACT",
  "???"
};

/**********************************************************************
 **********************************************************************/
const char*
osm_get_port_state_str_fixed_width(
  IN uint8_t port_state )
{
  if( port_state > IB_LINK_ACTIVE )
    port_state = IB_LINK_ACTIVE + 1;
  return( __osm_port_state_str_fixed_width[port_state] );
}

static const char* const __osm_node_type_str_fixed_width[] =
{
  "??",
  "CA",
  "SW",
  "RT",
};

/**********************************************************************
 **********************************************************************/
const char*
osm_get_node_type_str_fixed_width(
  IN uint32_t node_type )
{
  if( node_type >= IB_NOTICE_NODE_TYPE_ROUTER )
    node_type = 0;
  return( __osm_node_type_str_fixed_width[node_type] );
}

#define OSM_VENDOR_ID_INTEL         0x00D0B7
#define OSM_VENDOR_ID_MELLANOX      0x0002C9
#define OSM_VENDOR_ID_REDSWITCH     0x000617
#define OSM_VENDOR_ID_SILVERSTORM     0x00066A
#define OSM_VENDOR_ID_TOPSPIN    0x0005AD
#define OSM_VENDOR_ID_FUJITSU    0x00E000
#define OSM_VENDOR_ID_FUJITSU2      0x000B5D
#define OSM_VENDOR_ID_VOLTAIRE   0x0008F1
#define OSM_VENDOR_ID_YOTTAYOTTA    0x000453
#define OSM_VENDOR_ID_PATHSCALE     0x001175
#define OSM_VENDOR_ID_IBM           0x000255
#define OSM_VENDOR_ID_DIVERGENET    0x00084E
#define OSM_VENDOR_ID_FLEXTRONICS   0x000B8C
#define OSM_VENDOR_ID_AGILENT       0x0030D3
#define OSM_VENDOR_ID_OBSIDIAN      0x001777

/**********************************************************************
 **********************************************************************/
const char*
osm_get_manufacturer_str(
  IN uint64_t              const guid_ho )
{
  static const char* intel_str         = "Intel      ";
  static const char* mellanox_str      = "Mellanox   ";
  static const char* redswitch_str     = "Redswitch  ";
  static const char* silverstorm_str   = "SilverStorm ";
  static const char* topspin_str       = "Topspin    ";
  static const char* fujitsu_str       = "Fujitsu    ";
  static const char* voltaire_str      = "Voltaire   ";
  static const char* yotta_str         = "YottaYotta ";
  static const char* pathscale_str     = "PathScale  ";
  static const char* ibm_str           = "IBM        ";
  static const char* divergenet_str    = "DivergeNet ";
  static const char* flextronics_str   = "Flextronics ";
  static const char* agilent_str       = "Agilent    ";
  static const char* obsidian_str      = "Obsidian   ";
  static const char* unknown_str       = "Unknown    ";

  switch( (uint32_t)(guid_ho >> (5 * 8)) )
  {
  case OSM_VENDOR_ID_INTEL:
    return( intel_str );
  case OSM_VENDOR_ID_MELLANOX:
    return( mellanox_str );
  case OSM_VENDOR_ID_REDSWITCH:
    return( redswitch_str );
  case OSM_VENDOR_ID_SILVERSTORM:
    return( silverstorm_str );
  case OSM_VENDOR_ID_TOPSPIN:
    return( topspin_str );
  case OSM_VENDOR_ID_FUJITSU:
  case OSM_VENDOR_ID_FUJITSU2:
    return( fujitsu_str );
  case OSM_VENDOR_ID_VOLTAIRE:
    return( voltaire_str );
  case OSM_VENDOR_ID_YOTTAYOTTA:
    return( yotta_str );
  case OSM_VENDOR_ID_PATHSCALE:
    return( pathscale_str );
  case OSM_VENDOR_ID_IBM:
    return( ibm_str );
  case OSM_VENDOR_ID_DIVERGENET:
    return( divergenet_str );
  case OSM_VENDOR_ID_FLEXTRONICS:
    return( flextronics_str );
  case OSM_VENDOR_ID_AGILENT:
    return( agilent_str );
  case OSM_VENDOR_ID_OBSIDIAN:
    return( obsidian_str );
  default:
    return( unknown_str );
  }
}

static const char* const __osm_mtu_str_fixed_width[] =
{
  "??? ",
  "256 ",
  "512 ",
  "1024",
  "2048",
  "4096"
};

/**********************************************************************
 **********************************************************************/
const char*
osm_get_mtu_str(
  IN uint8_t               const mtu )
{
  if( mtu > IB_MTU_LEN_4096 )
    return( __osm_mtu_str_fixed_width[0] );
  else
    return( __osm_mtu_str_fixed_width[mtu] );
}

static const char* const __osm_lwa_str_fixed_width[] =
{
  "???",
  "1x ",
  "4x ",
  "???",
  "???",
  "???",
  "???",
  "???",
  "12x"
};

/**********************************************************************
 **********************************************************************/
const char*
osm_get_lwa_str(
  IN uint8_t               const lwa )
{
  if( lwa > 8 )
    return( __osm_lwa_str_fixed_width[0] );
  else
    return( __osm_lwa_str_fixed_width[lwa] );
}

/**********************************************************************
 **********************************************************************/
static const char* const __osm_lsa_str_fixed_width[] =
{
  "???",
  "2.5",
  "5  ",
  "???",
  "10 "
};

const char*
osm_get_lsa_str(
  IN uint8_t               const lsa )
{
  if( lsa > 4 )
    return( __osm_lsa_str_fixed_width[0] );
  else
    return( __osm_lsa_str_fixed_width[lsa] );
}

/**********************************************************************
 **********************************************************************/

const char* const __osm_sm_mgr_signal_str[] =
{
  "OSM_SM_SIGNAL_INIT",                 /* 0 */
  "OSM_SM_SIGNAL_DISCOVERY_COMPLETED",   /* 2 */
  "OSM_SM_SIGNAL_POLLING_TIMEOUT",       /* 3 */
  "OSM_SM_SIGNAL_DISCOVER",              /* 4 */
  "OSM_SM_SIGNAL_DISABLE",               /* 5 */
  "OSM_SM_SIGNAL_HANDOVER",              /* 6 */
  "OSM_SM_SIGNAL_HANDOVER_SENT",         /* 7 */
  "OSM_SM_SIGNAL_ACKNOWLEDGE",           /* 8 */
  "OSM_SM_SIGNAL_STANDBY",               /* 9 */
  "OSM_SM_SIGNAL_MASTER_OR_HIGHER_SM_DETECTED",        /* 10 */
  "OSM_SM_SIGNAL_MASTER_OR_HIGHER_SM_DETECTED_DONE",    /* 11 */
  "OSM_SM_SIGNAL_WAIT_FOR_HANDOVER",     /* 12 */
  "UNKNOWN STATE!!"                      /* 13 */

};

/**********************************************************************
 **********************************************************************/
const char*
osm_get_sm_mgr_signal_str(
  IN osm_sm_signal_t signal )
{
  if( signal > OSM_SM_SIGNAL_MAX )
    signal = OSM_SM_SIGNAL_MAX;
  return( __osm_sm_mgr_signal_str[signal] );
}

const char* const __osm_sm_mgr_state_str[] =
{
  "IB_SMINFO_STATE_NOTACTIVE",                /* 0 */
  "IB_SMINFO_STATE_DISCOVERING",              /* 1 */
  "IB_SMINFO_STATE_STANDBY",                     /* 2 */
  "IB_SMINFO_STATE_MASTER",                   /* 3 */
  "IB_SMINFO_STATE_INIT",                     /* 4 */
  "UNKNOWN STATE!!"                              /* 5 */

};

const char*
osm_get_sm_mgr_state_str(
  IN uint16_t state )
{
  if( state > IB_SMINFO_STATE_INIT )
    state = IB_SMINFO_STATE_INIT + 1;
  return( __osm_sm_mgr_state_str[state] );
}

