/*
 * Copyright (c) 2005 Voltaire, Inc. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_console.c 5677 2006-03-08 12:38:34Z halr $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#define _GNU_SOURCE		/* for getline */
#include <stdio.h>
#include <stdlib.h>
#include <sys/poll.h>
#include <opensm/osm_console.h>

#define OSM_COMMAND_LINE_LEN	120
#define OSM_COMMAND_PROMPT	"$ "

struct command {
	char *name;
	void (*help_function)(void);
	void (*parse_function)(char **p_last, osm_opensm_t *p_osm);
};

static const struct command console_cmds[];

static inline char *next_token(char **p_last)
{
	return strtok_r(NULL, " \t\n", p_last);
}

static void help_command()
{
	int i;

	printf("Supported commands and syntax:\n");
	printf("help [<command>]\n");
	/* skip help command */
	for (i = 1; console_cmds[i].name; i++)
		console_cmds[i].help_function();
}

static void help_loglevel()
{
	printf("loglevel [<log-level>]\n");
}

static void help_priority()
{
	printf("priority [<sm-priority>]\n");
}

/* more help routines go here */

static void help_parse(char **p_last, osm_opensm_t *p_osm)
{
	char *p_cmd;
	int i, found = 0;

	p_cmd = next_token(p_last);
	if (!p_cmd)
		help_command();
	else {
		for (i = 1; console_cmds[i].name; i++) {
			if (!strcmp(p_cmd, console_cmds[i].name)) {
				found = 1;
				console_cmds[i].help_function();
				break;
			}
		}
		if (!found) {
			printf("Command %s not found\n\n", p_cmd);
			help_command();
		}
	}
}

static void loglevel_parse(char **p_last, osm_opensm_t *p_osm)
{
	char *p_cmd;
	int level;

	p_cmd = next_token(p_last);
	if (!p_cmd)
		printf("Current log level is 0x%x\n", osm_log_get_level(&p_osm->log));
	else {
		/* Handle x, 0x, and decimal specification of log level */
		if (!strncmp(p_cmd, "x", 1)) { 
			p_cmd++;
			level = strtoul(p_cmd, NULL, 16); 
		} else {
			if (!strncmp(p_cmd, "0x", 2)) {
				p_cmd += 2;
				level = strtoul(p_cmd, NULL, 16); 
			} else
				level = strtol(p_cmd, NULL, 10);
		}
		if ((level >= 0) && (level < 256)) {
			printf("Setting log level to 0x%x\n", level);
			osm_log_set_level(&p_osm->log, level);
		} else
			printf("Invalid log level 0x%x\n", level);
	}
}

static void priority_parse(char **p_last, osm_opensm_t *p_osm)
{
	char *p_cmd;
	int priority;

	p_cmd = next_token(p_last);
	if (!p_cmd)
		printf("Current sm-priority is %d\n", p_osm->subn.opt.sm_priority);
	else {
		priority = strtol(p_cmd, NULL, 0);
		if (0 > priority || 15 < priority)
			printf("Invalid sm-priority %d; must be between 0 and 15\n", priority);
		else {
			printf("Setting sm-priority to %d\n", priority);
			p_osm->subn.opt.sm_priority = (uint8_t)priority;
			/* Does the SM state machine need a kick now ? */
		}
	}
}

/* more parse routines go here */

static const struct command console_cmds[] =
{
	{ "help",	&help_command,		&help_parse},
	{ "loglevel",	&help_loglevel,		&loglevel_parse},	
	{ "priority",	&help_priority,		&priority_parse},
	{ NULL,		NULL,			NULL}	/* end of array */
};

static void parse_cmd_line(char *line, osm_opensm_t *p_osm)
{
	char *p_cmd, *p_last;
	int i, found = 0;

	/* find first token which is the command */
	p_cmd = strtok_r(line, " \t\n", &p_last);
	if (p_cmd) {
		for (i = 0; console_cmds[i].name; i++) {
			if (!strcmp(p_cmd, console_cmds[i].name)) {
				found = 1;
				console_cmds[i].parse_function(&p_last, p_osm);
				break;
			}
		}
		if (!found) {
			printf("Command %s not found\n\n", p_cmd);
			help_command();
		}
	} else {
		printf("Error parsing command line: %s\n", line);
		return;
	}
}

void osm_console_prompt(void)
{
	printf("%s", OSM_COMMAND_PROMPT);
	fflush(stdout);
}

void osm_console(osm_opensm_t *p_osm)
{
	struct pollfd pollfd;
	char *p_line;
	size_t len;
	ssize_t n;

	pollfd.fd = 0;
	pollfd.events = POLLIN;
	pollfd.revents = 0;

	if (poll(&pollfd, 1, 10000) <= 0)
		return;

	if (pollfd.revents|POLLIN) {
		p_line = NULL;
		/* Get input line */
		n = getline(&p_line, &len, stdin);
		if (n > 0) {
			/* Parse and act on input */
			parse_cmd_line(p_line, p_osm);
			free(p_line);
		} else {
			printf("Input error\n");
			fflush(stdin);
		}
		osm_console_prompt();
	}
}

