/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2006 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_ucast_updn.c 7494 2006-05-25 13:40:00Z halr $
 */


/*
 * Abstract:
 *      Implementation of Up Down Algorithm using ranking & Min Hop
 *      Calculation functions
 *
 * Environment:
 *      Linux User Mode
 *
 * $Revision: 1.0 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <complib/cl_debug.h>
#include <complib/cl_qmap.h>
#include <opensm/osm_switch.h>
#include <opensm/osm_opensm.h>
#include <opensm/osm_ucast_updn.h>
#include <stdlib.h>


/* ///////////////////////////////// */
/*  Globals */
/* ///////////////////////////////// */
/*  This var is pre defined and initialized */
extern osm_opensm_t osm;

/**********************************************************************
 **********************************************************************/
/* This function returns direction based on rank and guid info of current &
   remote ports */

updn_switch_dir_t
__updn_get_dir(IN uint8_t cur_rank,
               IN uint8_t rem_rank,
               IN uint64_t cur_guid,
               IN uint64_t rem_guid)
{
  uint32_t i = 0, max_num_guids = osm.p_updn_ucast_routing->updn_ucast_reg_inputs.num_guids;
  uint64_t *p_guid = osm.p_updn_ucast_routing->updn_ucast_reg_inputs.guid_list;
  boolean_t cur_is_root = FALSE , rem_is_root = FALSE;

  /* HACK: comes to solve root nodes connection, in a classic subnet root nodes does not connect
     directly, but in case they are we assign to root node an UP direction to allow UPDN discover
     correctly (and not from the point of view of the last root node) the subnet .
  */
  for ( i = 0; i < max_num_guids; i++ )
  {
    if (cur_guid == p_guid[i])
      cur_is_root = TRUE;
    if (rem_guid == p_guid[i])
      rem_is_root = TRUE;
  }
  if (cur_is_root && rem_is_root)
    return UP;


  if (cur_rank < rem_rank)
    return DOWN;
  else if (cur_rank > rem_rank)
    return UP;
  else
  {
    /* Equal rank, decide by guid number, bigger == UP direction */
    if (cur_guid > rem_guid)
      return UP;
    else
      return DOWN;
  }
}

/**********************************************************************
 **********************************************************************/
/* This function creates a new element of updn_next_step_t type then return its
   pointer , Null if malloc has failed */
updn_next_step_t*
__updn_create_updn_next_step_t(IN updn_switch_dir_t state, IN osm_switch_t* const p_sw)
{
  updn_next_step_t *p_next_step;

  p_next_step = (updn_next_step_t*) cl_zalloc(sizeof(*p_next_step));

  if (p_next_step)
  {
    p_next_step->state = state;
    p_next_step->p_sw = p_sw;
  }

  return p_next_step;
}

/**********************************************************************
 **********************************************************************/
/* This function updates an element in the qmap list by guid index and rank value */
/* Return 0 if no need to futher update 1 if brought a new value */
int
__updn_update_rank(
  IN cl_qmap_t *p_guid_rank_tbl,
  IN ib_net64_t guid_index,
  IN uint8_t rank)
{
  updn_rank_t *p_updn_rank;

  p_updn_rank = (updn_rank_t*) cl_qmap_get(p_guid_rank_tbl, guid_index);
  if (p_updn_rank == (updn_rank_t*) cl_qmap_end(p_guid_rank_tbl))
  {
    p_updn_rank = (updn_rank_t*) cl_malloc(sizeof(updn_rank_t));
    CL_ASSERT (p_updn_rank);

    p_updn_rank->rank = rank;

    cl_qmap_insert(p_guid_rank_tbl, guid_index , &p_updn_rank->map_item);
    return 1;
  }
  else
  {
    if (p_updn_rank->rank > rank)
    {
      p_updn_rank->rank = rank;
      return 1;
    }
  }
  return 0;
}

/**********************************************************************
 * This function do the bfs of min hop table calculation by guid index as a
 * starting point.
 **********************************************************************/
int
__updn_bfs_by_node(IN osm_subn_t *p_subn, IN ib_net64_t guid_index, IN cl_qmap_t *p_guid_rank_tbl)
{
  /*  Init local vars */
  osm_port_t *p_port;
  osm_switch_t *p_self_node = NULL;
  uint8_t pn, pn_rem;
  osm_physp_t   *p_physp, *p_remote_physp;
  cl_list_t     *p_currList, *p_nextList;
  uint16_t root_lid, max_sw_lid;
  updn_next_step_t *p_updn_switch, *p_tmp;
  updn_switch_dir_t next_dir, current_dir;

  OSM_LOG_ENTER( &(osm.log), __updn_bfs_by_node);

  /* Init the list pointers */
  p_nextList = (cl_list_t*)cl_malloc(sizeof(cl_list_t));
  cl_list_construct( p_nextList );
  cl_list_init( p_nextList, 10 );
  p_currList = p_nextList;

  p_port = (osm_port_t*) cl_qmap_get(&(p_subn->port_guid_tbl),guid_index);
  /* TODO : check if p_port is not NULL */
  p_physp = osm_port_get_default_phys_ptr(p_port);
  /* Check valid pointer */
  if (!p_physp || !osm_physp_is_valid(p_physp ))
  {
    OSM_LOG_EXIT( &(osm.log));
    return 1;
  }
  /* The Root BFS - lid  */
  root_lid = cl_ntoh16(osm_physp_get_base_lid( p_physp ));
  /* printf ("-V- BFS through lid : 0x%x\n",root_lid); */
  osm_log(&(osm.log), OSM_LOG_DEBUG,
          "__updn_bfs_by_node:"
          "Starting lid : 0x%x \n", root_lid);

  if (osm_node_get_type( p_port->p_node ) == IB_NODE_TYPE_SWITCH)
  {
    p_self_node = osm_get_switch_by_guid(p_subn, guid_index);
    /* Update its Min Hop Table */
    osm_log(&(osm.log), OSM_LOG_DEBUG,
            "__updn_bfs_by_node:"
            "Update Min Hop Table of GUID 0x%" PRIx64 "\n"
            , cl_ntoh64(p_port->guid));
    osm_switch_set_hops(p_self_node, root_lid , 0, 0);
  }
  else
  {
    /* This is an HCA need to take its remote port  */
    p_remote_physp = p_physp->p_remote_physp;
    /*
      make sure that the following occur:
      1. The port isn't NULL
      2. The port is a valid port
    */
    if ( p_remote_physp && osm_physp_is_valid ( p_remote_physp ))
    {
      /* Check if the remote port is a switch, if it is update root_lid,
         Min Hop Table */
      if (osm_node_get_type(p_remote_physp->p_node) != IB_NODE_TYPE_SWITCH)
      {
        osm_log(&(osm.log), OSM_LOG_ERROR,
                "__updn_bfs_by_node: ERR AA07: "
                "This is a non switched subnet OR non valid connection, cannot perform UPDN algorithm\n");
        OSM_LOG_EXIT( &(osm.log));
        return 1;
      } else
      {
        p_self_node = osm_get_switch_by_guid(p_subn,
                                             osm_physp_get_port_guid
                                             (p_remote_physp));
        max_sw_lid = osm_switch_get_max_lid_ho(p_self_node);
        if ((1 <= root_lid) && (root_lid <= max_sw_lid))
          /* Update its Min Hop Table */
        {
          /* NOTE : Check if there is a function which prints the Min Hop Table */
          osm_log(&(osm.log), OSM_LOG_DEBUG,
                  "__updn_bfs_by_node:"
                  "Update Min Hop Table of GUID 0x%" PRIx64 "\n"
                  , cl_ntoh64(p_remote_physp->port_guid));
          osm_switch_set_hops(p_self_node, root_lid
                              , p_remote_physp->port_num, 1);

        } else
        {
          osm_log(&(osm.log), OSM_LOG_ERROR,
                  "__updn_bfs_by_node: ERR AA09: "
                  " Invalid lid value 0x%x for switch 0x%" PRIx64 "\n", root_lid
                  , cl_ntoh64(p_self_node->p_node->node_info.port_guid));
          OSM_LOG_EXIT( &(osm.log));
          return 1;
        }
      }
    }
  }

  CL_ASSERT(p_self_node);

  osm_log(&(osm.log), OSM_LOG_DEBUG,
          "__updn_bfs_by_node:"
          "Starting from switch - port GUID 0x%" PRIx64 "\n"
          , cl_ntoh64(p_self_node->p_node->node_info.port_guid));

  /* Update list with the updn_next_step_t new element */
  /* NOTE : When inserting an item which is a pointer to a struct , does remove
     action also free its memory */
  if (!(p_tmp=__updn_create_updn_next_step_t(UP, p_self_node)))
  {
    osm_log(&(osm.log), OSM_LOG_ERROR,
            "__updn_bfs_by_node:  ERR AA08: "
            "Could not create updn_next_step_t\n");
    return 1;
  }

  cl_list_insert_tail(p_currList,p_tmp );

  /* BFS the list till no next element */
  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "__updn_bfs_by_node:"
          "BFS the subnet [\n");

  while (!cl_is_list_empty(p_currList))
  {
    osm_log(&(osm.log), OSM_LOG_DEBUG,
            "__updn_bfs_by_node:"
            "Starting a new iteration with %d elements in current list\n", cl_list_count(p_currList));
    /* Init the switch directed list */
    p_nextList = (cl_list_t*)cl_malloc(sizeof(cl_list_t));
    cl_list_construct( p_nextList );
    cl_list_init( p_nextList, 10 );
    /* Go over all current list items till it's empty */
    /*       printf ("-V- In inner while\n"); */
    p_updn_switch = (updn_next_step_t*)cl_list_remove_head( p_currList );
    /* While there is a pointer to updn struct we continue to BFS */
    while (p_updn_switch)
    {
      current_dir = p_updn_switch->state;
      osm_log(&(osm.log), OSM_LOG_DEBUG,
              "__updn_bfs_by_node:"
              "Visiting port GUID 0x%" PRIx64 "\n"
              , cl_ntoh64(p_updn_switch->p_sw->p_node->node_info.port_guid));
      /* Go over all ports of the switch and find unvisited remote nodes */
      for ( pn = 0; pn < osm_switch_get_num_ports(p_updn_switch->p_sw); pn++ )
      {
        /*  printf("-V- Inner for in port num %d\n", pn); */
        osm_node_t *p_remote_node;
        cl_list_iterator_t updn_switch_iterator;
        boolean_t HasVisited = FALSE;
        ib_net64_t remote_guid,current_guid;
        updn_rank_t *p_rem_rank, *p_cur_rank;
        uint8_t current_min_hop, remote_min_hop, set_hop_return_value;
        osm_switch_t *p_remote_sw;

        current_guid = osm_node_get_node_guid(p_updn_switch->p_sw->p_node);
        p_remote_node = osm_node_get_remote_node( p_updn_switch->p_sw->p_node
                                                  , pn, &pn_rem );
        /* If no remote node OR remote node is not a SWITCH
           continue to next pn */
        if( !p_remote_node ||
            (osm_node_get_type(p_remote_node) != IB_NODE_TYPE_SWITCH) )
          continue;
        /* Fetch remote guid only after validation of remote node */
        remote_guid = osm_node_get_node_guid(p_remote_node);
        /*  printf ("-V- Current guid : 0x%" PRIx64 " Remote guid : 0x%" PRIx64 "\n" */
        /*  , cl_ntoh64(current_guid), cl_ntoh64(remote_guid)); */
        p_remote_sw = osm_get_switch_by_guid(p_subn, remote_guid);
        p_rem_rank = (updn_rank_t*)cl_qmap_get(p_guid_rank_tbl, remote_guid);
        p_cur_rank = (updn_rank_t*)cl_qmap_get(p_guid_rank_tbl, current_guid);
        /* Decide which direction to mark it (UP/DOWN) */
        next_dir = __updn_get_dir (p_cur_rank->rank, p_rem_rank->rank,
                                   current_guid, remote_guid);

        osm_log(&(osm.log), OSM_LOG_DEBUG,
                "__updn_bfs_by_node:"
                "move from 0x%016" PRIx64 " rank: %u "
                "to 0x%016" PRIx64" rank: %u\n",
                cl_ntoh64(current_guid), p_cur_rank->rank,
                cl_ntoh64(remote_guid), p_rem_rank->rank);
        /* Check if this is a legal step : the only illegal step is going
           from DOWN to UP */
        if ((current_dir == DOWN) && (next_dir == UP))
        {
          osm_log(&(osm.log), OSM_LOG_DEBUG,
                  "__updn_bfs_by_node:"
                  "Avoiding move from 0x%016" PRIx64 " to 0x%016" PRIx64"\n",
                  cl_ntoh64(current_guid), cl_ntoh64(remote_guid));
          /* Illegal step */
          continue;
        }
        /* Set MinHop value for the current lid */
        current_min_hop = osm_switch_get_least_hops(p_updn_switch->p_sw,root_lid);
        /* Check hop count if better insert into NextState list && update
           the remote node Min Hop Table */
        remote_min_hop = osm_switch_get_hop_count(p_remote_sw,root_lid, pn_rem);
        if (current_min_hop + 1 < remote_min_hop)
        {
          osm_log(&(osm.log), OSM_LOG_DEBUG,
                  "__updn_bfs_by_node (less):"
                  "Setting Min Hop Table of switch: 0x%" PRIx64
                  "\n\t\tCurrent hop count is: %d, next hop count: %d"
                  "\n\tlid to set: 0x%x"
                  "\n\tport number: %d"
                  " \n\thops number: %d\n"
                  , cl_ntoh64(remote_guid), remote_min_hop,current_min_hop + 1, root_lid, pn_rem, current_min_hop + 1);
          set_hop_return_value=osm_switch_set_hops(p_remote_sw, root_lid, pn_rem, current_min_hop + 1);
          if (set_hop_return_value)
          {
            osm_log(&(osm.log), OSM_LOG_ERROR,
                    "__updn_bfs_by_node (less) ERR AA01: "
                    "Invalid value returned from set min hop is: %d\n", set_hop_return_value);
          }
          /* Check if remote port is allready has been visited */
          updn_switch_iterator = cl_list_head(p_nextList);
          while( updn_switch_iterator != cl_list_end(p_nextList) )
          {
            updn_next_step_t *p_updn;
            p_updn = (updn_next_step_t*)cl_list_obj(updn_switch_iterator);
            /* Mark HasVisited only if:
               1. Same node guid
               2. Same direction
            */
            if ((p_updn->p_sw->p_node == p_remote_node) && (p_updn->state == next_dir))
              HasVisited = TRUE;
            updn_switch_iterator = cl_list_next(updn_switch_iterator);
          }
          if (!HasVisited)
          {
            /* Insert updn_switch item into the next list */
            if(!(p_tmp=__updn_create_updn_next_step_t(next_dir, p_remote_sw)))
            {
              osm_log(&(osm.log), OSM_LOG_ERROR,
                      "__updn_bfs_by_node:  ERR AA11: "
                      "Could not create updn_next_step_t\n");
              return 1;
            }
            osm_log(&(osm.log), OSM_LOG_DEBUG,
                    "__updn_bfs_by_node: "
                    "Inserting a new element to the next list: guid=0x%" PRIx64 " %s\n"
                    , cl_ntoh64(p_tmp->p_sw->p_node->node_info.port_guid),
                    (p_tmp->state == UP ? "UP" : "DOWN")
                    );
            cl_list_insert_tail(p_nextList, p_tmp);
          }
          /* If the same value only update entry - at the min hop table */
        } else if (current_min_hop + 1 == osm_switch_get_hop_count(p_remote_sw
                                                                   , root_lid
                                                                   , pn_rem))
        {
          osm_log(&(osm.log), OSM_LOG_DEBUG,
                  "__updn_bfs_by_node (equal):"
                  "Setting Min Hop Table of switch: 0x%" PRIx64
                  "\n\t\tCurrent hop count is: %d, next hop count: %d"
                  "\n\tlid to set: 0x%x"
                  "\n\tport number: %d"
                  "\n\thops number: %d\n"
                  , cl_ntoh64(remote_guid), osm_switch_get_hop_count(p_remote_sw, root_lid, pn_rem)
                  , current_min_hop + 1, root_lid, pn_rem, current_min_hop + 1);
          set_hop_return_value = osm_switch_set_hops(p_remote_sw, root_lid, pn_rem, current_min_hop + 1);

          if (set_hop_return_value)
          {
            osm_log(&(osm.log), OSM_LOG_ERROR,
                    "__updn_bfs_by_node (less) ERR AA12: "
                    "Invalid value returned from set min hop is: %d\n", set_hop_return_value);
          }
        }
      }
      cl_free (p_updn_switch);
      p_updn_switch = (updn_next_step_t*)cl_list_remove_head( p_currList );
    }
    /* Cleanup p_currList */
    cl_list_destroy( p_currList );
    cl_free (p_currList);

    /* Reassign p_currList to p_nextList */
    p_currList = p_nextList;
  }
  /* Cleanup p_currList - Had the pointer to cl_list_t */
  cl_list_destroy( p_currList );
  cl_free (p_currList);
  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "__updn_bfs_by_node:"
          "BFS the subnet ]\n");
  OSM_LOG_EXIT( &(osm.log));
  return 0;
}

/**********************************************************************
 **********************************************************************/
void
updn_destroy(
  IN updn_t* const p_updn )
{
  cl_map_item_t *p_map_item;
  uint64_t *p_guid_list_item;

  /* Destroy the updn struct */
  p_map_item = cl_qmap_head( &p_updn->guid_rank_tbl);
  while( p_map_item != cl_qmap_end( &p_updn->guid_rank_tbl))
  {
    osm_log (&(osm.log), OSM_LOG_DEBUG,
             "osm_subn_calc_up_down_min_hop_table: "
             "guid = 0x%" PRIx64 " rank = %u\n",
             cl_ntoh64(cl_qmap_key(p_map_item)), ((updn_rank_t *)p_map_item)->rank);
    cl_qmap_remove_item( &p_updn->guid_rank_tbl, p_map_item);
    cl_free( (updn_rank_t *)p_map_item);
    p_map_item = cl_qmap_head( &p_updn->guid_rank_tbl);
  }

  /* free the array of guids */
  if (p_updn->updn_ucast_reg_inputs.guid_list)
    cl_free(p_updn->updn_ucast_reg_inputs.guid_list);

  /* destroy the list of root nodes */
  while ((p_guid_list_item = cl_list_remove_head( p_updn->p_root_nodes )))
    cl_free( p_guid_list_item );

  cl_list_remove_all( p_updn->p_root_nodes );
  cl_list_destroy( p_updn->p_root_nodes );
  cl_free ( p_updn->p_root_nodes );
  cl_free (p_updn);
}

updn_t*
updn_construct(void)
{
  updn_t* p_updn;

  OSM_LOG_ENTER( &(osm.log), updn_construct);
  p_updn = cl_zalloc(sizeof(updn_t));
  OSM_LOG_EXIT( &(osm.log) );
  return(p_updn);
}

/**********************************************************************
 **********************************************************************/
cl_status_t
updn_init(
  IN updn_t* const p_updn )
{
  cl_list_t * p_list;
  FILE*           p_updn_guid_file;
  char            line[MAX_UPDN_GUID_FILE_LINE_LENGTH];
  uint64_t * p_tmp;
  cl_list_iterator_t guid_iterator;
  ib_api_status_t status = IB_SUCCESS;

  OSM_LOG_ENTER( &(osm.log), updn_init );
  /* Make sure the p_updn isn't NULL */
  if (!p_updn)
  {
    status = IB_ERROR;
    goto Exit_Bad;
  }
  p_updn->state = UPDN_INIT;
  cl_qmap_init( &p_updn->guid_rank_tbl);
  p_list = (cl_list_t*)cl_malloc(sizeof(cl_list_t));
  if (!p_list)
  {
    status = IB_ERROR;
    goto Exit_Bad;
  }

  cl_list_construct( p_list );
  cl_list_init( p_list, 10 );
  p_updn->p_root_nodes = p_list;
  p_updn->updn_ucast_reg_inputs.num_guids = 0;
  p_updn->updn_ucast_reg_inputs.guid_list = NULL;
  p_updn->auto_detect_root_nodes = FALSE;
  /* Check if updn is activated , then fetch root nodes */
  if (osm.subn.opt.updn_activate)
  {
    /*
       Check the source for root node list, if file parse it, otherwise
       wait for a callback to activate auto detection
    */
    if (osm.subn.opt.updn_guid_file)
    {
      /* Now parse guid from file */
      p_updn_guid_file = fopen(osm.subn.opt.updn_guid_file, "r");
      if (p_updn_guid_file == NULL)
      {
        osm_log( &osm.log, OSM_LOG_ERROR,
                 "osm_opensm_init : ERR AA02: "
                 "Failed to open guid list file (%s)\n",
                 osm.subn.opt.updn_guid_file);
        status = IB_NOT_FOUND;
        goto Exit;
      }

      while ( fgets(line, MAX_UPDN_GUID_FILE_LINE_LENGTH, p_updn_guid_file) )
      {
        if (strcspn(line, " ,;.") == strlen(line))
        {
          /* Skip Empty Lines anywhere in the file - only one char means the Null termination */
          if (strlen(line) > 1)
          {
            p_tmp = cl_malloc(sizeof(uint64_t));
            *p_tmp = strtoull(line, NULL, 16);
            cl_list_insert_tail(osm.p_updn_ucast_routing->p_root_nodes, p_tmp);
          }
        }
        else
        {
          osm_log( &osm.log, OSM_LOG_ERROR,
                   "osm_opensm_init: ERR AA03: "
                   "Bad formatted guid in file (%s) : %s\n"
                   , osm.subn.opt.updn_guid_file,line);
          status = IB_NOT_FOUND;
          break;
        }
      }

      /* For Debug Purposes ... */
      osm_log( &osm.log, OSM_LOG_DEBUG,
               "osm_opensm_init: "
               "UPDN - Root nodes fetching by file %s\n",
               osm.subn.opt.updn_guid_file);
      guid_iterator = cl_list_head(osm.p_updn_ucast_routing->p_root_nodes);
      while( guid_iterator != cl_list_end(osm.p_updn_ucast_routing->p_root_nodes) )
      {
        osm_log( &osm.log, OSM_LOG_DEBUG,
                 "osm_opensm_init: "
                 "Inserting GUID 0x%" PRIx64 " as root node\n",
                 *((uint64_t*)cl_list_obj(guid_iterator)) );
        guid_iterator = cl_list_next(guid_iterator);
      }
    }
    else
    {
      osm.p_updn_ucast_routing->auto_detect_root_nodes = TRUE;
    }
    /* If auto mode detection reuired - will be executed in main b4 the assignment of UI Ucast */
  }

  goto Exit;

  Exit_Bad :
    return 1;
  Exit :
    OSM_LOG_EXIT( &(osm.log) );
  return (status);
}

/**********************************************************************
 **********************************************************************/
/* NOTE : PLS check if we need to decide that the first */
/*        rank is a SWITCH for BFS purpose */
int
updn_subn_rank(
  IN uint64_t root_guid,
  IN uint8_t base_rank,
  IN updn_t* p_updn)
{
  /* Init local vars */
  osm_port_t *p_root_port = NULL;
  uint16_t tbl_size;
  uint8_t rank = base_rank;
  osm_physp_t   *p_physp, *p_remote_physp, *p_physp_temp;
  cl_list_t     *p_currList,*p_nextList;
  cl_status_t did_cause_update;
  uint8_t num_ports, port_num;

  OSM_LOG_ENTER( &(osm.log), updn_subn_rank);

  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "updn_subn_rank: "
          "Ranking starts from GUID 0x%" PRIx64 "\n", root_guid);

  /* Init the list pointers */
  p_nextList = (cl_list_t*)cl_malloc(sizeof(cl_list_t));
  cl_list_construct( p_nextList );
  cl_list_init( p_nextList, 10 );
  p_currList = p_nextList;

  /* Check valid subnet & guid */
  tbl_size = (uint16_t)(cl_qmap_count(&(osm.subn.port_guid_tbl)));
  if (tbl_size == 0)
  {
    osm_log(&(osm.log), OSM_LOG_ERROR,
            "updn_subn_rank: ERR AA04: "
            "Port guid table is empty, cannot perform ranking\n");
    OSM_LOG_EXIT( &(osm.log));
    return 1;
  }

  p_root_port = (osm_port_t*) cl_qmap_get(&(osm.subn.port_guid_tbl), \
                                          cl_ntoh64(root_guid));
  if( p_root_port == (osm_port_t*)cl_qmap_end( &(osm.subn.port_guid_tbl) ) )
  {

    osm_log(&(osm.log), OSM_LOG_ERROR,
            "updn_subn_rank: ERR AA05: "
            "Wrong guid value: 0x%" PRIx64 "\n", root_guid);
    OSM_LOG_EXIT( &(osm.log));
    return 1;
  }

  /* Rank the first chosen guid anyway since its the base rank */
  osm_log(&(osm.log), OSM_LOG_DEBUG,
          "updn_subn_rank: "
          "Ranking port GUID 0x%" PRIx64 "\n",root_guid);

  __updn_update_rank(&p_updn->guid_rank_tbl, cl_ntoh64(root_guid), rank);
  /*
    HACK: We are assuming SM is running on HCA, so when getting the default
    port we'll get the port connected to the rest of the subnet. If SM is
    running on SWITCH - we should try to get a dr path from all switch ports.
  */
  p_physp = osm_port_get_default_phys_ptr( p_root_port );
  CL_ASSERT( p_physp );
  CL_ASSERT( osm_physp_is_valid( p_physp ) );
  /* We can safely add the node to the list */
  cl_list_insert_tail(p_nextList, p_physp);
  /* Assign pointer to the list for BFS */
  p_currList = p_nextList;

  /* BFS the list till its empty */
  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "updn_subn_rank: "
          "BFS the subnet [\n");

  while (!cl_is_list_empty(p_currList))
  {
    rank++;
    p_nextList = (cl_list_t*)cl_malloc(sizeof(cl_list_t));
    cl_list_construct( p_nextList );
    cl_list_init( p_nextList, 10 );
    p_physp = (osm_physp_t*)cl_list_remove_head( p_currList );
    /* Go over all remote nodes and rank them (if not allready visited) till
       no elemtent in the list p_currList */
    while ( p_physp != NULL )
    {
      num_ports = osm_node_get_num_physp( p_physp->p_node );
      osm_log(&(osm.log), OSM_LOG_DEBUG,
              "updn_subn_rank: "
              "Handling port GUID 0x%" PRIx64 "\n", cl_ntoh64(p_physp->port_guid));
      for (port_num = 1; port_num < num_ports; port_num++)
      {
        ib_net64_t port_guid;

        /* Current port fetched in order to get remote side */
        p_physp_temp = osm_node_get_physp_ptr( p_physp->p_node, port_num );
        p_remote_physp = p_physp_temp->p_remote_physp;

        /*
          make sure that all the following occure on p_remote_physp:
          1. The port isn't NULL
          2. The port is a valid port
        */
        if ( p_remote_physp &&
             osm_physp_is_valid ( p_remote_physp ))
        {
          port_guid = p_remote_physp->port_guid;
          osm_log(&(osm.log), OSM_LOG_DEBUG,
                  "updn_subn_rank: "
                  "Visiting remote port GUID 0x%" PRIx64 "\n"
                  , cl_ntoh64(port_guid));
          /* Was it visited ?
             Only if the pointer equal to cl_qmap_end its not
             found in the list */
          osm_log(&(osm.log), OSM_LOG_DEBUG,
                  "updn_subn_rank: "
                  "Ranking port GUID 0x%" PRIx64 "\n", cl_ntoh64(port_guid));
          did_cause_update = __updn_update_rank(&p_updn->guid_rank_tbl, port_guid, rank);

          osm_log(&(osm.log), OSM_LOG_VERBOSE,
                  "updn_subn_rank: "
                  "Rank of port GUID 0x%" PRIx64 " = %u\n", cl_ntoh64(port_guid),
                  ((updn_rank_t*)cl_qmap_get(&p_updn->guid_rank_tbl, port_guid))->rank
                  );

          if (did_cause_update)
          {
            cl_list_insert_tail(p_nextList, p_remote_physp);
          }
        }
      }
      /* Propagte through the next item in the p_currList */
      p_physp = (osm_physp_t*)cl_list_remove_head( p_currList );
    }
    /* First free the allocation of cl_list pointer then reallocate */
    cl_list_destroy( p_currList );
    cl_free(p_currList);
    /* p_currList is empty - need to assign it to p_nextList */
    p_currList = p_nextList;
  }

  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "updn_subn_rank: "
          "BFS the subnet ]\n");

  cl_list_destroy( p_currList );
  cl_free(p_currList);

  /* Print Summary of ranking */
  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "updn_subn_rank: "
          "Rank Info :\n\t Root Guid = 0x%" PRIx64 "\n\t Max Node Rank = %d\n"
          , cl_ntoh64(p_root_port->guid),rank);
  p_updn->state = UPDN_RANK;
  OSM_LOG_EXIT( &(osm.log));
  return 0;
}

/**********************************************************************
 **********************************************************************/
int
osm_subn_set_up_down_min_hop_table(
  IN updn_t* p_updn)
{
  /* Init local vars */
  osm_subn_t *p_subn = &(osm.subn);
  osm_switch_t *p_next_sw,*p_sw;
  osm_port_t *p_next_port,*p_port;
  ib_net64_t port_guid;

  OSM_LOG_ENTER( &(osm.log), osm_subn_set_up_down_min_hop_table );
  if (p_updn->state == UPDN_INIT)
  {
    osm_log(&(osm.log), OSM_LOG_ERROR,
            "osm_subn_set_up_down_min_hop_table: ERR AA06: "
            "Calculating Min Hop only allowed after ranking\n");
    OSM_LOG_EXIT( &(osm.log) );
    return 1;
  }

  /* Check if its a non switched subnet .. */
  if ( cl_is_qmap_empty( &p_subn->sw_guid_tbl ) )
  {
    osm_log(&(osm.log), OSM_LOG_ERROR,
            "osm_subn_set_up_down_min_hop_table: ERR AA10: "
            "This is a non switched subnet, cannot perform UPDN algorithm\n");
    OSM_LOG_EXIT( &(osm.log));
    return 1;
  }
  /* Go over all the switches in the subnet - for each init their Min Hop
     Table */
  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "osm_subn_set_up_down_min_hop_table: "
          "Init Min Hop Table of all switches [\n");

  p_next_sw = (osm_switch_t*)cl_qmap_head( &p_subn->sw_guid_tbl );
  while( p_next_sw != (osm_switch_t*)cl_qmap_end( &p_subn->sw_guid_tbl ) )
  {
    uint16_t max_lid_ho, lid_ho;

    p_sw = p_next_sw;
    p_next_sw = (osm_switch_t*)cl_qmap_next( &p_sw->map_item );
    /* Clear Min Hop Table && FWD Tbls - This should caused opensm to
       rebuild its FWD tables , post setting Min Hop Tables */
    osm_lid_matrix_clear(&(p_sw->lmx));
    max_lid_ho = osm_switch_get_max_lid_ho(p_sw);
    for (lid_ho = 1; lid_ho <= max_lid_ho; lid_ho++)
      osm_switch_set_path(p_sw, lid_ho, OSM_NO_PATH,TRUE);
  }

  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "osm_subn_set_up_down_min_hop_table: "
          "Init Min Hop Table of all switches ]\n");

  /* Now do the BFS for each port  in the subnet */
  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "osm_subn_set_up_down_min_hop_table: "
          "BFS through all port guids in the subnet [\n");
  p_next_port = (osm_port_t*)cl_qmap_head( &p_subn->port_guid_tbl );
  while( p_next_port != (osm_port_t*)cl_qmap_end( &p_subn->port_guid_tbl ) )
  {
    p_port = p_next_port;
    p_next_port = (osm_port_t*)cl_qmap_next( &p_port->map_item );
    port_guid = cl_qmap_key(&(p_port->map_item));
    osm_log(&(osm.log), OSM_LOG_DEBUG,
            "BFS through port GUID 0x%" PRIx64 "\n"
            , cl_ntoh64(port_guid));
    if(__updn_bfs_by_node(p_subn,port_guid,
                          &p_updn->guid_rank_tbl))
    {
      OSM_LOG_EXIT( &(osm.log) );
      return 1;
    }
  }

  osm_log(&(osm.log), OSM_LOG_INFO,
          "osm_subn_set_up_down_min_hop_table: "
          "BFS through all port guids in the subnet ]\n");
  /* Cleanup */
  OSM_LOG_EXIT( &(osm.log) );
  return 0;
}

/**********************************************************************
 **********************************************************************/
int
osm_subn_calc_up_down_min_hop_table(
  IN uint32_t num_guids,
  IN uint64_t * guid_list,
  IN updn_t* p_updn)
{
  uint8_t idx = 0;
  cl_map_item_t *p_map_item;
  int status;

  OSM_LOG_ENTER( &(osm.log), osm_subn_calc_up_down_min_hop_table );
  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "osm_subn_calc_up_down_min_hop_table: "
          "Ranking all port guids in the list\n");
  if (num_guids == 0)
  {
    osm_log(&(osm.log), OSM_LOG_ERROR,
            "osm_subn_calc_up_down_min_hop_table: "
            "No guids were given or number of guids is 0\n");
    return 1;
  }

  for (idx = 0; idx < num_guids; idx++)
  {
    /* Apply the ranking for each guid given by user  - bypass illegal ones */
    updn_subn_rank(guid_list[idx], 0, p_updn);
  }
  /* After multiple ranking need to set Min Hop Table by UpDn algorithm  */
  osm_log(&(osm.log), OSM_LOG_VERBOSE,
          "osm_subn_calc_up_down_min_hop_table: "
          "Setting all switches' Min Hop Table\n");

  status = osm_subn_set_up_down_min_hop_table (p_updn);

  /* Cleanup updn rank tbl */
  p_map_item = cl_qmap_head( &p_updn->guid_rank_tbl);
  while( p_map_item != cl_qmap_end( &p_updn->guid_rank_tbl))
  {
    osm_log (&(osm.log), OSM_LOG_DEBUG,
             "osm_subn_calc_up_down_min_hop_table: "
             "guid = 0x%" PRIx64 " rank = %u\n",
             cl_ntoh64(cl_qmap_key(p_map_item)), ((updn_rank_t *)p_map_item)->rank);
    cl_qmap_remove_item( &p_updn->guid_rank_tbl, p_map_item);
    cl_free( (updn_rank_t *)p_map_item);
    p_map_item = cl_qmap_head( &p_updn->guid_rank_tbl);
  }

  OSM_LOG_EXIT( &(osm.log));
  return status;

}

/**********************************************************************
 **********************************************************************/
/* UPDN callback function */
int __osm_updn_call(void *ctx)
{
  OSM_LOG_ENTER(&(osm.log), __osm_updn_call);
  /* First auto detect root nodes - if required */
  if ( ((updn_t*)ctx)->auto_detect_root_nodes )
  {
    /*    printf ("-V- b4 osm_updn_find_root_nodes_by_min_hop\n");*/
    osm_updn_find_root_nodes_by_min_hop( ((updn_t*)ctx) );
  }
  /*  printf ("-V- after osm_updn_find_root_nodes_by_min_hop\n"); */
  /* Only if there are assigned root nodes do the algorithm , otherwise perform do nothing */
  if ( ((updn_t*)ctx)->updn_ucast_reg_inputs.num_guids > 0)
  {
    osm_log (&(osm.log), OSM_LOG_DEBUG,
             "__osm_updn_call: "
             "activating UPDN algorithm\n");
    osm_subn_calc_up_down_min_hop_table( ((updn_t*)ctx)->updn_ucast_reg_inputs.num_guids,
                                         ((updn_t*)ctx)->updn_ucast_reg_inputs.guid_list
                                         , ((updn_t*)ctx) );
  }
  else
    osm_log (&(osm.log), OSM_LOG_INFO,
             "__osm_updn_call: "
             "disable UPDN algorithm, no root nodes were found\n");
  
  OSM_LOG_EXIT(&(osm.log));
  return 0;
}

/**********************************************************************
 **********************************************************************/
/* UPDN convert cl_list to guid array in updn struct */
void __osm_updn_convert_list2array(IN updn_t * p_updn)
{
  uint32_t i = 0, max_num = 0;
  uint64_t *p_guid;

  OSM_LOG_ENTER(&(osm.log), __osm_updn_convert_list2array);
  p_updn->updn_ucast_reg_inputs.num_guids = cl_list_count(
    p_updn->p_root_nodes);
  if (p_updn->updn_ucast_reg_inputs.guid_list)
    cl_free(p_updn->updn_ucast_reg_inputs.guid_list);
  p_updn->updn_ucast_reg_inputs.guid_list = (uint64_t *)cl_zalloc(
    p_updn->updn_ucast_reg_inputs.num_guids*sizeof(uint64_t));
  if (!cl_is_list_empty(p_updn->p_root_nodes))
  {
    while( (p_guid = (uint64_t*)cl_list_remove_head(p_updn->p_root_nodes)) )
    {
      p_updn->updn_ucast_reg_inputs.guid_list[i] = *p_guid;
      cl_free(p_guid);
      i++;
    }
    max_num = i;
    for (i = 0; i < max_num; i++ )
      osm_log (&(osm.log), OSM_LOG_DEBUG,
               "__osm_updn_convert_list2array: "
               "Map GUID 0x%" PRIx64 " into UPDN array\n",
               p_updn->updn_ucast_reg_inputs.guid_list[i]);
  }
  /* Since we need the template list for other sweeps, we wont destroy & free it */
  OSM_LOG_EXIT(&(osm.log));
}

/**********************************************************************
 **********************************************************************/
/* Registration function to ucast routing manager (instead of
   Min Hop Algorithm) */
int
osm_updn_reg_calc_min_hop_table(
  IN updn_t * p_updn,
  IN osm_subn_opt_t* p_opt )
{
  OSM_LOG_ENTER(&(osm.log), osm_updn_reg_calc_min_hop_table);
  /*
     If root nodes were supplied by the user - we need to convert into array
     otherwise, will be created & converted in callback function activation
  */
  if (!p_updn->auto_detect_root_nodes)
  {
    __osm_updn_convert_list2array(p_updn);
  }
  osm_log (&(osm.log), OSM_LOG_DEBUG,
           "osm_updn_reg_calc_min_hop_table: "
           "assigning ucast fdb UI function with updn callback\n");
  p_opt->pfn_ui_ucast_fdb_assign = __osm_updn_call;
  p_opt->ui_ucast_fdb_assign_ctx = (void *)p_updn;
  OSM_LOG_EXIT(&(osm.log));
  return 0;
}

/**********************************************************************
 **********************************************************************/
/* Find Root nodes automatically by Min Hop Table info */
int
osm_updn_find_root_nodes_by_min_hop( OUT updn_t *  p_updn )
{
  osm_switch_t *p_next_sw, *p_sw;
  osm_port_t   *p_next_port, *p_port;
  osm_physp_t  *p_physp;
  uint32_t      numCas = 0;
  uint32_t      numSws = cl_qmap_count(&osm.subn.sw_guid_tbl);
  cl_qmap_t     min_hop_hist; /* Histogram container */
  updn_hist_t  *p_updn_hist, *p_up_ht;
  uint8_t       maxHops = 0; /* contain the max histogram index */
  uint64_t     *p_guid;
  cl_list_t    *p_root_nodes_list = p_updn->p_root_nodes;
  cl_map_t      ca_by_lid_map; /* map holding all CA lids  */
  uint16_t self_lid_ho;

  OSM_LOG_ENTER(&(osm.log), osm_updn_find_root_nodes_by_min_hop);
  osm_log (&(osm.log), OSM_LOG_DEBUG,
           "osm_updn_find_root_nodes_by_min_hop: "
           "current number of ports in the subnet is %d\n",
           cl_qmap_count(&osm.subn.port_guid_tbl));
  /* Init the required vars */
  cl_qmap_init( &min_hop_hist );
  cl_map_construct( &ca_by_lid_map );
  cl_map_init( &ca_by_lid_map, 10 );

  /* EZ:
     p_ca_list = (cl_list_t*)cl_malloc(sizeof(cl_list_t)); 
     cl_list_construct( p_ca_list ); 
     cl_list_init( p_ca_list, 10 );
  */

  /* Find the Maximum number of Cas for `histogram normalization */
  osm_log (&(osm.log), OSM_LOG_VERBOSE,
           "osm_updn_find_root_nodes_by_min_hop: "
           "Find the number of CA and store them in cl_list\n");
  p_next_port = (osm_port_t*)cl_qmap_head( &osm.subn.port_guid_tbl );
  while( p_next_port != (osm_port_t*)cl_qmap_end( &osm.subn.port_guid_tbl ) ) {
    p_port = p_next_port;
    p_next_port = (osm_port_t*)cl_qmap_next( &p_next_port->map_item );
    if ( osm_node_get_type(p_port->p_node) == IB_NODE_TYPE_CA )
    {
      p_physp = osm_port_get_default_phys_ptr(p_port);
      self_lid_ho = cl_ntoh16( osm_physp_get_base_lid(p_physp) );
      numCas++;
      /* EZ:
         self = cl_malloc(sizeof(uint16_t));
         *self = self_lid_ho;
         cl_list_insert_tail(p_ca_list, self);
      */
      cl_map_insert( &ca_by_lid_map, self_lid_ho, (void *)0x1);
      osm_log (&(osm.log), OSM_LOG_DEBUG,
               "osm_updn_find_root_nodes_by_min_hop: "
               "Inserting into array GUID 0x%" PRIx64 ", Lid: 0x%x\n",
               cl_ntoh64(osm_port_get_guid(p_port)), self_lid_ho);
    }
  }
  osm_log (&(osm.log), OSM_LOG_DEBUG,
           "osm_updn_find_root_nodes_by_min_hop: "
           "Found %u CA, %u SW in the subnet\n", numCas, numSws);
  p_next_sw = (osm_switch_t*)cl_qmap_head( &osm.subn.sw_guid_tbl );
  osm_log (&(osm.log), OSM_LOG_VERBOSE,
           "osm_updn_find_root_nodes_by_min_hop: "
           "Passing through all switches to collect Min Hop info\n");
  while( p_next_sw != (osm_switch_t*)cl_qmap_end( &osm.subn.sw_guid_tbl ) )
  {
    uint16_t max_lid_ho, lid_ho;
    uint8_t hop_val;
    uint16_t numHopBarsOverThd1 = 0;
    uint16_t numHopBarsOverThd2 = 0;
    double thd1, thd2;

    p_sw = p_next_sw;
    /* Roll to the next switch */
    p_next_sw = (osm_switch_t*)cl_qmap_next( &p_sw->map_item );

    /* Clear Min Hop Table && FWD Tbls - This should caused opensm to
       rebuild its FWD tables , post setting Min Hop Tables */
    max_lid_ho = osm_switch_get_max_lid_ho(p_sw);
    /* Get base lid of switch by retrieving port 0 lid of node pointer */
    self_lid_ho = cl_ntoh16( osm_node_get_base_lid( p_sw->p_node, 0 ) );
    osm_log (&(osm.log), OSM_LOG_DEBUG,
             "osm_updn_find_root_nodes_by_min_hop: "
             "Passing through switch lid 0x%x\n", self_lid_ho);
    for (lid_ho = 1; lid_ho <= max_lid_ho; lid_ho++)
    {
      /* Skip lids which are not CAs - 
         for the histogram purposes we care only about CAs */
      
      /* EZ:
         boolean_t LidFound = FALSE;
         cl_list_iterator_t ca_lid_iterator= cl_list_head(p_ca_list);
         while( (ca_lid_iterator != cl_list_end(p_ca_list)) && !LidFound )
         {
         uint16_t *p_lid;
         
         p_lid = (uint16_t*)cl_list_obj(ca_lid_iterator);
         if ( *p_lid == lid_ho )
         LidFound = TRUE;
         ca_lid_iterator = cl_list_next(ca_lid_iterator);
         
         }
         if ( LidFound )
      */
      if (cl_map_get( &ca_by_lid_map, lid_ho ))
      {
        hop_val = osm_switch_get_least_hops( p_sw, lid_ho );
        if (hop_val > maxHops)
          maxHops = hop_val;
        p_updn_hist = 
          (updn_hist_t*)cl_qmap_get( &min_hop_hist , (uint64_t)hop_val );
        if ( p_updn_hist == (updn_hist_t*)cl_qmap_end( &min_hop_hist))
        {
          /* New entry in the histogram , first create it */
          p_updn_hist = (updn_hist_t*) cl_malloc(sizeof(updn_hist_t));
          CL_ASSERT (p_updn_hist);
          p_updn_hist->bar_value = 1;
          cl_qmap_insert(&min_hop_hist, (uint64_t)hop_val, &p_updn_hist->map_item);
          osm_log (&(osm.log), OSM_LOG_DEBUG,
                   "osm_updn_find_root_nodes_by_min_hop: "
                   "Creating new entry in histogram %u with bar value 1\n",
                   hop_val);
        }
        else
        {
          /* Entry exist in the table , just increment the value */
          p_updn_hist->bar_value++;
          osm_log (&(osm.log), OSM_LOG_DEBUG,
                   "osm_updn_find_root_nodes_by_min_hop: "
                   "Updating entry in histogram %u with bar value %d\n", hop_val,
                   p_updn_hist->bar_value);
        }
      }
    }

    /* Now recognize the spines by requiring one bar to be above 90% of the
       number of CAs */
    thd1 = numCas * 0.9;
    thd2 = numCas * 0.05;
    osm_log (&(osm.log), OSM_LOG_DEBUG,
             "osm_updn_find_root_nodes_by_min_hop: "
             "Pass over the histogram value and find only one root node above "
             "thd1 = %f && thd2 = %f\n", thd1, thd2);

    p_updn_hist = (updn_hist_t*) cl_qmap_head( &min_hop_hist );
    while( p_updn_hist != (updn_hist_t*)cl_qmap_end( &min_hop_hist ) )
    {
      p_up_ht = p_updn_hist;
      p_updn_hist = (updn_hist_t*)cl_qmap_next( &p_updn_hist->map_item ) ;
      if ( p_up_ht->bar_value > thd1 )
        numHopBarsOverThd1++;
      if ( p_up_ht->bar_value > thd2 )
        numHopBarsOverThd2++;
      osm_log (&(osm.log), OSM_LOG_DEBUG,
               "osm_updn_find_root_nodes_by_min_hop: "
               "Passing through histogram - Hop Index %u: "
               "numHopBarsOverThd1 = %u, numHopBarsOverThd2 = %u\n",
               (uint16_t)cl_qmap_key((cl_map_item_t*)p_up_ht), numHopBarsOverThd1 ,
               numHopBarsOverThd2);
    }

    /* destroy the qmap table and all its content - no longer needed */
    osm_log (&(osm.log), OSM_LOG_DEBUG,
             "osm_updn_find_root_nodes_by_min_hop: "
             "Cleanup: delete histogram "
             "UPDN - Root nodes fetching by auto detect\n");
    p_updn_hist = (updn_hist_t*) cl_qmap_head( &min_hop_hist );
    while ( p_updn_hist != (updn_hist_t*)cl_qmap_end( &min_hop_hist ) )
    {
      cl_qmap_remove_item( &min_hop_hist, (cl_map_item_t*)p_updn_hist );
      cl_free( p_updn_hist );
      p_updn_hist = (updn_hist_t*) cl_qmap_head( &min_hop_hist );
    }

    /* If thd conditions are valid insert the root node to the list */
    if ( (numHopBarsOverThd1 == 1) && (numHopBarsOverThd2 == 1) )
    {
      p_guid = cl_malloc(sizeof(uint64_t));
      *p_guid = cl_ntoh64(osm_node_get_node_guid(p_sw->p_node));
      osm_log (&(osm.log), OSM_LOG_DEBUG,
               "osm_updn_find_root_nodes_by_min_hop: "
               "Inserting GUID 0x%" PRIx64 " as root node\n",
               *p_guid);
      cl_list_insert_tail(p_root_nodes_list, p_guid);
    }
  }

  /* destroy the map of CA lids */
  cl_map_remove_all( &ca_by_lid_map );
  cl_map_destroy( &ca_by_lid_map );

  /* Now convert the cl_list to array */
  __osm_updn_convert_list2array(p_updn);
 
  OSM_LOG_EXIT(&(osm.log));
  return 0;
}
