/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_sa_sminfo_record.c 6540 2006-04-20 16:55:11Z halr $
 */


/*
 * Abstract:
 *    Implementation of osm_smir_rcv_t.
 * This object represents the SMInfo Receiver object.
 * This object is part of the opensm family of objects.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.7 $
 */

/*
  Next available error code: 0x403
*/

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <iba/ib_types.h>
#include <complib/cl_memory.h>
#include <complib/cl_qmap.h>
#include <complib/cl_passivelock.h>
#include <complib/cl_debug.h>
#include <complib/cl_qlist.h>
#include <opensm/osm_sa_sminfo_record.h>
#include <opensm/osm_sa_response.h>
#include <opensm/osm_madw.h>
#include <opensm/osm_log.h>
#include <opensm/osm_subnet.h>
#include <opensm/osm_mad_pool.h>
#include <vendor/osm_vendor.h>
#include <vendor/osm_vendor_api.h>
#include <opensm/osm_helper.h>
#include <opensm/osm_msgdef.h>
#include <opensm/osm_port.h>
#include <opensm/osm_pkey.h>

/**********************************************************************
 **********************************************************************/
void
osm_smir_rcv_construct(
  IN osm_smir_rcv_t* const p_rcv )
{
  cl_memclr( p_rcv, sizeof(*p_rcv) );
}

/**********************************************************************
 **********************************************************************/
void
osm_smir_rcv_destroy(
  IN osm_smir_rcv_t* const p_rcv )
{
  CL_ASSERT( p_rcv );

  OSM_LOG_ENTER( p_rcv->p_log, osm_smir_rcv_destroy );

  OSM_LOG_EXIT( p_rcv->p_log );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_smir_rcv_init(
  IN osm_smir_rcv_t* const p_rcv,
  IN osm_sa_resp_t* const p_resp,
  IN osm_mad_pool_t* const p_mad_pool,
  IN osm_subn_t* const p_subn,
  IN osm_stats_t* const p_stats,
  IN osm_log_t* const p_log,
  IN cl_plock_t* const p_lock )
{
  ib_api_status_t status = IB_SUCCESS;

  OSM_LOG_ENTER( p_log, osm_smir_rcv_init );

  osm_smir_rcv_construct( p_rcv );

  p_rcv->p_log = p_log;
  p_rcv->p_subn = p_subn;
  p_rcv->p_lock = p_lock;
  p_rcv->p_resp = p_resp;
  p_rcv->p_stats = p_stats;
  p_rcv->p_mad_pool = p_mad_pool;

  OSM_LOG_EXIT( p_rcv->p_log );
  return( status );
}

/**********************************************************************
 **********************************************************************/
void
osm_smir_rcv_process(
  IN osm_smir_rcv_t*       const p_rcv,
  IN const osm_madw_t*     const p_madw )
{
  const ib_sminfo_record_t*   p_sminfo_rec;
  ib_sminfo_record_t*         p_resp_sminfo_rec;
  const ib_sa_mad_t*       p_sa_mad;
  ib_sa_mad_t*          p_resp_sa_mad;
  osm_madw_t*              p_resp_madw;
  ib_api_status_t          status;
  osm_physp_t*                p_req_physp;
  ib_net64_t                  local_guid;
  osm_port_t*                 local_port;

  CL_ASSERT( p_rcv );

  OSM_LOG_ENTER( p_rcv->p_log, osm_smir_rcv_process );

  CL_ASSERT( p_madw );

  p_sa_mad = osm_madw_get_sa_mad_ptr( p_madw );
  p_sminfo_rec = (ib_sminfo_record_t*)ib_sa_mad_get_payload_ptr( p_sa_mad );

  /* update the requester physical port. */
  p_req_physp = osm_get_physp_by_mad_addr(p_rcv->p_log,
                                          p_rcv->p_subn,
                                          osm_madw_get_mad_addr_ptr(p_madw) );
  if (p_req_physp == NULL)
  {
    osm_log( p_rcv->p_log, OSM_LOG_ERROR,
             "osm_smir_rcv_process: ERR 2803: "
             "Cannot find requester physical port\n" );
    goto Exit;
  }

  CL_ASSERT( p_sa_mad->attr_id == IB_MAD_ATTR_SMINFO_RECORD );

  if ( (p_sa_mad->method != IB_MAD_METHOD_GET) &&
       (p_sa_mad->method != IB_MAD_METHOD_GETTABLE) )
  {
    osm_log( p_rcv->p_log, OSM_LOG_ERROR,
             "osm_smir_rcv_process: ERR 2804: "
             "Unsupported Method (%s)\n",
             ib_get_sa_method_str( p_sa_mad->method ) );
    osm_sa_send_error( p_rcv->p_resp, p_madw, IB_SA_MAD_STATUS_REQ_INVALID);
    goto Exit;
  }

  /*  
   *  Get a MAD to reply. Address of Mad is in the received mad_wrapper
   */
  p_resp_madw = osm_mad_pool_get(p_rcv->p_mad_pool,
                                 p_madw->h_bind,
                                 sizeof(ib_sminfo_record_t)+IB_SA_MAD_HDR_SIZE,
                                 &p_madw->mad_addr );
  if( !p_resp_madw )
  {
    osm_log(p_rcv->p_log, OSM_LOG_ERROR,
            "osm_smir_rcv_process: ERR 2801: "
            "Unable to acquire response MAD\n" );
    goto Exit;
  }

  p_resp_sa_mad = osm_madw_get_sa_mad_ptr( p_resp_madw );
  p_resp_sminfo_rec =
    (ib_sminfo_record_t*)ib_sa_mad_get_payload_ptr( p_resp_sa_mad );

  p_resp_sminfo_rec->resv0 = 0;

  /* HACK: This handling is incorrect. The record to return is
     records of known SMs by our SM, and not just the details of
     our SM. */
  /* check the matching of pkeys with the local physp the SM is on. */
  local_guid = p_rcv->p_subn->sm_port_guid;
  local_port = (osm_port_t*)cl_qmap_get( &p_rcv->p_subn->port_guid_tbl, local_guid );
  if (FALSE == 
      osm_physp_share_pkey( p_rcv->p_log, p_req_physp,
                            osm_port_get_default_phys_ptr(local_port) ))
  {
    osm_log(p_rcv->p_log, OSM_LOG_ERROR,
            "osm_smir_rcv_process: ERR 2802: "
            "Cannot get sminfo record - pkey violation\n" );
    goto Exit;
  }

  cl_plock_acquire( p_rcv->p_lock );

  /*  get our local sm_base_lid to send in the sminfo */
  p_resp_sminfo_rec->lid = p_rcv->p_subn->sm_base_lid;
  p_resp_sminfo_rec->sm_info.guid = p_rcv->p_subn->sm_port_guid;
  p_resp_sminfo_rec->sm_info.sm_key = p_rcv->p_subn->opt.sm_key;
  p_resp_sminfo_rec->sm_info.act_count =
    cl_ntoh32(p_rcv->p_stats->qp0_mads_sent);
  p_resp_sminfo_rec->sm_info.pri_state = p_rcv->p_subn->sm_state;

  cl_plock_release( p_rcv->p_lock );

  p_sa_mad = osm_madw_get_sa_mad_ptr( p_madw );

  /*  Copy the MAD header back into the response mad */
  cl_memcpy( p_resp_sa_mad, p_sa_mad, IB_SA_MAD_HDR_SIZE );
  p_resp_sa_mad->method = (uint8_t)(p_resp_sa_mad->method | 0x80);
  /* Fill in the offset (paylen will be done by the rmpp SAR) */
  p_resp_sa_mad->attr_offset =
    ib_get_attr_offset( sizeof(ib_sminfo_record_t) );

  /* C15-0.1.5 - always return SM_Key = 0 (table 185 p 884) */
  p_resp_sa_mad->sm_key = 0;

  status = osm_vendor_send( p_resp_madw->h_bind, p_resp_madw, FALSE );
  if( status != IB_SUCCESS )
  {
    osm_log(p_rcv->p_log, OSM_LOG_ERROR,
            "osm_smir_rcv_process: ERR 2802: "
            "Error sending MAD (%s)\n",
            ib_get_err_str( status ) );
    goto Exit;
  }

 Exit:
  OSM_LOG_EXIT( p_rcv->p_log );
}
