/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2006 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_pkey_mgr.c 6505 2006-04-17 23:16:40Z halr $
 */


/*
 * Abstract:
 *    Implementation of osm_pkey_mgr_t.
 * This object represents the P_Key Manager object.
 * This object is part of the opensm family of objects.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.7 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <iba/ib_types.h>
#include <complib/cl_qmap.h>
#include <complib/cl_debug.h>
#include <opensm/osm_node.h>
#include <opensm/osm_pkey_mgr.h>

/**********************************************************************
 **********************************************************************/
void
osm_pkey_mgr_construct(
   IN osm_pkey_mgr_t * const p_mgr )
{
   CL_ASSERT( p_mgr );
   cl_memclr( p_mgr, sizeof( *p_mgr ) );
}

/**********************************************************************
 **********************************************************************/
void
osm_pkey_mgr_destroy(
   IN osm_pkey_mgr_t * const p_mgr )
{
   CL_ASSERT( p_mgr );

   OSM_LOG_ENTER( p_mgr->p_log, osm_pkey_mgr_destroy );

   OSM_LOG_EXIT( p_mgr->p_log );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_pkey_mgr_init(
   IN osm_pkey_mgr_t * const p_mgr,
   IN osm_subn_t * const p_subn,
   IN osm_log_t * const p_log,
   IN osm_req_t * const p_req,
   IN cl_plock_t * const p_lock )
{
   ib_api_status_t status = IB_SUCCESS;

   OSM_LOG_ENTER( p_log, osm_pkey_mgr_init );

   osm_pkey_mgr_construct( p_mgr );

   p_mgr->p_log = p_log;
   p_mgr->p_subn = p_subn;
   p_mgr->p_lock = p_lock;
   p_mgr->p_req = p_req;

   OSM_LOG_EXIT( p_mgr->p_log );
   return ( status );
}

/**********************************************************************
 **********************************************************************/
boolean_t
__osm_pkey_mgr_process_physical_port(
   IN const osm_pkey_mgr_t * const p_mgr,
   IN osm_node_t * p_node,
   IN uint8_t port_num,
   IN osm_physp_t * p_physp )
{
   boolean_t return_val = FALSE; /* TRUE if IB_DEFAULT_PKEY was inserted */
   osm_madw_context_t context;
   ib_pkey_table_t *block = NULL;
   uint16_t block_index;
   uint16_t num_of_blocks;
   const osm_pkey_tbl_t *p_pkey_tbl;
   uint32_t attr_mod;
   uint32_t i;
   ib_net16_t pkey;
   ib_api_status_t status;
   boolean_t block_with_empty_entry_found;

   OSM_LOG_ENTER( p_mgr->p_log, __osm_pkey_mgr_process_physical_port );

   /*
    * Send a new entry for the pkey table for this node that includes
    * IB_DEFAULT_PKEY when IB_DEFAULT_PARTIAL_PKEY or IB_DEFAULT_PKEY 
    * don't exist 
    */
   if ( ( osm_physp_has_pkey( p_mgr->p_log,
                              IB_DEFAULT_PKEY,
                              p_physp ) == FALSE ) &&
        ( osm_physp_has_pkey( p_mgr->p_log,
                              IB_DEFAULT_PARTIAL_PKEY, p_physp ) == FALSE ) )
   {
      context.pkey_context.node_guid = osm_node_get_node_guid( p_node );
      context.pkey_context.port_guid = osm_physp_get_port_guid( p_physp );
      context.pkey_context.set_method = TRUE;

      p_pkey_tbl = osm_physp_get_pkey_tbl( p_physp );
      num_of_blocks = osm_pkey_tbl_get_num_blocks( p_pkey_tbl );
      block_with_empty_entry_found = FALSE;

      for ( block_index = 0; block_index < num_of_blocks; block_index++ )
      {
         block = osm_pkey_tbl_block_get( p_pkey_tbl, block_index );
         for ( i = 0; i < IB_NUM_PKEY_ELEMENTS_IN_BLOCK; i++ )
         {
            pkey = block->pkey_entry[i];
            if ( ib_pkey_is_invalid( pkey ) )
            {
               block->pkey_entry[i] = IB_DEFAULT_PKEY;
               block_with_empty_entry_found = TRUE;
               break;
            }
         }
         if ( block_with_empty_entry_found )
         {
            break;
         }
      }

      if ( block_with_empty_entry_found == FALSE )
      {
         osm_log( p_mgr->p_log, OSM_LOG_ERROR,
                  "__osm_pkey_mgr_process_physical_port: ERR 0501: "
                  "No empty entry was found to insert IB_DEFAULT_PKEY for node "
                  "0x%016" PRIx64 " and port %u\n",
                  cl_ntoh64( osm_node_get_node_guid( p_node ) ), port_num );
      }
      else
      {
         /* Building the attribute modifier */
         if ( osm_node_get_type( p_node ) == IB_NODE_TYPE_SWITCH )
         {
            /* Port num | Block Index */
            attr_mod = port_num << 16 | block_index;
         }
         else
         {
            attr_mod = block_index;
         }

         status = osm_req_set( p_mgr->p_req,
                               osm_physp_get_dr_path_ptr( p_physp ),
                               ( uint8_t * ) block,
                               sizeof( *block ),
                               IB_MAD_ATTR_P_KEY_TABLE,
                               cl_hton32( attr_mod ),
                               CL_DISP_MSGID_NONE, &context );
         return_val = TRUE;     /*IB_DEFAULT_PKEY was inserted */

         if ( osm_log_is_active( p_mgr->p_log, OSM_LOG_VERBOSE ) )
         {
            osm_log( p_mgr->p_log, OSM_LOG_VERBOSE,
                     "__osm_pkey_mgr_process_physical_port:  "
                     "IB_DEFAULT_PKEY was inserted for node 0x%016" PRIx64
                     " and port %u\n",
                     cl_ntoh64( osm_node_get_node_guid( p_node ) ),
                     port_num );
         }
      }
   }
   else 
   {
      /* default key or partial default key already exist */
      if ( osm_log_is_active( p_mgr->p_log, OSM_LOG_VERBOSE ) )
      {
        osm_log( p_mgr->p_log, OSM_LOG_VERBOSE,
                 "__osm_pkey_mgr_process_physical_port:  "
                 "No need to insert IB_DEFAULT_PKEY for node 0x%016" PRIx64
                 " port %u\n",
                 cl_ntoh64( osm_node_get_node_guid( p_node ) ), port_num );
      }
   }

   OSM_LOG_EXIT( p_mgr->p_log );
   return ( return_val );
}

/**********************************************************************
 **********************************************************************/
osm_signal_t
osm_pkey_mgr_process(
   IN const osm_pkey_mgr_t * const p_mgr )
{
   cl_qmap_t *p_node_guid_tbl;
   osm_node_t *p_node;
   osm_node_t *p_next_node;
   uint8_t port_num;
   osm_physp_t *p_physp;
   osm_signal_t signal = OSM_SIGNAL_DONE;

   CL_ASSERT( p_mgr );

   OSM_LOG_ENTER( p_mgr->p_log, osm_pkey_mgr_process );

   p_node_guid_tbl = &p_mgr->p_subn->node_guid_tbl;

   CL_PLOCK_EXCL_ACQUIRE( p_mgr->p_lock );

   p_next_node = ( osm_node_t * ) cl_qmap_head( p_node_guid_tbl );
   while ( p_next_node != ( osm_node_t * ) cl_qmap_end( p_node_guid_tbl ) )
   {
      p_node = p_next_node;
      p_next_node = ( osm_node_t * ) cl_qmap_next( &p_next_node->map_item );

      for ( port_num = 0; port_num < osm_node_get_num_physp( p_node );
            port_num++ )
      {
         p_physp = osm_node_get_physp_ptr( p_node, port_num );
         if ( osm_physp_is_valid( p_physp ) )
         {
            if ( __osm_pkey_mgr_process_physical_port
                 ( p_mgr, p_node, port_num, p_physp ) )
            {
               if ( osm_log_is_active( p_mgr->p_log, OSM_LOG_VERBOSE ) )
               {
                  osm_log( p_mgr->p_log, OSM_LOG_VERBOSE,
                           "osm_pkey_mgr_process:  "
                           "Adding IB_DEFAULT_PKEY for pkey table of node "
                           "0x%016" PRIx64 " port %u\n",
                           cl_ntoh64( osm_node_get_node_guid( p_node ) ),
                           port_num );
               }
               signal = OSM_SIGNAL_DONE_PENDING;
            }
         }
      }
   }

   CL_PLOCK_RELEASE( p_mgr->p_lock );
   OSM_LOG_EXIT( p_mgr->p_log );
   return ( signal );
}
