/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2006 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_opensm.c 5677 2006-03-08 12:38:34Z halr $
 */


/*
 * Abstract:
 *    Implementation of osm_opensm_t.
 * This object represents the opensm super object.
 * This object is part of the opensm family of objects.
 *
 * Environment:
 *    Linux User Mode
 *
 * $Revision: 1.7 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <complib/cl_memory.h>
#include <complib/cl_dispatcher.h>
#include <complib/cl_passivelock.h>
#include <vendor/osm_vendor_api.h>
#include <opensm/osm_version.h>
#ifdef OSM_VENDOR_INTF_OPENIB
#include <opensm/osm_svn_revision.h>
#endif
#include <opensm/osm_base.h>
#include <opensm/osm_opensm.h>
#include <opensm/osm_log.h>
#include <opensm/osm_subnet.h>
#include <opensm/osm_sm.h>
#include <opensm/osm_vl15intf.h>

/**********************************************************************
 **********************************************************************/
void
osm_opensm_construct(
   IN osm_opensm_t * const p_osm )
{
   cl_memclr( p_osm, sizeof( *p_osm ) );
   osm_subn_construct( &p_osm->subn );
   osm_sm_construct( &p_osm->sm );
   osm_sa_construct( &p_osm->sa );
   osm_db_construct( &p_osm->db );
   osm_mad_pool_construct( &p_osm->mad_pool );
   osm_vl15_construct( &p_osm->vl15 );
   osm_log_construct( &p_osm->log );
}

/**********************************************************************
 **********************************************************************/
void
osm_opensm_destroy(
   IN osm_opensm_t * const p_osm )
{

   /* in case of shutdown through exit proc - no ^C */
   osm_exit_flag = TRUE;

   /* 
    * First of all - Clear the is_sm bit.
    */
   if( p_osm->sm.mad_ctrl.h_bind )
      osm_vendor_set_sm( p_osm->sm.mad_ctrl.h_bind, FALSE );

   /* shut down the SA
    * - unbind from QP1 messages
    */
   osm_sa_shutdown( &p_osm->sa );

   /* shut down the SM 
    * - make sure the SM sweeper thread exited
    * - unbind from QP0 messages
    */
   osm_sm_shutdown( &p_osm->sm );

   /* cleanup all messages on VL15 fifo that were not sent yet */
   osm_vl15_shutdown( &p_osm->vl15, &p_osm->mad_pool );

   /* shut down the dispatcher - so no new messages cross */
   cl_disp_shutdown( &p_osm->disp );

   /* do the destruction in reverse order as init */
   updn_destroy( p_osm->p_updn_ucast_routing );
   osm_sa_destroy( &p_osm->sa );
   osm_sm_destroy( &p_osm->sm );
   osm_db_destroy( &p_osm->db );
   osm_vl15_destroy( &p_osm->vl15, &p_osm->mad_pool );
   osm_mad_pool_destroy( &p_osm->mad_pool );
   osm_vendor_delete( &p_osm->p_vendor );
   osm_subn_destroy( &p_osm->subn );
   cl_disp_destroy( &p_osm->disp );

   cl_plock_destroy( &p_osm->lock );

   cl_mem_display(  );

   osm_log_destroy( &p_osm->log );
}

/**********************************************************************
 **********************************************************************/
static void
osm_opensm_create_mcgroups(
   IN osm_opensm_t * const p_osm,
   IN const osm_subn_opt_t * const p_opt )
{
   OSM_LOG_ENTER( &p_osm->log, osm_opensm_create_mcgroups );
   osm_sa_create_template_record_ipoib( &p_osm->sa, p_opt );
   OSM_LOG_EXIT( &p_osm->log );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_opensm_init(
   IN osm_opensm_t * const p_osm,
   IN const osm_subn_opt_t * const p_opt )
{
   ib_api_status_t status;

   /* Can't use log macros here, since we're initializing the log. */
   osm_opensm_construct( p_osm );

   status = osm_log_init( &p_osm->log, p_opt->force_log_flush,
                          p_opt->log_flags, p_opt->log_file, p_opt->accum_log_file );
   if( status != IB_SUCCESS )
      return ( status );

#ifndef OSM_VENDOR_INTF_OPENIB
   /* If there is a log level defined - add the OSM_VERSION to it. */
   osm_log( &p_osm->log,
            osm_log_get_level( &p_osm->log ) & ( OSM_LOG_SYS ^ 0xFF ), "%s\n",
            OSM_VERSION );
   /* Write the OSM_VERSION to the SYS_LOG */
   osm_log( &p_osm->log, OSM_LOG_SYS, "%s\n", OSM_VERSION );   /* Format Waived */
#else
   if (strlen(OSM_SVN_REVISION))
   {
      /* If there is a log level defined - add OSM_VERSION and OSM_SVN_REVISION to it. */
      osm_log( &p_osm->log,
               osm_log_get_level( &p_osm->log ) & ( OSM_LOG_SYS ^ 0xFF ), "%s OpenIB svn %s\n",
               OSM_VERSION, OSM_SVN_REVISION );
      /* Write the OSM_VERSION and OSM_SVN_REVISION to the SYS_LOG */
      osm_log( &p_osm->log, OSM_LOG_SYS, "%s OpenIB svn %s\n", OSM_VERSION, OSM_SVN_REVISION );   /* Format Waived */
   }
   else
   {
      /* If there is a log level defined - add the OSM_VERSION to it. */
      osm_log( &p_osm->log,
               osm_log_get_level( &p_osm->log ) & ( OSM_LOG_SYS ^ 0xFF ), "%s\n",
               OSM_VERSION );
      /* Write the OSM_VERSION to the SYS_LOG */
      osm_log( &p_osm->log, OSM_LOG_SYS, "%s\n", OSM_VERSION );   /* Format Waived */
   }
#endif

   osm_log( &p_osm->log, OSM_LOG_FUNCS, "osm_opensm_init: [\n" ); /* Format Waived */

   status = cl_plock_init( &p_osm->lock );
   if( status != IB_SUCCESS )
      goto Exit;

   if( p_opt->single_thread )
   {
      osm_log( &p_osm->log, OSM_LOG_INFO,
               "osm_opensm_init: Forcing single threaded dispatcher.\n" );
      status = cl_disp_init( &p_osm->disp, 1, "opensm" );
   }
   else
   {
      /*
       * Normal behavior is to initialize the dispatcher with
       * one thread per CPU, as specified by a thread count of '0'.
       */
      status = cl_disp_init( &p_osm->disp, 0, "opensm" );
   }
   if( status != IB_SUCCESS )
      goto Exit;

   status = osm_subn_init( &p_osm->subn, p_opt );
   if( status != IB_SUCCESS )
      goto Exit;

   p_osm->p_vendor =
      osm_vendor_new( &p_osm->log, p_opt->transaction_timeout );
   if( p_osm->p_vendor == NULL )
   {
      status = IB_INSUFFICIENT_RESOURCES;
      goto Exit;
   }

   status = osm_mad_pool_init( &p_osm->mad_pool, &p_osm->log );
   if( status != IB_SUCCESS )
      goto Exit;

   status = osm_vl15_init( &p_osm->vl15,
                           p_osm->p_vendor,
                           &p_osm->log, &p_osm->stats, p_opt->max_wire_smps,
                           &p_osm->subn, &p_osm->disp, &p_osm->lock );
   if( status != IB_SUCCESS )
      goto Exit;

   /* the DB is in use by the SM and SA so init before */
   status = osm_db_init( &p_osm->db, &p_osm->log );
   if( status != IB_SUCCESS )
      goto Exit;

   status = osm_sm_init( &p_osm->sm,
                         &p_osm->subn,
                         &p_osm->db,
                         p_osm->p_vendor,
                         &p_osm->mad_pool,
                         &p_osm->vl15,
                         &p_osm->log,
                         &p_osm->stats, &p_osm->disp, &p_osm->lock );

   if( status != IB_SUCCESS )
      goto Exit;

   status = osm_sa_init( &p_osm->sm,
                         &p_osm->sa,
                         &p_osm->subn,
                         p_osm->p_vendor,
                         &p_osm->mad_pool,
                         &p_osm->log,
                         &p_osm->stats, &p_osm->disp, &p_osm->lock );

   if( status != IB_SUCCESS )
      goto Exit;

   osm_opensm_create_mcgroups( p_osm, p_opt );

   /* HACK - the UpDown manager should have been a part of the osm_sm_t */
   /* Init updn struct */
   p_osm->p_updn_ucast_routing = updn_construct(  );
   status = updn_init( p_osm->p_updn_ucast_routing );
   if( status != IB_SUCCESS )
      goto Exit;

 Exit:
   osm_log( &p_osm->log, OSM_LOG_FUNCS, "osm_opensm_init: ]\n" ); /* Format Waived */
   return ( status );
}

/**********************************************************************
 **********************************************************************/
ib_api_status_t
osm_opensm_bind(
   IN osm_opensm_t * const p_osm,
   IN const ib_net64_t guid )
{
   ib_api_status_t status;

   OSM_LOG_ENTER( &p_osm->log, osm_opensm_bind );

   status = osm_sm_bind( &p_osm->sm, guid );
   if( status != IB_SUCCESS )
      goto Exit;

   status = osm_sa_bind( &p_osm->sa, guid );
   if( status != IB_SUCCESS )
      goto Exit;
 Exit:
   OSM_LOG_EXIT( &p_osm->log );
   return ( status );
}
