/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_pi_config_ctrl.h 1743 2005-02-06 09:38:35Z shaharf $
 */


/*
 * Abstract:
 * 	Declaration of osm_pi_conf_ctrl_t.
 *	This object represents a controller that performs a 
 *	Set(PortInfo) for the specified port.
 *	This object is part of the OpenSM family of objects.
 *
 * Environment:
 * 	Linux User Mode
 *
 * $Revision: 1.4 $
 */

#ifndef _OSM_PI_CONFIG_CTRL_H_
#define _OSM_PI_CONFIG_CTRL_H_

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif /* __cplusplus */

BEGIN_C_DECLS

/****h* OpenSM/Port Info Configuration Controller
* NAME
*	Port Info Configuration Controller
*
* DESCRIPTION
*	The Port Info Configuration Controller object encapsulates the
*	information needed to Set(PortInfo) at the specified port.
*
*	The Port Info Configuration Controller object is thread safe.
*
*	This object should be treated as opaque and should be
*	manipulated only through the provided functions.
*
* AUTHOR
*	Steve King, Intel
*
*********/
/****s* OpenSM: Port Info Configuration Controller/osm_pi_conf_ctrl_t
* NAME
*	osm_pi_conf_ctrl_t
*
* DESCRIPTION
*	Port Info Configuration Controller structure.
*
*	This object should be treated as opaque and should
*	be manipulated only through the provided functions.
*
* SYNOPSIS
*/
typedef struct _osm_pi_conf_ctrl
{
	osm_subn_t					*p_subn;
	osm_log_t					*p_log;
	cl_plock_t					*p_lock;
	cl_dispatcher_t				*p_disp;

} osm_pi_conf_ctrl_t;
/*
* FIELDS
*	p_subn
*		Pointer to the Subnet object for this subnet.
*
*	p_log
*		Pointer to the log object.
*
*	p_lock
*		Pointer to the serializing lock.
*
*	p_disp
*		Pointer to the Dispatcher.
*
* SEE ALSO
*	Port Info Configuration Controller object
*********/
/****f* OpenSM: Port Info Configuration Controller/osm_pi_conf_ctrl_construct
* NAME
*	osm_pi_conf_ctrl_construct
*
* DESCRIPTION
*	This function constructs a Port Info Configuration Controller object.
*
* SYNOPSIS
*/
void osm_pi_conf_ctrl_construct(
	IN osm_pi_conf_ctrl_t* const p_ctrl );
/*
* PARAMETERS
*	p_ctrl
*		[in] Pointer to a Port Info Configuration Controller object to construct.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Allows calling osm_pi_conf_ctrl_init, osm_pi_conf_ctrl_destroy,
*	and osm_pi_conf_ctrl_is_inited.
*
*	Calling osm_pi_conf_ctrl_construct is a prerequisite to calling any other
*	method except osm_pi_conf_ctrl_init.
*
* SEE ALSO
*	Port Info Configuration Controller object, osm_pi_conf_ctrl_init,
*	osm_pi_conf_ctrl_destroy, osm_pi_conf_ctrl_is_inited
*********/

/****f* OpenSM: Port Info Configuration Controller/osm_pi_conf_ctrl_destroy
* NAME
*	osm_pi_conf_ctrl_destroy
*
* DESCRIPTION
*	The osm_pi_conf_ctrl_destroy function destroys the object, releasing
*	all resources.
*
* SYNOPSIS
*/
void osm_pi_conf_ctrl_destroy(
	IN osm_pi_conf_ctrl_t* const p_ctrl );
/*
* PARAMETERS
*	p_ctrl
*		[in] Pointer to the object to destroy.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Performs any necessary cleanup of the specified
*	Port Info Configuration Controller object.
*	Further operations should not be attempted on the destroyed object.
*	This function should only be called after a call to
*	osm_pi_conf_ctrl_construct or osm_pi_conf_ctrl_init.
*
* SEE ALSO
*	Port Info Configuration Controller object, osm_pi_conf_ctrl_construct,
*	osm_pi_conf_ctrl_init
*********/

/****f* OpenSM: Port Info Configuration Controller/osm_pi_conf_ctrl_init
* NAME
*	osm_pi_conf_ctrl_init
*
* DESCRIPTION
*	The osm_pi_conf_ctrl_init function initializes a
*	Port Info Configuration Controller object for use.
*
* SYNOPSIS
*/
ib_api_status_t osm_pi_conf_ctrl_init(
	IN osm_pi_conf_ctrl_t* const p_ctrl,
	IN osm_subn_t* const p_subn,
	IN osm_log_t* const p_log,
	IN cl_plock_t* const p_lock,
	IN cl_dispatcher_t* const p_disp );
/*
* PARAMETERS
*	p_ctrl
*		[in] Pointer to an osm_pi_conf_ctrl_t object to initialize.
*
*	p_subn
*		[in] Pointer to the Subnet object for this subnet.
*
*	p_log
*		[in] Pointer to the log object.
*
*	p_lock
*		[in] Pointer to the OpenSM serializing lock.
*
*	p_disp
*		[in] Pointer to the OpenSM central Dispatcher.
*
* RETURN VALUES
*	CL_SUCCESS if the Port Info Configuration Controller object was initialized
*	successfully.
*
* NOTES
*	Allows calling other Port Info Configuration Controller methods.
*
* SEE ALSO
*	Port Info Configuration Controller object, osm_pi_conf_ctrl_construct,
*	osm_pi_conf_ctrl_destroy, osm_pi_conf_ctrl_is_inited
*********/

/****f* OpenSM: Port Info Configuration Controller/osm_pi_conf_ctrl_is_inited
* NAME
*	osm_pi_conf_ctrl_is_inited
*
* DESCRIPTION
*	Indicates if the object has been initialized with osm_pi_conf_ctrl_init.
*
* SYNOPSIS
*/
boolean_t osm_pi_conf_ctrl_is_inited(
	IN const osm_pi_conf_ctrl_t* const p_ctrl );
/*
* PARAMETERS
*	p_ctrl
*		[in] Pointer to an osm_pi_conf_ctrl_t object.
*
* RETURN VALUES
*	TRUE if the object was initialized successfully,
*	FALSE otherwise.
*
* NOTES
*	The osm_pi_conf_ctrl_construct or osm_pi_conf_ctrl_init must be
*	called before using	this function.
*
* SEE ALSO
*	Port Info Configuration Controller object, osm_pi_conf_ctrl_construct,
*	osm_pi_conf_ctrl_init
*********/

/****f* OpenSM: Port Info Configuration Controller/osm_pi_conf_ctrl_process
* NAME
*	osm_pi_conf_ctrl_process
*
* DESCRIPTION
*	Initiate a PortInfo configuration.
*
* SYNOPSIS
*/
ib_api_status_t osm_pi_conf_ctrl_process(
	IN const osm_pi_conf_ctrl_t* const p_ctrl,
	IN const ib_guid_t guid );
/*
* PARAMETERS
*	p_ctrl
*		[in] Pointer to an osm_pi_conf_ctrl_t object.
*
*	guid
*		[in] GUID of port to configure.
*
* RETURN VALUES
*	CL_SUCCESS if configuration processing was successfully
*	initiated.
*
* NOTES
*	A success status here does not indicate that
*	the PortInfo configuration process completed successfully.
*
* SEE ALSO
*	Port Info Configuration Controller
*********/
	
END_C_DECLS

#endif /* _OSM_PI_CONFIG_CTRL_H_ */
