/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_partition.h 1743 2005-02-06 09:38:35Z shaharf $
 */


/*
 * Abstract:
 * 	Declaration of osm_prtn_t.
 *	This object represents an IBA Partition.
 *	This object is part of the OpenSM family of objects.
 *
 * Environment:
 * 	Linux User Mode
 *
 * $Revision: 1.4 $
 */

#ifndef _OSM_PARTITION_H_
#define _OSM_PARTITION_H_

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif /* __cplusplus */

BEGIN_C_DECLS

/****h* OpenSM/Partition
* NAME
*	Partition
*
* DESCRIPTION
*	The Partition object encapsulates the information needed by the
*	OpenSM to manage Partitions.  The OpenSM allocates one Partition
*	object per Partition in the IBA subnet.
*
*	The Partition is not thread safe, thus callers must provide
*	serialization.
*
*	This object should be treated as opaque and should be
*	manipulated only through the provided functions.
*
* AUTHOR
*	Steve King, Intel
*
*********/

/****s* OpenSM: Partition/osm_prtn_t
* NAME
*	osm_prtn_t
*
* DESCRIPTION
*	Partition structure.
*
*	The osm_prtn_t object should be treated as opaque and should
*	be manipulated only through the provided functions.
*
* SYNOPSIS
*/
typedef struct _osm_prtn
{
	uint16_t						pkey;
	cl_map							port_guid_tbl;

} osm_prtn_t;
/*
* FIELDS
*	pkey
*		The IBA defined P_KEY of this Partition.
*
*	port_guid_tbl
*		Container of pointers to all Port objects in the Partition,
*		indexed by port GUID.
*
* SEE ALSO
*	Partition
*********/

/****f* OpenSM: Partition/osm_prtn_construct
* NAME
*	osm_prtn_construct
*
* DESCRIPTION
*	This function constructs a Partition.
*
* SYNOPSIS
*/
void osm_prtn_construct(
	IN osm_prtn_t* const p_prtn );
/*
* PARAMETERS
*	p_prtn
*		[in] Pointer to a Partition to construct.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Allows calling osm_prtn_init, osm_prtn_destroy, and osm_prtn_is_inited.
*
*	Calling osm_prtn_construct is a prerequisite to calling any other
*	method except osm_prtn_init.
*
* SEE ALSO
*	Partition, osm_prtn_init, osm_prtn_destroy, osm_prtn_is_inited
*********/

/****f* OpenSM: Partition/osm_prtn_destroy
* NAME
*	osm_prtn_destroy
*
* DESCRIPTION
*	The osm_prtn_destroy function destroys a Partition, releasing
*	all resources.
*
* SYNOPSIS
*/
void osm_prtn_destroy(
	IN osm_prtn_t* const p_prtn );
/*
* PARAMETERS
*	p_prtn
*		[in] Pointer to a Partition to destroy.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Performs any necessary cleanup of the specified Partition.
*	Further operations should not be attempted on the destroyed object.
*	This function should only be called after a call to osm_prtn_construct or
*	osm_prtn_init.
*
* SEE ALSO
*	Partition, osm_prtn_construct, osm_prtn_init
*********/

/****f* OpenSM: Partition/osm_prtn_init
* NAME
*	osm_prtn_init
*
* DESCRIPTION
*	The osm_prtn_init function initializes a Partition for use.
*
* SYNOPSIS
*/
ib_api_status_t osm_prtn_init(
	IN osm_prtn_t* const p_prtn );
/*
* PARAMETERS
*	p_prtn
*		[in] Pointer to an osm_prtn_t object to initialize.
*
* RETURN VALUES
*	CL_SUCCESS if initialization was successful.
*
* NOTES
*	Allows calling other Partition methods.
*
* SEE ALSO
*	Partition, osm_prtn_construct, osm_prtn_destroy,
*	osm_prtn_is_inited
*********/

/****f* OpenSM: Partition/osm_prtn_is_inited
* NAME
*	osm_prtn_is_inited
*
* DESCRIPTION
*	Indicates if the object has been initialized with osm_prtn_init.
*
* SYNOPSIS
*/
boolean_t osm_ptrn_is_inited(
	IN const osm_prtn_t* const p_prtn );
/*
* PARAMETERS
*	p_prtn
*		[in] Pointer to an osm_prtn_t object.
*
* RETURN VALUES
*	TRUE if the object was initialized successfully,
*	FALSE otherwise.
*
* NOTES
*	The osm_prtn_construct or osm_prtn_init must be called before using
*	this function.
*
* SEE ALSO
*	Partition, osm_prtn_construct, osm_prtn_init
*********/

/****f* OpenSM: Partition/osm_prtn_is_guid
* NAME
*	osm_prtn_is_guid
*
* DESCRIPTION
*	Indicates if a port is a member of the partition.
*
* SYNOPSIS
*/
boolean_t osm_prtn_is_guid(
	IN const osm_prtn_t* const p_prtn,
	IN const uint64 guid );
/*
* PARAMETERS
*	p_prtn
*		[in] Pointer to an osm_prtn_t object.
*
*	guid
*		[in] Port GUID.
*
* RETURN VALUES
*	TRUE if the specified port GUID is a member of the partition,
*	FALSE otherwise.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Partition/osm_prtn_get_pkey
* NAME
*	osm_prtn_get_pkey
*
* DESCRIPTION
*	Gets the IBA defined P_KEY value for this Partition.
*
* SYNOPSIS
*/
uint16_t osm_prtn_get_pkey(
	IN const osm_prtn_t* const p_prtn );
/*
* PARAMETERS
*	p_prtn
*		[in] Pointer to an osm_prtn_t object.
*
* RETURN VALUES
*	P_KEY value for this Partition.
*
* NOTES
*
* SEE ALSO
*********/

END_C_DECLS

#endif /* _OSM_PARTITION_H_ */
