/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_mcm_info.h 1743 2005-02-06 09:38:35Z shaharf $
 */


/*
 * Abstract:
 * 	Declaration of osm_mcm_info_t.
 *	This object represents a Multicast Forwarding Information object.
 *	This object is part of the OpenSM family of objects.
 *
 * Environment:
 * 	Linux User Mode
 *
 * $Revision: 1.4 $
 */

#ifndef _OSM_MCM_INFO_H_
#define _OSM_MCM_INFO_H_

#include <iba/ib_types.h>
#include <complib/cl_qlist.h>
#include <opensm/osm_base.h>

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif /* __cplusplus */

BEGIN_C_DECLS

/****s* OpenSM: Multicast Member Info/osm_mcm_info_t
* NAME
*	osm_mcm_info_t
*
* DESCRIPTION
*	Multicast Membership Info object.
*	This object contains information about a nodes membership
*	in a particular multicast group.
*
*	This object should be treated as opaque and should
*	be manipulated only through the provided functions.
*
* SYNOPSIS
*/
typedef struct _osm_mcm_info
{
	cl_list_item_t				list_item;
	ib_net16_t					mlid;

} osm_mcm_info_t;
/*
* FIELDS
*	list_item
*		Linkage structure for cl_qlist.  MUST BE FIRST MEMBER!
*
*	mlid
*		MLID of this multicast group.
*
* SEE ALSO
*********/

/****f* OpenSM: Multicast Member Info/osm_mcm_info_construct
* NAME
*	osm_mcm_info_construct
*
* DESCRIPTION
*	This function constructs a Multicast Member Info object.
*
* SYNOPSIS
*/
static inline void
osm_mcm_info_construct(
	IN osm_mcm_info_t* const p_mcm )
{
	cl_memclr( p_mcm, sizeof(*p_mcm) );
}
/*
* PARAMETERS
*	p_mcm
*		[in] Pointer to a Multicast Member Info object to construct.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Multicast Member Info/osm_mcm_info_destroy
* NAME
*	osm_mcm_info_destroy
*
* DESCRIPTION
*	The osm_mcm_info_destroy function destroys the object, releasing
*	all resources.
*
* SYNOPSIS
*/
void
osm_mcm_info_destroy(
	IN osm_mcm_info_t* const p_mcm );
/*
* PARAMETERS
*	p_mcm
*		[in] Pointer to a Multicast Member Info object to destroy.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Performs any necessary cleanup of the specified Multicast Member Info object.
*	Further operations should not be attempted on the destroyed object.
*	This function should only be called after a call to osm_mtree_construct or
*	osm_mtree_init.
*
* SEE ALSO
*	Multicast Member Info object, osm_mtree_construct, osm_mtree_init
*********/

/****f* OpenSM: Multicast Member Info/osm_mcm_info_init
* NAME
*	osm_mcm_info_init
*
* DESCRIPTION
*	Initializes a Multicast Member Info object for use.
*
* SYNOPSIS
*/
void
osm_mcm_info_init(
	IN osm_mcm_info_t* const p_mcm,
	IN const ib_net16_t mlid );
/*
* PARAMETERS
*	p_mcm
*		[in] Pointer to an osm_mcm_info_t object to initialize.
*
*	mlid
*		[in] MLID value for this multicast group.
*
* RETURN VALUES
*	None.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Multicast Member Info/osm_mcm_info_new
* NAME
*	osm_mcm_info_new
*
* DESCRIPTION
*	Returns an initialized a Multicast Member Info object for use.
*
* SYNOPSIS
*/
osm_mcm_info_t*
osm_mcm_info_new(
  IN const ib_net16_t mlid );
/*
* PARAMETERS
*	mlid
*		[in] MLID value for this multicast group.
*
* RETURN VALUES
*	Pointer to an initialized tree node.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Multicast Member Info/osm_mcm_info_delete
* NAME
*	osm_mcm_info_delete
*
* DESCRIPTION
*	Destroys and deallocates the specified object.
*
* SYNOPSIS
*/
void
osm_mcm_info_delete(
	IN osm_mcm_info_t* const p_mcm );
/*
* PARAMETERS
*	p_mcm
*		Pointer to the object to destroy.
*
* RETURN VALUES
*	None.
*
* NOTES
*
* SEE ALSO
*********/

END_C_DECLS

#endif		/* _OSM_MCM_INFO_H_ */
