/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_matrix.h 7623 2006-06-02 10:42:30Z halr $
 */


/*
 * Abstract:
 * 	Declaration of osm_lid_matrix_t.
 *	This object represents a two dimensional array of port numbers
 *	and LID values.
 *	This object is part of the OpenSM family of objects.
 *
 * Environment:
 * 	Linux User Mode
 *
 * $Revision: 1.5 $
 */


#ifndef _OSM_MATRIX_H_
#define _OSM_MATRIX_H_

#include <iba/ib_types.h>
#include <complib/cl_vector.h>
#include <opensm/osm_base.h>

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif /* __cplusplus */

BEGIN_C_DECLS

/****h* OpenSM/LID Matrix
* NAME
*	LID Matrix
*
* DESCRIPTION
*	The LID Matrix object encapsulates the information needed by the
*	OpenSM to manage fabric routes.  It is a two dimensional array
*	index by LID value and Port Number.  Each element contains the
*	number of hops from that Port Number to the LID.
*	Every Switch object contains a LID Matrix.
*
*	The LID Matrix is not thread safe, thus callers must provide
*	serialization.
*
*	This object should be treated as opaque and should be
*	manipulated only through the provided functions.
*
* AUTHOR
*	Steve King, Intel
*
*********/

/****s* OpenSM: LID Matrix/osm_lid_matrix_t
* NAME
*	osm_lid_matrix_t
*
* DESCRIPTION
*
*	The LID Matrix object encapsulates the information needed by the
*	OpenSM to manage fabric routes.  It is a two dimensional array
*	index by LID value and Port Number.  Each element contains the
*	number of hops from that Port Number to the LID.
*	Every Switch object contains a LID Matrix.
*
*	The LID Matrix is not thread safe, thus callers must provide
*	serialization.
*
*	The num_ports index into the matrix serves a special purpose, in that it
*	contains the shortest hop path for that LID through any port.
*
*	This object should be treated as opaque and should be
*	manipulated only through the provided functions.
*
* SYNOPSIS
*/
typedef struct _osm_lid_matrix_t
{
	cl_vector_t			lid_vec;
	uint8_t				num_ports;

} osm_lid_matrix_t;
/*
* FIELDS
*	lid_vec
*		Vector (indexed by LID) of port arrays (indexed by port number)
*
*	num_ports
*		Number of ports at each entry in the LID vector.
*
* SEE ALSO
*********/

/****f* OpenSM: LID Matrix/osm_lid_matrix_construct
* NAME
*	osm_lid_matrix_construct
*
* DESCRIPTION
*	This function constructs a LID Matrix object.
*
* SYNOPSIS
*/
static inline void
osm_lid_matrix_construct(
	IN osm_lid_matrix_t* const p_lmx )
{
	p_lmx->num_ports = 0;
	cl_vector_construct( &p_lmx->lid_vec );
}
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to a LID Matrix object to construct.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Allows calling osm_lid_matrix_init, osm_lid_matrix_destroy
*
*	Calling osm_lid_matrix_construct is a prerequisite to calling any other
*	method except osm_lid_matrix_init.
*
* SEE ALSO
*	LID Matrix object, osm_lid_matrix_init, osm_lid_matrix_destroy
*********/

/****f* OpenSM: LID Matrix/osm_lid_matrix_destroy
* NAME
*	osm_lid_matrix_destroy
*
* DESCRIPTION
*	The osm_lid_matrix_destroy function destroys a node, releasing
*	all resources.
*
* SYNOPSIS
*/
void osm_lid_matrix_destroy(
	IN osm_lid_matrix_t* const p_lmx );
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to a LID Matrix object to destroy.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Performs any necessary cleanup of the specified LID Matrix object.
*	Further operations should not be attempted on the destroyed object.
*	This function should only be called after a call to osm_lid_matrix_construct or
*	osm_lid_matrix_init.
*
* SEE ALSO
*	LID Matrix object, osm_lid_matrix_construct, osm_lid_matrix_init
*********/
/****f* OpenSM: LID Matrix/osm_lid_matrix_init
* NAME
*	osm_lid_matrix_init
*
* DESCRIPTION
*	Initializes a LID Matrix object for use.
*
* SYNOPSIS
*/
ib_api_status_t
osm_lid_matrix_init(
	IN osm_lid_matrix_t* const p_lmx,
	IN const uint8_t num_ports );
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to an osm_lid_matrix_t object to initialize.
*
*	num_ports
*		[in] Number of ports at each LID index.  This value is fixed
*		at initialization time.
*
* RETURN VALUES
*	IB_SUCCESS on success
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: LID Matrix/osm_lid_matrix_get
* NAME
*	osm_lid_matrix_get
*
* DESCRIPTION
*	Returns the hop count at the specified LID/Port intersection.
*
* SYNOPSIS
*/
static inline uint8_t
osm_lid_matrix_get(
	IN const osm_lid_matrix_t* const p_lmx,
	IN const uint16_t lid_ho,
	IN const uint8_t port_num )
{
	CL_ASSERT( port_num < p_lmx->num_ports );
	CL_ASSERT( lid_ho <cl_vector_get_size(&p_lmx->lid_vec) );
	return( ((uint8_t *)cl_vector_get_ptr(
			&p_lmx->lid_vec, lid_ho ))[port_num] );
}
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to an osm_lid_matrix_t object.
*
*	lid_ho
*		[in] LID value (host order) for which to return the hop count
*
*	port_num
*		[in] Port number in the switch
*
* RETURN VALUES
*	Returns the hop count at the specified LID/Port intersection.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: LID Matrix/osm_lid_matrix_get_max_lid_ho
* NAME
*	osm_lid_matrix_get_max_lid_ho
*
* DESCRIPTION
*	Returns the maximum LID (host order) value contained
*	in the matrix.
*
* SYNOPSIS
*/
static inline uint16_t
osm_lid_matrix_get_max_lid_ho(
	IN const osm_lid_matrix_t* const p_lmx )
{
	return( (uint16_t)(cl_vector_get_size( &p_lmx->lid_vec ) - 1 ) );
}
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to an osm_lid_matrix_t object.
*
* RETURN VALUES
*	Returns the maximum LID (host order) value contained
*	in the matrix.
*
* NOTES
*
* SEE ALSO
*********/
/****f* OpenSM: LID Matrix/osm_lid_matrix_get_num_ports
* NAME
*	osm_lid_matrix_get_num_ports
*
* DESCRIPTION
*	Returns the number of ports in this lid matrix.
*
* SYNOPSIS
*/
static inline uint8_t
osm_lid_matrix_get_num_ports(
	IN const osm_lid_matrix_t* const p_lmx )
{
	return( p_lmx->num_ports );
}
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to an osm_lid_matrix_t object.
*
* RETURN VALUES
*	Returns the number of ports in this lid matrix.
*
* NOTES
*
* SEE ALSO
*********/
/****f* OpenSM: LID Matrix/osm_lid_matrix_get_least_hops
* NAME
*	osm_lid_matrix_get_least_hops
*
* DESCRIPTION
*	Returns the least number of hops for specified lid
*
* SYNOPSIS
*/
static inline uint8_t
osm_lid_matrix_get_least_hops(
	IN const osm_lid_matrix_t* const p_lmx,
	IN const uint16_t lid_ho )
{
	if( lid_ho > osm_lid_matrix_get_max_lid_ho( p_lmx ) )
		return( OSM_NO_PATH );

	return( ((uint8_t *)cl_vector_get_ptr(
			&p_lmx->lid_vec, lid_ho ))[p_lmx->num_ports] );
}
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to an osm_lid_matrix_t object.
*
*	lid_ho
*		[in] LID (host order) for which to retrieve the shortest hop count.
*
* RETURN VALUES
*	Returns the least number of hops for specified lid
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: LID Matrix/osm_lid_matrix_set
* NAME
*	osm_lid_matrix_set
*
* DESCRIPTION
*	Sets the hop count at the specified LID/Port intersection.
*
* SYNOPSIS
*/
cl_status_t
osm_lid_matrix_set(
	IN osm_lid_matrix_t* const p_lmx,
	IN const uint16_t lid_ho,
	IN const uint8_t port_num,
	IN const uint8_t val );
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to an osm_lid_matrix_t object.
*
*	lid_ho
*		[in] LID value (host order) to index into the vector.
*
*	port_num
*		[in] port number index into the vector entry.
*
*	val
*		[in] value (number of hops) to assign to this entry.
*
* RETURN VALUES
*	Returns the hop count at the specified LID/Port intersection.
*
* NOTES
*
* SEE ALSO
*********/
/****f* OpenSM: LID Matrix/osm_lid_matrix_set_min_lid_size
* NAME
*	osm_lid_matrix_set_min_lid_size
*
* DESCRIPTION
*	Sets the size of the matrix to at least accomodate the
*	specified LID value (host ordered)
*
* SYNOPSIS
*/
static inline cl_status_t
osm_lid_matrix_set_min_lid_size(
	IN osm_lid_matrix_t* const p_lmx,
	IN const uint16_t lid_ho )
{
	return( cl_vector_set_min_size( &p_lmx->lid_vec, lid_ho + 1 ) );
}
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to an osm_lid_matrix_t object.
*
*	lid_ho
*		[in] Minimum LID value (host order) to accomodate.
*
* RETURN VALUES
*	Sets the size of the matrix to at least accomodate the
*	specified LID value (host ordered)
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: LID Matrix/osm_lid_matrix_clear
* NAME
*	osm_lid_matrix_clear
*
* DESCRIPTION
*	Clears a LID Matrix object in anticipation of a rebuild.
*
* SYNOPSIS
*/
void
osm_lid_matrix_clear(
	IN osm_lid_matrix_t* const p_lmx );
/*
* PARAMETERS
*	p_lmx
*		[in] Pointer to an osm_lid_matrix_t object to clear.
*
* RETURN VALUES
*	None.
*
* NOTES
*
* SEE ALSO
*********/

END_C_DECLS

#endif		/* _OSM_MATRIX_H_ */
