/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_switch.h 3798 2005-10-17 21:45:20Z halr $
 */


/*
 * Abstract:
 * 	Declaration of osm_switch_t.
 *	This object represents an IBA switch.
 *	This object is part of the OpenSM family of objects.
 *
 * Environment:
 * 	Linux User Mode
 *
 * $Revision: 1.6 $
 */

#ifndef _OSM_SWITCH_H_
#define _OSM_SWITCH_H_

#include <iba/ib_types.h>
#include <opensm/osm_base.h>
#include <opensm/osm_madw.h>
#include <opensm/osm_node.h>
#include <opensm/osm_port.h>
#include <opensm/osm_matrix.h>
#include <opensm/osm_fwd_tbl.h>
#include <opensm/osm_mcast_tbl.h>
#include <opensm/osm_port_profile.h>

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif /* __cplusplus */

BEGIN_C_DECLS

/****h* OpenSM/Switch
* NAME
*	Switch
*
* DESCRIPTION
*	The Switch object encapsulates the information needed by the
*	OpenSM to manage switches.  The OpenSM allocates one switch object
*	per switch in the IBA subnet.
*
*	The Switch object is not thread safe, thus callers must provide
*	serialization.
*
*	This object should be treated as opaque and should be
*	manipulated only through the provided functions.
*
* AUTHOR
*	Steve King, Intel
*
*********/

/****h* OpenSM/Switch
* NAME
*	Switch
*
* DESCRIPTION
*	The Switch object encapsulates the information needed by the
*	OpenSM to manage switches.  The OpenSM allocates one switch object
*	per switch in the IBA subnet.
*
*	The Switch object is not thread safe, thus callers must provide
*	serialization.
*
*	This object should be treated as opaque and should be
*	manipulated only through the provided functions.
*
* AUTHOR
*	Steve King, Intel
*
*********/
/****s* OpenSM: Switch/osm_switch_t
* NAME
*	osm_switch_t
*
* DESCRIPTION
*	Switch structure.
*
*	This object should be treated as opaque and should
*	be manipulated only through the provided functions.
*
* SYNOPSIS
*/
typedef struct _osm_switch
{
	cl_map_item_t				map_item;
	osm_node_t					*p_node;
	ib_switch_info_t			switch_info;
	osm_fwd_tbl_t				fwd_tbl;
	osm_lid_matrix_t			lmx;
	osm_port_profile_t			*p_prof;
	osm_mcast_tbl_t				mcast_tbl;
	uint32_t					discovery_count;

} osm_switch_t;
/*
* FIELDS
*	map_item
*		Linkage structure for cl_qmap.  MUST BE FIRST MEMBER!
*
*	p_node
*		Pointer to the Node object for this switch.
*
*	switch_info
*		IBA defined SwitchInfo structure for this switch.
*
*	fwd_tbl
*		This switch's forwarding table.
*
*	lmx
*		LID Matrix for this switch containing the hop count
*		to every LID from every port.
*
*	p_pro
*		Pointer to array of Port Profile objects for this switch.
*
*	mcast_tbl
*		Multicast forwarding table for this switch.
*
*	discovery_count
*		The number of times this switch has been discovered
*		during the current fabric sweep.  This number is reset
*		to zero at the start of a sweep.
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_construct
* NAME
*	osm_switch_construct
*
* DESCRIPTION
*	This function constructs a Switch object.
*
* SYNOPSIS
*/
void
osm_switch_construct(
	IN osm_switch_t* const p_sw );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to a Switch object to construct.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Allows calling osm_switch_init, osm_switch_destroy, and osm_switch_is_inited.
*
*	Calling osm_switch_construct is a prerequisite to calling any other
*	method except osm_switch_init.
*
* SEE ALSO
*	Switch object, osm_switch_init, osm_switch_destroy, osm_switch_is_inited
*********/

/****f* OpenSM: Switch/osm_switch_destroy
* NAME
*	osm_switch_destroy
*
* DESCRIPTION
*	The osm_switch_destroy function destroys the object, releasing
*	all resources.
*
* SYNOPSIS
*/
void
osm_switch_destroy(
	IN osm_switch_t* const p_sw );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the object to destroy.
*
* RETURN VALUE
*	None.
*
* NOTES
*	Performs any necessary cleanup of the specified object.
*	Further operations should not be attempted on the destroyed object.
*	This function should only be called after a call to osm_switch_construct
*	or osm_switch_init.
*
* SEE ALSO
*	Switch object, osm_switch_construct, osm_switch_init
*********/

/****f* OpenSM: Switch/osm_switch_destroy
* NAME
*	osm_switch_destroy
*
* DESCRIPTION
*	Destroys and deallocates the object.
*
* SYNOPSIS
*/
void
osm_switch_delete(
	IN OUT osm_switch_t** const pp_sw );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the object to destroy.
*
* RETURN VALUE
*	None.
*
* NOTES
*
* SEE ALSO
*	Switch object, osm_switch_construct, osm_switch_init
*********/


/****f* OpenSM: Switch/osm_switch_init
* NAME
*	osm_switch_init
*
* DESCRIPTION
*	The osm_switch_init function initializes a Switch object for use.
*
* SYNOPSIS
*/
ib_api_status_t
osm_switch_init(
	IN osm_switch_t* const p_sw,
	IN osm_node_t* const p_node,
	IN const osm_madw_t* const p_madw );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object to initialize.
*
*	p_node
*		[in] Pointer to the node object of this switch
*
*	p_madw
*		[in] Pointer to the MAD Wrapper containing the switch's
*		SwitchInfo attribute.
*
* RETURN VALUES
*	IB_SUCCESS if the Switch object was initialized successfully.
*
* NOTES
*	Allows calling other node methods.
*
* SEE ALSO
*	Switch object, osm_switch_construct, osm_switch_destroy,
*	osm_switch_is_inited
*********/
/****f* OpenSM: Switch/osm_switch_new
* NAME
*	osm_switch_new
*
* DESCRIPTION
*	The osm_switch_init function initializes a Switch object for use.
*
* SYNOPSIS
*/
osm_switch_t*
osm_switch_new(
	IN osm_node_t* const p_node,
	IN const osm_madw_t* const p_madw );
/*
* PARAMETERS
*	p_node
*		[in] Pointer to the node object of this switch
*
*	p_madw
*		[in] Pointer to the MAD Wrapper containing the switch's
*		SwitchInfo attribute.
*
* RETURN VALUES
*	Pointer to the new initialized switch object.
*
* NOTES
*
* SEE ALSO
*	Switch object, osm_switch_construct, osm_switch_destroy,
*********/

/****f* OpenSM: Switch/osm_switch_is_leaf_lid
* NAME
*	osm_switch_is_leaf_lid
*
* DESCRIPTION
*	Indicates if the specified LID is the switch's LID, or is a leaf
*	of the switch.
*
* SYNOPSIS
*/
static inline boolean_t
osm_switch_is_leaf_lid(
	IN const osm_switch_t* const p_sw,
	IN const uint16_t lid_ho )
{
	return( osm_lid_matrix_get_least_hops( &p_sw->lmx, lid_ho ) <= 1 );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
*	lid_ho
*		[in] LID (host order) to compare.
*
* RETURN VALUES
*	TRUE if the LID is the switch's LID or is a leaf of the switch,
*	FALSE otherwise.
*
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_hop_count
* NAME
*	osm_switch_get_hop_count
*
* DESCRIPTION
*	Returns the hop count at the specified LID/Port intersection.
*
* SYNOPSIS
*/
static inline uint8_t
osm_switch_get_hop_count(
	IN const osm_switch_t* const p_sw,
	IN const uint16_t lid_ho,
	IN const uint8_t port_num )
{
	return( osm_lid_matrix_get( &p_sw->lmx, lid_ho, port_num ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to a Switch object.
*
*	lid_ho
*		[in] LID value (host order) for which to return the hop count
*
*	port_num
*		[in] Port number in the switch
*
* RETURN VALUES
*	Returns the hop count at the specified LID/Port intersection.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_get_fwd_tbl_ptr
* NAME
*	osm_switch_get_fwd_tbl_ptr
*
* DESCRIPTION
*	Returns a pointer to the switch's forwarding table.
*
* SYNOPSIS
*/
static inline osm_fwd_tbl_t*
osm_switch_get_fwd_tbl_ptr(
	IN const osm_switch_t* const p_sw )
{
	return( (osm_fwd_tbl_t*)&p_sw->fwd_tbl );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to a Switch object.
*
* RETURN VALUES
*	Returns a pointer to the switch's forwarding table.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_set_hops
* NAME
*	osm_switch_set_hops
*
* DESCRIPTION
*	Sets the hop count at the specified LID/Port intersection.
*
* SYNOPSIS
*/
static inline cl_status_t
osm_switch_set_hops(
	IN osm_switch_t* const p_sw,
	IN const uint16_t lid_ho,
	IN const uint8_t port_num,
	IN const uint8_t num_hops )
{
	return( osm_lid_matrix_set( &p_sw->lmx, lid_ho, port_num, num_hops ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to a Switch object.
*
*	lid_ho
*		[in] LID value (host order) for which to set the count.
*
*	port_num
*		[in] port number for which to set the count.
*
*	num_hops
*		[in] value to assign to this entry.
*
* RETURN VALUES
*	Returns the hop count at the specified LID/Port intersection.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_set_hops
* NAME
*	osm_switch_set_hops
*
* DESCRIPTION
*	Sets the size of the switch's routing table to at least accomodate the
*	specified LID value (host ordered)
*
* SYNOPSIS
*/
static inline cl_status_t
osm_switch_set_min_lid_size(
	IN osm_switch_t* const p_sw,
	IN const uint16_t lid_ho )
{
	return( osm_lid_matrix_set_min_lid_size( &p_sw->lmx, lid_ho ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to a Switch object.
*
*	lid_ho
*		[in] LID value (host order) for which to set the count.
*
* RETURN VALUES
*	Sets the size of the switch's routing table to at least accomodate the
*	specified LID value (host ordered)
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_get_least_hops
* NAME
*	osm_switch_get_least_hops
*
* DESCRIPTION
*	Returns the number of hops in the short path to this lid from
*	any port on the switch.
*
* SYNOPSIS
*/
static inline uint8_t
osm_switch_get_least_hops(
	IN const osm_switch_t* const p_sw,
	IN const uint16_t lid_ho )
{
	return( osm_lid_matrix_get_least_hops( &p_sw->lmx, lid_ho ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
*	lid_ho
*		[in] LID (host order) for which to retrieve the shortest hop count.
*
* RETURN VALUES
*	Returns the number of hops in the short path to this lid from
*	any port on the switch.
*
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_port_by_lid
* NAME
*	osm_switch_get_port_by_lid
*
* DESCRIPTION
*	Returns the switch port number on which the specified LID is routed.
*
* SYNOPSIS
*/
static inline uint8_t
osm_switch_get_port_by_lid(
	IN const osm_switch_t* const p_sw,
	IN const uint16_t lid_ho )
{
	return( osm_fwd_tbl_get( &p_sw->fwd_tbl, lid_ho ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
*	lid_ho
*		[in] LID (host order) for which to retrieve the shortest hop count.
*
* RETURN VALUES
*	Returns the switch port on which the specified LID is routed.
*
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_lid
* NAME
*	osm_switch_get_lid
*
* DESCRIPTION
*	Gets the switch's LID.
*
* SYNOPSIS
*/
ib_net16_t
osm_switch_get_lid(
	IN const osm_switch_t* const p_sw );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	Returns the switch's LID.  A value of zero means no LID has
*	been assigned to the switch.
*
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_physp_ptr
* NAME
*	osm_switch_get_physp_ptr
*
* DESCRIPTION
*	Gets the Physical Port Object at the specified port number.
*
* SYNOPSIS
*/
osm_physp_t*
osm_switch_get_physp_ptr(
	IN const osm_switch_t* const p_sw,
	IN const uint32_t port_num );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
*	port_num
*		[in] Port number for which to retrieve the Physical Port Object.
*
* RETURN VALUES
*	Returns a pointer to the Physical Port Object object at the specified
*	port number.
*	A return value of zero means the port number was out of range.
*	
*
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_route_by_lid
* NAME
*	osm_switch_get_route_by_lid
*
* DESCRIPTION
*	Gets the physical port object that routes the specified LID.
*
* SYNOPSIS
*/
static inline osm_physp_t*
osm_switch_get_route_by_lid(
  IN const osm_switch_t* const p_sw,
  IN const ib_net16_t lid )
{
  uint8_t port_num;
  CL_ASSERT( p_sw );
  CL_ASSERT( lid );

  port_num = osm_fwd_tbl_get( &p_sw->fwd_tbl, cl_ntoh16( lid ) );
  /* 
     In order to avoid holes in the subnet (usually happens when
     running UPDN alogrithm ), i.e. cases where port is
     unreachable through a switch (we put a OSM_NO_PATH value at 
     the port entry , we do not assert on unreachable lids entries
     at the fwd table but return NULL 
  */
  if (port_num != OSM_NO_PATH)
    return( osm_node_get_physp_ptr( p_sw->p_node, port_num ) );
  else
    return NULL;
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
*	lid
*		[in] LID for which to find a route.  This must be a unicast
*		LID value < 0xC000.
*
* RETURN VALUES
*	Returns a pointer to the Physical Port Object object that
*	routes the specified LID.  A return value of zero means
*	there is no route for the lid through this switch.
*	The lid value must be a unicast LID.
*	
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_si_ptr
* NAME
*	osm_switch_get_si_ptr
*
* DESCRIPTION
*	Returns a pointer to the SwitchInfo for this switch.
*
* SYNOPSIS
*/
static inline ib_switch_info_t*
osm_switch_get_si_ptr(
	IN const osm_switch_t* const p_sw )
{
	return( (ib_switch_info_t*)&p_sw->switch_info );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	Returns a pointer to the SwitchInfo for this switch.
*	
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_max_block_id
* NAME
*	osm_switch_get_max_block_id
*
* DESCRIPTION
*	Returns the maximum block ID (host order) of this switch.
*
* SYNOPSIS
*/
static inline uint32_t
osm_switch_get_max_block_id(
	IN const osm_switch_t* const p_sw )
{
	return( (uint32_t)(osm_fwd_tbl_get_size( &p_sw->fwd_tbl ) /
			osm_fwd_tbl_get_lids_per_block( &p_sw->fwd_tbl ) ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	Returns the maximum block ID (host order) of this switch.
*	
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_max_block_id_in_use
* NAME
*	osm_switch_get_max_block_id_in_use
*
* DESCRIPTION
*	Returns the maximum block ID (host order) of this switch that
*	is used for unicast routing.
*
* SYNOPSIS
*/
static inline uint32_t
osm_switch_get_max_block_id_in_use(
	IN const osm_switch_t* const p_sw )
{
	return( osm_fwd_tbl_get_max_block_id_in_use( &p_sw->fwd_tbl,
			cl_ntoh16( p_sw->switch_info.lin_top ) ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	Returns the maximum block ID (host order) of this switch.
*	
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_node_ptr
* NAME
*	osm_switch_get_node_ptr
*
* DESCRIPTION
*	Returns a pointer to the Node object for this switch.
*
* SYNOPSIS
*/
static inline osm_node_t*
osm_switch_get_node_ptr(
	IN const osm_switch_t* const p_sw )
{
	return( p_sw->p_node );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	Returns a pointer to the LID matrix for this switch.
*	
* NOTES
*
* SEE ALSO
*	Switch object
*********/

/****f* OpenSM: Switch/osm_switch_get_max_lid_ho
* NAME
*	osm_switch_get_max_lid_ho
*
* DESCRIPTION
*	Returns the maximum LID (host order) value contained
*	in the switch routing tables.
*
* SYNOPSIS
*/
static inline uint16_t
osm_switch_get_max_lid_ho(
	IN const osm_switch_t* const p_sw )
{
	return( osm_lid_matrix_get_max_lid_ho( &p_sw->lmx ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to a switch object.
*
* RETURN VALUES
*	Returns the maximum LID (host order) value contained
*	in the switch routing tables.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_get_num_ports
* NAME
*	osm_switch_get_num_ports
*
* DESCRIPTION
*	Returns the number of ports in this switch.
*
* SYNOPSIS
*/
static inline uint8_t
osm_switch_get_num_ports(
	IN const osm_switch_t* const p_sw )
{
	return( osm_lid_matrix_get_num_ports( &p_sw->lmx ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	Returns the number of ports in this switch.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_get_fwd_tbl_block
* NAME
*	osm_switch_get_fwd_tbl_block
*
* DESCRIPTION
*	Retrieve a forwarding table block.
*
* SYNOPSIS
*/
boolean_t
osm_switch_get_fwd_tbl_block(
	IN const osm_switch_t* const p_sw,
	IN const uint32_t block_id,
	OUT uint8_t* const p_block );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
*	block_ID
*		[in] The block_id to retrieve.
*
*	p_block
*		[out] Pointer to the 64 byte array to store the
*		forwarding table clock specified by block_id.
*
* RETURN VALUES
*	Returns true if there are more blocks necessary to 
*	configure all the LIDs reachable from this switch.
*	FALSE otherwise.
*
* NOTES
*
* SEE ALSO
*********/


/****f* OpenSM: Switch/osm_switch_supports_mcast
* NAME
*	osm_switch_supports_mcast
*
* DESCRIPTION
*	Indicates if a switch supports multicast.
*
* SYNOPSIS
*/
static inline boolean_t
osm_switch_supports_mcast(
	IN const osm_switch_t* const p_sw )
{
	return( p_sw->switch_info.mcast_cap != 0 );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	Returns TRUE if the switch supports multicast.
*	FALSE otherwise.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_set_switch_info
* NAME
*	osm_switch_set_switch_info
*
* DESCRIPTION
*	Updates the switch info attribute of this switch.
*
* SYNOPSIS
*/
static inline void
osm_switch_set_switch_info(
	IN osm_switch_t* const p_sw,
	IN const ib_switch_info_t* const p_si )
{
	CL_ASSERT( p_sw );
	CL_ASSERT( p_si );
	p_sw->switch_info = *p_si;
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to a Switch object.
*
*	p_si
*		[in] Pointer to the SwitchInfo attribute for this switch.
*
* RETURN VALUES
*	None.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_set_path
* NAME
*	osm_switch_set_path
*
* DESCRIPTION
*	Sets the port to route the specified LID.
*
* SYNOPSIS
*/
static inline void
osm_switch_set_path(
	IN osm_switch_t* const p_sw,
	IN const uint16_t lid_ho,
	IN const uint8_t port,
	IN const boolean_t ignore_port_prof
	)
{
	CL_ASSERT( p_sw );
	osm_fwd_tbl_set( &p_sw->fwd_tbl, lid_ho, port );
	if (! ignore_port_prof)
	  osm_port_prof_path_count_inc( &p_sw->p_prof[port] );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch object.
*
*	lid_ho
*		[in] LID value (host order) for which to set the route.
*
*	port
*		[in] Port to route the specified LID value.
*
* RETURN VALUE
*	None.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_set_ft_block
* NAME
*	osm_switch_set_ft_block
*
* DESCRIPTION
*	Copies in the specified block into the switch's Forwarding Table object.
*
* SYNOPSIS
*/
static inline ib_api_status_t
osm_switch_set_ft_block(
	IN osm_switch_t* const p_sw,
	IN const uint8_t* const p_block,
	IN const uint32_t block_num )
{
	CL_ASSERT( p_sw );
	return( osm_fwd_tbl_set_block( &p_sw->fwd_tbl, p_block, block_num ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch object.
*
*	p_block
*		[in] Pointer to the forwarding table block.
*
*	block_num
*		[in] Block number for this block
*
* RETURN VALUE
*	None.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_set_mft_block
* NAME
*	osm_switch_set_mft_block
*
* DESCRIPTION
*	Sets a block of multicast port masks into the multicast table.
*
* SYNOPSIS
*/
static inline ib_api_status_t
osm_switch_set_mft_block(
	IN osm_switch_t* const p_sw,
	IN const ib_net16_t* const p_block,
	IN const uint16_t block_num,
	IN const uint8_t position )
{
	CL_ASSERT( p_sw );
	return( osm_mcast_tbl_set_block( &p_sw->mcast_tbl, p_block,
			block_num, position ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch object.
*
*	p_block
*		[in] Pointer to the block of port masks to set.
*
*	block_num
*		[in] Block number (0-511) to set.
*
*	position
*		[in] Port mask position (0-15) to set.
*
* RETURN VALUE
*	IB_SUCCESS on success.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_recommend_path
* NAME
*	osm_switch_recommend_path
*
* DESCRIPTION
*	Returns the recommended port on which to route this LID.
*  In cases where LMC > 0, the remote side system and node 
*  used for the routing are tracked in the provided arrays 
*  (and counts) such that other lid for the same port will
*  try and avoid going through the same remote system/node.
*
* SYNOPSIS
*/
uint8_t
osm_switch_recommend_path(
	IN const osm_switch_t* const p_sw,
	IN const uint16_t lid_ho,
	IN const boolean_t ignore_existing,
	IN OUT uint64_t *remote_sys_guids,
	IN OUT uint16_t *p_num_used_sys,
	IN OUT uint64_t *remote_node_guids,
	IN OUT uint16_t *p_num_used_nodes,
	IN const uint32_t max_routes_subscribed,
	IN boolean_t      ui_ucast_fdb_assign_func_defined
 );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch object.
*
*	lid_ho
*		[in] LID value (host order) for which to get a path advisory.
*
*	ignore_existing
*		[in] Set to cause the switch to choose the optimal route
*		regardless of existing paths.
*		If false, the switch will choose an existing route if one exists,
*		otherwise will choose the optimal route.
*
*	remote_sys_guids
*     [in out] The array of remote system guids already used to route 
*     the other lids of the same target port (if LMC > 0).
*  
*  p_num_used_sys
*     [in out] The number of remote systems used for routing to the port.
*
*	remote_node_guids
*     [in out] The array of remote node guids already used to route 
*     the other lids of the same target port (if LMC > 0).
*  
*  p_num_used_nodes
*     [in out] The number of remote nodes used for routing to the port.
*
*  max_routes_subscribed
*     [in] The maximum allowed number of target lids routed through 
*     a specific port of the switch. If the port already assigned 
*     (in the lfdb) this number of target lids - it will not be used
*     even if it has the smallest hops count to the target lid.
*
*  ui_ucast_fdb_assign_func_defined
*     [in] If TRUE - this means that there is a ui ucast_fdb_assign table
*     function defined (in pfn_ui_ucast_fdb_assign in subnet opts). This
*     means that all current entries in the fdbs will be used.
*     If FALSE - such function isn't defined. Do the minimum hop checks 
*     before accepting the current fdbs.
*
* RETURN VALUE
*	Returns the recommended port on which to route this LID.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_recommend_mcast_path
* NAME
*	osm_switch_recommend_mcast_path
*
* DESCRIPTION
*	Returns the recommended port on which to route this LID.
*
* SYNOPSIS
*/
uint8_t
osm_switch_recommend_mcast_path(
	IN osm_switch_t* const p_sw,
	IN const uint16_t lid_ho,
	IN const uint16_t mlid_ho,
	IN const boolean_t ignore_existing );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch object.
*
*	lid_ho
*		[in] LID value (host order) for of the node for with to get
*		the multicast path.
*
*	mlid_ho
*		[in] MLID for the multicast group in question.
*
*	ignore_existing
*		[in] Set to cause the switch to choose the optimal route
*		regardless of existing paths.
*		If false, the switch will choose an existing route if one exists,
*		otherwise will choose the optimal route.
*
* RETURN VALUE
*	Returns the recommended port on which to route this LID.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_get_fwd_tbl_size
* NAME
*	osm_switch_get_fwd_tbl_size
*
* DESCRIPTION
*	Returns the number of entries available in the forwarding table.
*
* SYNOPSIS
*/
static inline uint16_t
osm_switch_get_fwd_tbl_size(
	IN const osm_switch_t* const p_sw )
{
	return( osm_fwd_tbl_get_size( &p_sw->fwd_tbl ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch.
*
* RETURN VALUE
*	Returns the number of entries available in the forwarding table.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_get_mcast_fwd_tbl_size
* NAME
*	osm_switch_get_mcast_fwd_tbl_size
*
* DESCRIPTION
*	Returns the number of entries available in the multicast forwarding table.
*
* SYNOPSIS
*/
static inline uint16_t
osm_switch_get_mcast_fwd_tbl_size(
	IN const osm_switch_t* const p_sw )
{
	return( cl_ntoh16( p_sw->switch_info.mcast_cap ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch.
*
* RETURN VALUE
*	Returns the number of entries available in the multicast forwarding table.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_path_count_get
* NAME
*	osm_switch_path_count_get
*
* DESCRIPTION
*	Returns the count of the number of paths going through this port.
*
* SYNOPSIS
*/
static inline uint32_t
osm_switch_path_count_get(
	IN const osm_switch_t* const p_sw,
	IN const uint8_t port_num )
{
	return( osm_port_prof_path_count_get( &p_sw->p_prof[port_num] ) );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the Switch object.
*
*	port_num
*		[in] Port number for which to get path count.
*
* RETURN VALUE
*	Returns the count of the number of paths going through this port.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_prepare_path_rebuild
* NAME
*	osm_switch_prepare_path_rebuild
*
* DESCRIPTION
*	Prepares a switch to rebuild pathing information.
*
* SYNOPSIS
*/
void
osm_switch_prepare_path_rebuild(
	IN osm_switch_t* const p_sw );
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the Switch object.
*
* RETURN VALUE
*	None.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_get_mcast_tbl_ptr
* NAME
*	osm_switch_get_mcast_tbl_ptr
*
* DESCRIPTION
*	Returns a pointer to the switch's multicast table.
*
* SYNOPSIS
*/
static inline osm_mcast_tbl_t*
osm_switch_get_mcast_tbl_ptr(
	IN const osm_switch_t* const p_sw )
{
	return( (osm_mcast_tbl_t*)&p_sw->mcast_tbl );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch.
*
* RETURN VALUE
*	Returns a pointer to the switch's multicast table.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Switch/osm_switch_is_in_mcast_tree
* NAME
*	osm_switch_is_in_mcast_tree
*
* DESCRIPTION
*	Returns true if this switch already belongs in the tree for the specified
*	multicast group.
*
* SYNOPSIS
*/
static inline boolean_t
osm_switch_is_in_mcast_tree(
	IN const osm_switch_t* const p_sw,
	IN const uint16_t mlid_ho )
{
	const osm_mcast_tbl_t* p_tbl;
	p_tbl = &p_sw->mcast_tbl;
	if( p_tbl )
		return( osm_mcast_tbl_is_any_port( &p_sw->mcast_tbl, mlid_ho ) );
	else
		return( FALSE );
}
/*
* PARAMETERS
*	p_sw
*		[in] Pointer to the switch.
*
*	mlid_ho
*		[in] MLID (host order) of the multicast tree to check.
*
* RETURN VALUE
*	Returns true if this switch already belongs in the tree for the specified
*	multicast group.
*
* NOTES
*
* SEE ALSO
*********/

/****f* OpenSM: Node/osm_switch_discovery_count_get
* NAME
*	osm_switch_discovery_count_get
*
* DESCRIPTION
*	Returns a pointer to the physical port object at the
*	specified local port number.
*
* SYNOPSIS
*/
static inline uint32_t
osm_switch_discovery_count_get(
	IN const osm_switch_t* const p_switch )
{
	return( p_switch->discovery_count );
}
/*
* PARAMETERS
*	p_switch
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	Returns the discovery count for this node.
*
* NOTES
*
* SEE ALSO
*	Node object
*********/
/****f* OpenSM: Node/osm_switch_discovery_count_reset
* NAME
*	osm_switch_discovery_count_reset
*
* DESCRIPTION
*	Resets the discovery count for this node to zero.
*	This operation should be performed at the start of a sweep.
*
* SYNOPSIS
*/
static inline void
osm_switch_discovery_count_reset(
	IN osm_switch_t* const p_switch )
{
	p_switch->discovery_count = 0;
}
/*
* PARAMETERS
*	p_switch
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	None.
*
* NOTES
*
* SEE ALSO
*	Node object
*********/
/****f* OpenSM: Node/osm_switch_discovery_count_inc
* NAME
*	osm_switch_discovery_count_inc
*
* DESCRIPTION
*	Increments the discovery count for this node.
*
* SYNOPSIS
*/
static inline void
osm_switch_discovery_count_inc(
	IN osm_switch_t* const p_switch )
{
	p_switch->discovery_count++;
}
/*
* PARAMETERS
*	p_switch
*		[in] Pointer to an osm_switch_t object.
*
* RETURN VALUES
*	None.
*
* NOTES
*
* SEE ALSO
*	Node object
*********/

END_C_DECLS

#endif /* _OSM_SWITCH_H_ */
