/*
 * Copyright (c) 2004-2006 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_sa.h 5206 2006-01-30 13:44:20Z halr $
 */


/*
 * Abstract:
 * 	Declaration of osm_sa_t.
 *	This object represents an IBA subnet.
 *	This object is part of the OpenSM family of objects.
 *
 * Environment:
 * 	Linux User Mode
 *
 * $Revision: 1.6 $
 */

#ifndef _OSM_SA_H_
#define _OSM_SA_H_

#include <iba/ib_types.h>
#include <complib/cl_passivelock.h>
#include <complib/cl_event.h>
#include <complib/cl_thread.h>
#include <complib/cl_timer.h>
#include <opensm/osm_stats.h>
#include <complib/cl_dispatcher.h>
#include <opensm/osm_subnet.h>
#include <vendor/osm_vendor.h>
#include <opensm/osm_mad_pool.h>
#include <opensm/osm_sa_response.h>
#include <opensm/osm_log.h>
#include <opensm/osm_sa_node_record_ctrl.h>
#include <opensm/osm_sa_portinfo_record_ctrl.h>
#include <opensm/osm_sa_guidinfo_record_ctrl.h>
#include <opensm/osm_sa_link_record_ctrl.h>
#include <opensm/osm_sa_path_record_ctrl.h>
#include <opensm/osm_sa_sminfo_record_ctrl.h>
#include <opensm/osm_sa_mad_ctrl.h>
#include <opensm/osm_sa_mcmember_record_ctrl.h>
#include <opensm/osm_sa_service_record_ctrl.h>
#include <opensm/osm_sa_class_port_info_ctrl.h>
#include <opensm/osm_sa_informinfo_ctrl.h>
#include <opensm/osm_sa_slvl_record_ctrl.h>
#include <opensm/osm_sa_vlarb_record_ctrl.h>
#include <opensm/osm_sa_pkey_record_ctrl.h>
#include <opensm/osm_sa_lft_record_ctrl.h>

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif /* __cplusplus */

BEGIN_C_DECLS

/****h* OpenSM/SA
* NAME
*	SA
*
* DESCRIPTION
*	The SA object encapsulates the information needed by the
*	OpenSM to instantiate a subnet administrator.  The OpenSM allocates
*	one SA object per subnet manager.
*
*	The SA object is thread safe.
*
*	This object should be treated as opaque and should
*	be manipulated only through the provided functions.
*
* AUTHOR
*	Ranjit Pandit, Intel
*	Anil Keshavamurthy, Intel	
*
*********/

/****d* OpenSM: SA/osm_sa_state_t
* NAME
*	osm_sa_state_t
*
* DESCRIPTION
*	Enumerates the possible states of SA object.
*
* SYNOPSIS
*/
typedef enum _osm_sa_state
{
	OSM_SA_STATE_INIT = 0,
	OSM_SA_STATE_READY

} osm_sa_state_t;
/***********/

/****s* OpenSM: SM/osm_sa_t
* NAME
*	osm_sa_t
*
* DESCRIPTION
*	Subnet Administrator structure.
*
*	This object should be treated as opaque and should
*	be manipulated only through the provided functions.
*
* SYNOPSIS
*/
typedef struct _osm_sa
{
	osm_sa_state_t				state;
	osm_subn_t					*p_subn;
	osm_vendor_t				*p_vendor;
	osm_log_t					*p_log;
	osm_mad_pool_t				*p_mad_pool;
	cl_dispatcher_t				*p_disp;
	cl_plock_t					*p_lock;
	atomic32_t					sa_trans_id;
	osm_sa_mad_ctrl_t			mad_ctrl;
	osm_sa_resp_t				resp;
	osm_cpi_rcv_t				cpi_rcv;
	osm_cpi_rcv_ctrl_t		cpi_rcv_ctrl;
	osm_nr_rcv_t				nr_rcv;
	osm_nr_rcv_ctrl_t			nr_rcv_ctrl;
	osm_pir_rcv_t				pir_rcv;
	osm_pir_rcv_ctrl_t			pir_rcv_ctrl;
	osm_gir_rcv_t				gir_rcv;
	osm_gir_rcv_ctrl_t			gir_rcv_ctrl;
	osm_lr_rcv_t				lr_rcv;
	osm_lr_rcv_ctrl_t			lr_rcv_ctrl;
	osm_pr_rcv_t				pr_rcv;
	osm_pr_rcv_ctrl_t			pr_rcv_ctrl;
	osm_smir_rcv_t				smir_rcv;
	osm_smir_ctrl_t				smir_ctrl;
	osm_mcmr_recv_t				mcmr_rcv;
	osm_mcmr_rcv_ctrl_t			mcmr_rcv_ctlr;
	osm_sr_rcv_t				sr_rcv;
	osm_sr_rcv_ctrl_t			sr_rcv_ctrl;
	
  /* InformInfo Receiver */
  osm_infr_rcv_t				infr_rcv;
  osm_infr_rcv_ctrl_t		infr_rcv_ctrl;

  /* VL Arbitrartion Query */
  osm_vlarb_rec_rcv_t       vlarb_rec_rcv;
  osm_vlarb_rec_rcv_ctrl_t  vlarb_rec_rcv_ctrl;

  /* SLtoVL Map Query */
  osm_slvl_rec_rcv_t       slvl_rec_rcv;
  osm_slvl_rec_rcv_ctrl_t  slvl_rec_rcv_ctrl;
  
  /* P_Key table Query */
  osm_pkey_rec_rcv_t       pkey_rec_rcv;
  osm_pkey_rec_rcv_ctrl_t  pkey_rec_rcv_ctrl;

  /* LinearForwardingTable Query */
  osm_lftr_rcv_t           lftr_rcv;
  osm_lftr_rcv_ctrl_t      lftr_rcv_ctrl;
  
} osm_sa_t;
/*
* FIELDS
*	state
		State of this SA object
*	p_subn
*		Pointer to the Subnet object for this subnet.
*
*	p_vendor
*		Pointer to the vendor specific interfaces object.
*
*	p_log
*		Pointer to the log object.
*
*	p_mad_pool
*		Pointer to the MAD pool.
*
*	p_disp
*		Pointer to dispatcher
*
*	p_lock
*		Pointer to Lock for serialization
*
*	sa_trans_id
*		Transaction ID
*
*	mad_ctrl
*		Mad Controller
*
*	resp
*		Response object
*
*	nr
*
*	nr_ctrl
*
*	pir_rcv
*
*	pir_rcv_ctrl
*
*	lr
*
*	lr_ctrl
*
*	pr
*
*	pr_ctrl
*
*	smir
*
*	smir_ctrl
*
* SEE ALSO
*	SM object
*********/

/****f* OpenSM: SA/osm_sa_construct
* NAME
*	osm_sa_construct
*
* DESCRIPTION
*	This function constructs an SA object.
*
* SYNOPSIS
*/
void osm_sa_construct(
	IN osm_sa_t* const p_sa );
/*
* PARAMETERS
*	p_sa
*		[in] Pointer to a SA object to construct.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Allows calling osm_sa_init, osm_sa_destroy, and osm_sa_is_inited.
*
*	Calling osm_sa_construct is a prerequisite to calling any other
*	method except osm_sa_init.
*
* SEE ALSO
*	SA object, osm_sa_init, osm_sa_destroy, osm_sa_is_inited
*********/

/****f* OpenSM: SA/osm_sa_shutdown
* NAME
*	osm_sa_shutdown
*
* DESCRIPTION
*	The osm_sa_shutdown function shutdowns an SA, unregistering from all
*  dispatcher messages and unbinding the QP1 mad service 
*
* SYNOPSIS
*/
void osm_sa_shutdown(
	IN osm_sa_t* const p_sa );
/*
* PARAMETERS
*	p_sa
*		[in] Pointer to a SA object to shutdown.
*
* RETURN VALUE
*	This function does not return a value.
*
* SEE ALSO
*	SA object, osm_sa_construct, osm_sa_init
*********/

/****f* OpenSM: SA/osm_sa_destroy
* NAME
*	osm_sa_destroy
*
* DESCRIPTION
*	The osm_sa_destroy function destroys an SA, releasing
*	all resources.
*
* SYNOPSIS
*/
void osm_sa_destroy(
	IN osm_sa_t* const p_sa );
/*
* PARAMETERS
*	p_sa
*		[in] Pointer to a SA object to destroy.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Performs any necessary cleanup of the specified SA object.
*	Further operations should not be attempted on the destroyed object.
*	This function should only be called after a call to osm_sa_construct or
*	osm_sa_init.
*
* SEE ALSO
*	SA object, osm_sa_construct, osm_sa_init
*********/

/****f* OpenSM: SA/osm_sa_init
* NAME
*	osm_sa_init
*
* DESCRIPTION
*	The osm_sa_init function initializes a SA object for use.
*
* SYNOPSIS
*/
ib_api_status_t osm_sa_init(
	IN osm_sm_t* const p_sm,
	IN osm_sa_t* const p_sa,
	IN osm_subn_t* const p_subn,
	IN osm_vendor_t* const p_vendor,
	IN osm_mad_pool_t* const p_mad_pool,
	IN osm_log_t* const p_log,
	IN osm_stats_t*	const p_stats,
	IN cl_dispatcher_t* const p_disp,
	IN cl_plock_t* const p_lock );
/*
* PARAMETERS
*	p_sa
*		[in] Pointer to an osm_sa_t object to initialize.
*
*	p_subn
*		[in] Pointer to the Subnet object for this subnet.
*
*	p_vendor
*		[in] Pointer to the vendor specific interfaces object.
*
*	p_mad_pool
*		[in] Pointer to the MAD pool.
*
*	p_log
*		[in] Pointer to the log object.
*
*	p_stats
*		[in] Pointer to the statistics object.
*
*	p_disp
*		[in] Pointer to the OpenSM central Dispatcher.
*
*	p_lock
*		[in] Pointer to the OpenSM serializing lock.
*
* RETURN VALUES
*	CL_SUCCESS if the SA object was initialized successfully.
*
* NOTES
*	Allows calling other SA methods.
*
* SEE ALSO
*	SA object, osm_sa_construct, osm_sa_destroy,
*	osm_sa_is_inited
*********/

/****f* OpenSM: SA/osm_sa_is_inited
* NAME
*	osm_sa_is_inited
*
* DESCRIPTION
*	Indicates if the object has been initialized with osm_sa_init.
*
* SYNOPSIS
*/
boolean_t osm_sa_is_inited(
	IN const osm_sa_t* const p_sa );
/*
* PARAMETERS
*	p_sa
*		[in] Pointer to an osm_sa_t object.
*
* RETURN VALUES
*	TRUE if the object was initialized successfully,
*	FALSE otherwise.
*
* NOTES
*	The osm_sa_construct or osm_sa_init must be called before using
*	this function.
*
* SEE ALSO
*	SA object, osm_sa_construct, osm_sa_init
*********/

/****f* OpenSM: SA/osm_sa_bind
* NAME
*	osm_sa_bind
*
* DESCRIPTION
*	Binds the SA object to a port guid.
*
* SYNOPSIS
*/
ib_api_status_t
osm_sa_bind(
	IN osm_sa_t* const p_sa,
	IN const ib_net64_t port_guid );
/*
* PARAMETERS
*	p_sa
*		[in] Pointer to an osm_sa_t object to bind.
*
*	port_guid
*		[in] Local port GUID with which to bind.
*
*
* RETURN VALUES
*	None
*
* NOTES
*	A given SA object can only be bound to one port at a time.
*
* SEE ALSO
*********/

/****f* OpenSM: SA/osm_sa_add_well_known_mc_record
* NAME
*	osm_sa_add_well_known_mc_record
*
* DESCRIPTION
*	Adds the well known Multicast group to the SA database. This 
*	should be called by SM before programming the switches and after
*	SA has been initialized
*
* SYNOPSIS
*/

void
osm_sa_add_well_known_mc_record(
	osm_mcmr_recv_t* const p_ctrl,
	const ib_member_rec_t * const p_well_know_mc_rec);
/*
* PARAMETERS
*	p_ctrl
*		[in] Pointer to an osm_mcmr_recv_t object.
*
*	p_well_know_mc_rec
*		[in] pointer to ib_member_rec_t structure.
*
*
* RETURN VALUES
*	None
*
* NOTES
*	Called by SM after SA has been initialized and before programming the switches
*
* SEE ALSO
*********/


/****f* OpenSM: SA/osm_sa_create_template_record_ipoib
* NAME
*	osm_sa_create_template_record_ipoib
*
* DESCRIPTION
*	Creates the well known MC record and calls osm_sa_add_well_known_mc_record. This 
*	should be called by SM before programming the switches and after
*	SA has been initialized
*
* SYNOPSIS
*/
void
osm_sa_create_template_record_ipoib(
	IN osm_sa_t* const p_sa,
	IN const osm_subn_opt_t* const p_opt );

/*
* PARAMETERS
*	p_sa
*		[in] Pointer to an osm_sa_t object.
*
*	p_opt
*		[in] pointer to cmd line option structure.
*
*
* RETURN VALUES
*	None
*
* NOTES
*	Called by SM after SA has been initialized and before programming the switches
*
* SEE ALSO
*********/

/****g* OpenSM: SA/osm_ipoib_mgid
* NAME
*	osm_ipoib_mgid
*
* DESCRIPTION
*	The MGID of the IPoIB Multicast Group
*
* SYNOPSIS
*/
extern ib_gid_t osm_ipoib_mgid;
/*
* SEE ALSO
*********/

END_C_DECLS

#endif		/* _OSM_SA_H_ */
