/*
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: osm_pkey_mgr.h 6544 2006-04-20 19:16:22Z halr $
 */


/*
 * Abstract:
 * 	Declaration of osm_pkey_mgr_t.
 *	This object represents the P_Key Manager object.
 *	This object is part of the OpenSM family of objects.
 *
 * Environment:
 * 	Linux User Mode
 *
 * $Revision: 1.4 $
 */


#ifndef _OSM_PKEY_MGR_H_
#define _OSM_PKEY_MGR_H_

#include <complib/cl_passivelock.h>
#include <opensm/osm_subnet.h>
#include <opensm/osm_req.h>
#include <opensm/osm_log.h>

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif /* __cplusplus */

BEGIN_C_DECLS

/****h* OpenSM/P_Key Manager
* NAME
*	P_Key Manager
*
* DESCRIPTION
*	The P_Key Manager object manage the p_key tables of all
*	objects in the subnet
*
* AUTHOR
*	Ofer Gigi, Mellanox
*
*********/
/****s* OpenSM: P_Key Manager/osm_pkey_mgr_t
* NAME
*	osm_pkey_mgr_t
*
* DESCRIPTION
*	p_Key Manager structure.
*
*
* SYNOPSIS
*/

typedef struct _osm_pkey_mgr
{
    osm_subn_t					*p_subn;
	osm_log_t					*p_log;
	osm_req_t					*p_req;
	cl_plock_t					*p_lock;

} osm_pkey_mgr_t;

/*
* FIELDS
*	p_subn
*		Pointer to the Subnet object for this subnet.
*
*	p_log
*		Pointer to the log object.
*
*	p_req
*		Pointer to the Request object.
*
*	p_lock
*		Pointer to the serializing lock.
*
* SEE ALSO
*	P_Key Manager object
*********/

/****** OpenSM: P_Key Manager/osm_pkey_mgr_construct
* NAME
*	osm_pkey_mgr_construct
*
* DESCRIPTION
*	This function constructs a P_Key Manager object.
*
* SYNOPSIS
*/
void 
osm_pkey_mgr_construct(
	IN osm_pkey_mgr_t* const p_mgr );
/*
* PARAMETERS
*	p_mgr
*		[in] Pointer to a P_Key Manager object to construct.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Allows calling osm_pkey_mgr_init, osm_pkey_mgr_destroy
*
*	Calling osm_pkey_mgr_construct is a prerequisite to calling any other
*	method except osm_pkey_mgr_init.
*
* SEE ALSO
*	P_Key Manager object, osm_pkey_mgr_init,
*	osm_pkey_mgr_destroy
*********/

/****f* OpenSM: P_Key Manager/osm_pkey_mgr_destroy
* NAME
*	osm_pkey_mgr_destroy
*
* DESCRIPTION
*	The osm_pkey_mgr_destroy function destroys the object, releasing
*	all resources.
*
* SYNOPSIS
*/
void
osm_pkey_mgr_destroy(
	IN osm_pkey_mgr_t* const p_mgr );
/*
* PARAMETERS
*	p_mgr
*		[in] Pointer to the object to destroy.
*
* RETURN VALUE
*	This function does not return a value.
*
* NOTES
*	Performs any necessary cleanup of the specified
*	P_Key Manager object.
*	Further operations should not be attempted on the destroyed object.
*	This function should only be called after a call to
*	osm_pkey_mgr_construct or osm_pkey_mgr_init.
*
* SEE ALSO
*	P_Key Manager object, osm_pkey_mgr_construct,
*	osm_pkey_mgr_init
*********/

/****f* OpenSM: P_Key Manager/osm_pkey_mgr_init
* NAME
*	osm_pkey_mgr_init
*
* DESCRIPTION
*	The osm_pkey_mgr_init function initializes a
*	P_Key Manager object for use.
*
* SYNOPSIS
*/
ib_api_status_t
osm_pkey_mgr_init(
	IN osm_pkey_mgr_t* const p_mgr,
	IN osm_subn_t* const p_subn,
	IN osm_log_t* const p_log,
	IN osm_req_t* const p_req,
	IN cl_plock_t* const p_lock );
/*
* PARAMETERS
*	p_mgr
*		[in] Pointer to an osm_pkey_mgr_t object to initialize.
*
*	p_subn
*		[in] Pointer to the Subnet object for this subnet.
*
*	p_log
*		[in] Pointer to the log object.
*
*	p_req
*		[in] Pointer to an osm_req_t object.
*
*	p_lock
*		[in] Pointer to the OpenSM serializing lock.
*
* RETURN VALUES
*	IB_SUCCESS if the P_Key Manager object was initialized
*	successfully.
*
* NOTES
*	Allows calling other P_Key Manager methods.
*
* SEE ALSO
*	P_Key Manager object, osm_pkey_mgr_construct,
*	osm_pkey_mgr_destroy
*********/

/****f* OpenSM: P_Key Manager/osm_pkey_mgr_process
* NAME
*	osm_pkey_mgr_process
*
* DESCRIPTION
*	This function enforces the pkey rules on the SM DB.
*
* SYNOPSIS
*/
osm_signal_t 
osm_pkey_mgr_process(
	IN const osm_pkey_mgr_t* const p_mgr );
/*
* PARAMETERS
*	p_mgr
*		[in] Pointer to an osm_pkey_mgr_t object.
*
* RETURN VALUES
*	None
*
* NOTES
*   Current Operations:
*   - Inserts IB_DEFAULT_PKEY to all node objects that don't have 
*      IB_DEFAULT_PARTIAL_PKEY or IB_DEFAULT_PKEY as part 
*     of their p_key table
*
* SEE ALSO
*	P_Key Manager
*********/

END_C_DECLS

#endif	/* _OSM_PKEY_MGR_H_ */
