/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: cl_waitobj.h 1743 2005-02-06 09:38:35Z shaharf $
 */



/*
 * Abstract:
 *	This header file defines the data structures and APIs for implementing
 *	wait objects.
 *
 * Environment:
 *	Linux Kernel Mode
 *
 * $Revision: 1.7 $
 */

#ifndef _CL_WAITOBJ_H_
#define _CL_WAITOBJ_H_

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS   }
#else /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif /* __cplusplus */

BEGIN_C_DECLS

/****h* Component Library/Wait Objects
* NAME
*	Wait Object
*
* DESCRIPTION
*	The Wait Object provides the capability for a user mode process to
*	create and wait on a kernel event.  An action on a wait object can
*	be done from both a user mode thread as well as a kernel thread.
*
******/

#include <complib/cl_types.h>

/****d* Component Library: Wait Objects/cl_wait_obj_handle_t
* NAME
*	cl_wait_obj_handle_t
*
* DESCRIPTION
*	Defines the handle for an OS wait object.
*
* SYNOPSIS
*/
typedef void *cl_wait_obj_handle_t;
/*
*
******/

/****i* Component Library: Wait Objects/cl_wait_ioctl_params_t
* NAME
*	cl_wait_ioctl_params
*
* DESCRIPTION
*	Defines parameters for the ioctl call to implement wait objects
*
* SYNOPSIS
*/
typedef struct cl_wait_ioctl_params
{
	uint32_t			wait_u_sec;
	cl_status_t			wait_status;

} cl_wait_ioctl_params_t;
/*
*
******/

/****i* Component Library: Wait Objects/cl_create_wait_obj_params_t
* NAME
*	cl_create_wait_obj_params
*
* DESCRIPTION
*	Defines parameters for the ioctl call to create wait objects
*
* SYNOPSIS
*/
typedef struct cl_create_wait_obj_params
{
	// Input
	boolean_t				auto_reset;
	// Output
	cl_status_t				status;

} cl_create_wait_obj_params_t;
/*
*
******/


#if defined(__KERNEL__)
/*
 * Kernel Mode Support for Wait Objects
 */

/* Internal helper functions for wait object */

cl_status_t
__cl_create_wait_object(
	IN	boolean_t				auto_reset,
	OUT	cl_wait_obj_handle_t	*p_wait_obj_handle);

cl_status_t
__cl_wait_on_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle,
	IN	uint32_t				wait_u_sec );

cl_status_t
__cl_signal_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle);

cl_status_t
__cl_clear_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle);

cl_status_t
__cl_destroy_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle);


/****i* Component Library: Wait Objects/cl_get_kernel_wait_object
* NAME
*	cl_get_kernel_wait_object
*
* DESCRIPTION
*	cl_get_kernel_wait_object -- Validates the wait object handle and
*								returns the kernel wait object handle.
*
* SYNOPSIS
*/
cl_wait_obj_handle_t
cl_get_kernel_wait_object(
	IN	cl_wait_obj_handle_t	user_mode_handle );
/*
* PARAMETERS
*	user_mode_handle
*		A handle to the wait object passed from user mode.
*
* RETURN VALUES
*	On successful validation, returns the kernel wait object handle.
*	Kernel threads use this handle to do any appropriate action
*	on this wait object.
*	On failure, returns NULL.
*
* NOTES
*	This API is used only in the kernel.
*
* SEE ALSO
*	cl_create_wait_object, cl_destroy_wait_object, cl_wait_on_wait_object,
*	cl_signal_wait_object, cl_clear_wait_object.
*
******/

/******/

#endif	//__KERNEL__

/*
 * Shared between user and kernel mode
 */

/****f* Component Library: Wait Objects/cl_create_wait_object
* NAME
*	cl_create_wait_object
*
* DESCRIPTION
*	cl_create_wait_object -- Creates a wait object.
*
* SYNOPSIS
*/
cl_status_t
cl_create_wait_object(
	IN	const boolean_t			auto_reset,
	OUT	cl_wait_obj_handle_t	*p_wait_obj_handle );
/*
* PARAMETERS
*	auto_reset
*		Specifies whether the signaled state should be reset automatically
*		or manually.  If set to TRUE, the state will be reset automatically.
*	p_wait_obj_handle
*		On successful creation, returns the wait object handle.
*
* RETURN VALUES
*	CL_SUCCESS
*		The wait object was created successfully.
*	CL_ERROR
*		The wait object creation failed.
* NOTES
*	Used in both kernel mode as well as user mode.
*
* SEE ALSO
*	cl_destroy_wait_object, cl_wait_on_wait_object,
*	cl_signal_wait_object, cl_clear_wait_object.
*
******/

/****f* Component Library: Wait Objects/cl_destroy_wait_object
* NAME
*	cl_destroy_wait_object
*
* DESCRIPTION
*	cl_destroy_wait_object -- Destroys a wait object.
*
* SYNOPSIS
*/
cl_status_t
cl_destroy_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle );
/*
* PARAMETERS
*	wait_obj_handle
*		A handle to the wait object that needs to be destroyed.
*
* RETURN VALUES
*	CL_SUCCESS
*		The wait object handle is destroyed.
* NOTES
*	Used in both kernel mode as well as user mode.
*
* SEE ALSO
*	cl_create_wait_object, cl_wait_on_wait_object,
*	cl_signal_wait_object, cl_clear_wait_object.
*
*********/

/****f* Component Library: Wait Objects/cl_wait_on_wait_object
* NAME
*	cl_wait_on_wait_object
* DESCRIPTION
*	cl_wait_on_wait_object -- Wait on this wait object until signalled
*								or timed out.
*
* SYNOPSIS
*/
cl_status_t
cl_wait_on_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle,
	IN	uint32_t				wait_u_sec );
/*
* PARAMETERS
*	wait_obj_handle
*		A handle to the wait object that the thread needs to be wait on.
*	wait_u_sec
*		The number of micro seconds to wait before timing out.
*
* RETURN VALUES
*	CL_SUCCESS
*		The wait completed successfully and the event is signalled.
*	CL_ERROR
*		Some error happened during the wait.
*	CL_NOT_DONE
*		The wait got interrupted due to some signal.
*	CL_TIMEOUT
*		The wait timed out.
*
* NOTES
*	Used in both kernel mode as well as user mode.
*
* SEE ALSO
*	cl_create_wait_object, cl_destroy_wait_object,
*	cl_signal_wait_object, cl_clear_wait_object.
*
*********/

/****f* Component Library: Wait Objects/cl_signal_wait_object
* NAME
*	cl_signal_wait_object
*
* DESCRIPTION
*	cl_signal_wait_object -- Signal the wait object.  The thread
*						that is waiting on this will be woken up.
*
* SYNOPSIS
*/
cl_status_t
cl_signal_wait_object(
	IN	cl_wait_obj_handle_t wait_obj_handle );
/*
* PARAMETERS
*	wait_obj_handle
*		A handle to the wait object that needs to be signaled.
*
* RETURN VALUES
*	CL_SUCCESS
*		The wait object is successfully signaled.
*
* NOTES
*	Used in both kernel mode as well as user mode.
*
* SEE ALSO
*	cl_create_wait_object, cl_destroy_wait_object,
*	cl_wait_on_wait_object, cl_clear_wait_object.
*
*********/

/****f* Component Library: Wait Objects/cl_clear_wait_object
* NAME
*	cl_clear_wait_object
*
* DESCRIPTION
*	cl_clear_wait_object -- Clear the signaled state of a wait object
*							and reset it.
*
* SYNOPSIS
*/
cl_status_t
cl_clear_wait_object(
	IN	cl_wait_obj_handle_t wait_obj_handle );
/*
* PARAMETERS
*	wait_obj_handle
*		A handle to the wait object whose state needs to be reset.
*
* RETURN VALUES
*	CL_SUCCESS
*		The wait object is reset successfully.
*
* NOTES
*	Used in both kernel mode as well as user mode.
*
* SEE ALSO
*	cl_create_wait_object, cl_destroy_wait_object,
*	cl_wait_on_wait_object, cl_signal_wait_object.
*
*********/

END_C_DECLS

#endif	// _CL_WAITOBJ_H_
