/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: cl_waitobj.c 4264 2005-11-30 21:10:36Z halr $
 */

/*
 * Abstract:
 *	This module defines
 *
 * Environment:
 *	Linux User Mode
 *
 * $Revision: 1.5 $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <complib/cl_types.h>
#include <complib/cl_device.h>
#include <complib/cl_debug.h>
#include <complib/cl_syshelper.h>
#include <complib/cl_waitobj.h>
#include <complib/cl_memory.h>


cl_status_t
cl_create_wait_object(
	IN	const boolean_t			auto_reset,
	OUT	cl_wait_obj_handle_t	*p_wait_obj_handle )
{
	cl_status_t					status = CL_SUCCESS;
	uintn_t						command;
	cl_create_wait_obj_params_t	ioctl_buf;
	uintn_t						num_bytes_returned;
	cl_dev_handle_t				device_handle;

	/*
	 * First open sysdev to get a new FD
	 */
	status = cl_open_device( SYSHELP_DEVICE_NAME, &device_handle );
	if ( status != CL_SUCCESS )
	{
		cl_msg_out( "Failed to open device %s, status (%s)\n",
							SYSHELP_DEVICE_NAME, CL_STATUS_MSG(status) );
		return status;
	}

	command = CREATE_WAIT_OBJ;
	ioctl_buf.auto_reset = auto_reset;

	status = cl_ioctl_device(device_handle,
							 command,
							 &ioctl_buf,	 /* result */
							 sizeof (cl_create_wait_obj_params_t),
							 &num_bytes_returned);

	status = (CL_SUCCESS != status) ? status : ioctl_buf.status;

	if ( status != CL_SUCCESS)
	{
		cl_msg_out("create_wait_object: failed to create waitobject (%s)\n",
							CL_STATUS_MSG(status) );
		return status;
	}

	*p_wait_obj_handle = (cl_wait_obj_handle_t)device_handle;

	return status;
}

cl_status_t
cl_wait_on_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle,
	IN	uint32_t				wait_u_sec )
{
	cl_status_t				status = CL_SUCCESS;
	cl_wait_ioctl_params_t	ioctl_params;
	uintn_t					command;
	cl_dev_handle_t			device_handle = (cl_dev_handle_t)wait_obj_handle;

	/* fill out the ioctl parameters */
	ioctl_params.wait_u_sec = wait_u_sec;
	ioctl_params.wait_status = CL_SUCCESS;

	command = WAITON_WAIT_OBJ;

	status = cl_ioctl_device(device_handle,
							 command,
							 &ioctl_params,
							 sizeof(cl_wait_ioctl_params_t),
							 NULL );

	if ( status != CL_SUCCESS)
	{
		cl_msg_out("wait_on_wait_object: cl_ioctl_device failed (%s)\n",
								CL_STATUS_MSG(status) );
		return status;
	}

	/*
	 * if the ioctl returned successfully, return the status
	 * of the wait
	 */

	return (ioctl_params.wait_status);
}

cl_status_t
cl_signal_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle )
{
	uintn_t					command;
	cl_status_t				status = CL_SUCCESS;
	void					*p_in_buf;
	cl_dev_handle_t			device_handle = (cl_dev_handle_t)wait_obj_handle;

	p_in_buf	= wait_obj_handle;

	command = TRIGGER_WAIT_OBJ;

	status = cl_ioctl_device(device_handle,
							 command,
							 &p_in_buf,
							 sizeof(p_in_buf),
							 NULL);

	if ( status != CL_SUCCESS)
	{
		cl_msg_out("trigger_wait_object: cl_ioctl_device failed (%s)\n",
							CL_STATUS_MSG(status) );
		return	status;
	}

	return status;
}

cl_status_t
cl_destroy_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle )
{
	cl_status_t				status = CL_SUCCESS;
	cl_dev_handle_t			device_handle = (cl_dev_handle_t)wait_obj_handle;

	cl_close_device( device_handle );

	return status;
}

cl_status_t
cl_clear_wait_object(
	IN	cl_wait_obj_handle_t	wait_obj_handle )
{
	uintn_t					command;
	cl_status_t				status = CL_SUCCESS;
	void					*p_in_buf;
	cl_dev_handle_t			device_handle = (cl_dev_handle_t)wait_obj_handle;

	p_in_buf	= wait_obj_handle;

	command = RESET_WAIT_OBJ;

	status = cl_ioctl_device(device_handle,
							 command,
							 &p_in_buf,
							 sizeof(p_in_buf),
							 NULL);

	if ( status != CL_SUCCESS)
	{
		cl_msg_out("clear_wait_object: cl_ioctl_device failed (%s)\n",
							CL_STATUS_MSG(status) );
		return	status;
	}

	return status;
}
