/*
 * Copyright (c) 2004, 2005 Voltaire, Inc. All rights reserved.
 * Copyright (c) 2002-2005 Mellanox Technologies LTD. All rights reserved.
 * Copyright (c) 1996-2003 Intel Corporation. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: cl_device.c 3379 2005-09-12 20:10:17Z halr $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

/*
 * Standard user mode includes
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <string.h>

#include <complib/cl_types.h>
#include <complib/cl_debug.h>
#include <complib/cl_device.h>


cl_status_t
cl_open_device(
	IN cl_dev_name_t	device_name,
	IN cl_dev_handle_t	*p_dev_handle )
{
	/* sanity check */
	if ( p_dev_handle == NULL)
	{
		return CL_INVALID_PARAMETER;
	}

	cl_dbg_out ("cl_open_device: opening device %s\n",
										device_name);

	*p_dev_handle = open(device_name, O_RDWR);

	if (*p_dev_handle < 0)
	{
		*p_dev_handle = 0;
		cl_msg_out("cl_open_dev: error opening %s (%s)\n",
								device_name, strerror(errno));
		return CL_ERROR;
	}
	else
	{
		return CL_SUCCESS;
	}
}

void
cl_close_device(
	IN cl_dev_handle_t dev_handle )
{
	int	status = 0;

	status = close (dev_handle);
	if (status)
	{
		cl_msg_out("cl_close_device: error closing device (%s)\n",
											strerror(errno));
	}
	return;
}

cl_status_t
cl_ioctl_device(
	IN	cl_dev_handle_t	dev_handle,
	IN	uint32_t		command,
	IN	void			*p_buf,
	IN	uintn_t			buf_size,
	OUT uintn_t			*p_num_bytes_ret )
{
	cl_ioctl_info_t		ioctl_args;
	int					retval = 0;

	/*
	 * Fill up ioctl_args and issue a real ioctl
	 */
	ioctl_args.command			= command;
	ioctl_args.p_buf			= p_buf;
	ioctl_args.buf_size			= buf_size;
	ioctl_args.num_bytes_ret	= 0;			/* for now */
	ioctl_args.io_status		= CL_SUCCESS;	/* lets start here */

	retval = ioctl(dev_handle, command, &ioctl_args);

	if (retval != 0)
	{
		cl_msg_out("cl_ioctl_device: error (%s) issuing command (0x%x)\n",
									strerror(errno), command);
		return CL_ERROR;
	}

	/*
	 * Set the Number of bytes returned from the Kernel.
	 * The driver sets the number of bytes returned in
	 * ioctl_args.num_bytes_ret
	 */
	if (p_num_bytes_ret != NULL)
	{
		*p_num_bytes_ret = ioctl_args.num_bytes_ret;
	}

	/*
	 * Return the status received from the Kernel
	 */

	return (ioctl_args.io_status);
}
