/*
 * Copyright (c) 2004-2006 Voltaire Inc.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * $Id: ibnetdiscover.c 5996 2006-03-24 13:50:20Z halr $
 */

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <time.h>
#include <string.h>
#include <getopt.h>
#include <ctype.h>

#include <common.h>
#include <umad.h>
#include <mad.h>

#define MAXHOPS	63
#define CA_NODE		1
#define SWITCH_NODE	2

static int timeout = 2000;		/* ms */
static int dumplevel = 0;
static int verbose;
static FILE *f;

#undef DEBUG
#define	DEBUG	if (verbose>1) IBWARN
#define IBERROR(fmt, args...)	iberror(__FUNCTION__, fmt, ## args)

static char *argv0 = "ibnetdiscover";

void
iberror(const char *fn, char *msg, ...)
{
	char buf[512], *s;
	va_list va;
	int n;

	va_start(va, msg);
	n = vsprintf(buf, msg, va);
	va_end(va);
	buf[n] = 0;

	if ((s = strrchr(argv0, '/')))
		argv0 = s + 1;

	if (ibdebug)
		printf("%s: iberror: [pid %d] %s: %s\n", argv0, getpid(), fn, buf);
	else
		printf("%s: iberror: %s\n", argv0, buf);

	exit(-1);
}

typedef struct Port Port;
typedef struct Node Node;

struct Port {
	Port *next;
	uint64_t portguid;
	int portnum;
	int lid;
	int lmc;
	int state;
	int physstate;

	Node *node;
	Port *remoteport;		/* null if SMA */
};

struct Node {
	Node *htnext;
	Node *dnext;
	Port *ports;
	ib_portid_t path;
	int type;
	int dist;
	int numports;
	int localport;
	int smalid;
	uint32_t devid;
	uint32_t vendid;
	uint64_t sysimgguid;
	uint64_t nodeguid;
	uint64_t portguid;
	char nodedesc[64];
	uint8_t nodeinfo[64];
};

Node *nodesdist[MAXHOPS+1];	/* last is Ca list */
Node *mynode;
Port *myport;
int maxhops_discovered = 0;

int
get_port(Port *port, int portnum, ib_portid_t *portid)
{
	char portinfo[64];
	void *pi = portinfo;

	port->portnum = portnum;

	if (!smp_query(pi, portid, IB_ATTR_PORT_INFO, portnum, timeout))
		return -1;

	mad_decode_field(pi, IB_PORT_LID_F, &port->lid);
	mad_decode_field(pi, IB_PORT_LMC_F, &port->lmc);
	mad_decode_field(pi, IB_PORT_STATE_F, &port->state);
	mad_decode_field(pi, IB_PORT_PHYS_STATE_F, &port->physstate);

	DEBUG("portid %s portnum %d: lid %d state %d physstate %d",
		portid2str(portid), portnum, port->lid, port->state, port->physstate);
	return 1;
}
/*
 * Returns 0 if non switch node is found, 1 if switch is found, -1 if error.
 */
int
get_node(Node *node, Port *port, ib_portid_t *portid)
{
	char portinfo[64];
	void *pi = portinfo, *ni = node->nodeinfo, *nd = node->nodedesc;

	if (!smp_query(ni, portid, IB_ATTR_NODE_INFO, 0, timeout))
		return -1;

	mad_decode_field(ni, IB_NODE_GUID_F, &node->nodeguid);
	mad_decode_field(ni, IB_NODE_TYPE_F, &node->type);
	mad_decode_field(ni, IB_NODE_NPORTS_F, &node->numports);
	mad_decode_field(ni, IB_NODE_DEVID_F, &node->devid);
	mad_decode_field(ni, IB_NODE_VENDORID_F, &node->vendid);
	mad_decode_field(ni, IB_NODE_SYSTEM_GUID_F, &node->sysimgguid);

	mad_decode_field(ni, IB_NODE_PORT_GUID_F, &node->portguid);
	mad_decode_field(ni, IB_NODE_LOCAL_PORT_F, &node->localport);
	port->portnum = node->localport;
	port->portguid = node->nodeguid;

	if (!smp_query(nd, portid, IB_ATTR_NODE_DESC, 0, timeout))
		return -1;

	if (!smp_query(pi, portid, IB_ATTR_PORT_INFO, 0, timeout))
		return -1;

	mad_decode_field(pi, IB_PORT_LID_F, &port->lid);
	mad_decode_field(pi, IB_PORT_LMC_F, &port->lmc);
	mad_decode_field(pi, IB_PORT_STATE_F, &port->state);
	mad_decode_field(pi, IB_PORT_PHYS_STATE_F, &port->physstate);

	if (node->type != SWITCH_NODE)
		return 0;

	node->smalid = port->lid;

	DEBUG("portid %s: got switch node %Lx '%s'",
	      portid2str(portid), node->nodeguid, nd);
	return 1;
}

static int
extend_dpath(ib_dr_path_t *path, int nextport)
{
	if (path->cnt+2 >= sizeof(path->p))
		return -1;
	++path->cnt;
	if (path->cnt > maxhops_discovered)
		maxhops_discovered = path->cnt;
	path->p[path->cnt] = nextport;
	return path->cnt;
}

static char *
clean_nodedesc(char *nodedesc)
{
	int i = 0;

	nodedesc[63] = '\0';
	for (i = 0; i < 64; i++) {
		if (iscntrl(nodedesc[i]) || nodedesc[i] == '\0') {
			nodedesc[i] = '\0'; 
			break;
		}
	}
	return (nodedesc);
}

static void
dump_endnode(ib_portid_t *path, char *prompt, Node *node, Port *port)
{
	if (!dumplevel)
		return;

#if __WORDSIZE == 64
	fprintf(f, "%s -> %s %s {%016lx} portnum %d lid 0x%x-0x%x \"%s\"\n",
		portid2str(path), prompt, node->type == SWITCH_NODE ? "switch" : "ca",
		node->nodeguid, node->type == SWITCH_NODE ? 0 : port->portnum,
		port->lid, port->lid + (1 << port->lmc) - 1,
		clean_nodedesc(node->nodedesc));
#else
	fprintf(f, "%s -> %s %s {%016Lx} portnum %d lid 0x%x-0x%x \"%s\"\n",
		portid2str(path), prompt, node->type == SWITCH_NODE ? "switch" : "ca",
		node->nodeguid, node->type == SWITCH_NODE ? 0 : port->portnum,
		port->lid, port->lid + (1 << port->lmc) - 1,
		clean_nodedesc(node->nodedesc));
#endif
}

#define HASHGUID(guid)		((uint32_t)(((uint32_t)(guid) * 101) ^ ((uint32_t)((guid) >> 32) * 103)))
#define HTSZ 137

Node *
insert_node(Node *new)
{
	static Node *nodestbl[HTSZ];
	int hash = HASHGUID(new->nodeguid) % HTSZ;
	Node *node ;

	for (node = nodestbl[hash]; node; node = node->htnext)
		if (node->nodeguid == new->nodeguid) {
			DEBUG("node %Lx already exists", new->nodeguid);
			return node;
		}

	new->htnext = nodestbl[hash];
	nodestbl[hash] = new;

	return 0;
}

Port *
link_port(Port *port, Node *node, Port *remoteport)
{
	Port *old;

	for (old = node->ports; old; old = old->next)
		if (old->portnum == port->portnum) {
			DEBUG("found old port %p (%d) for node %p",
			      old, port->portnum, node);
			return old;
		}

	if (dumplevel)
#if __WORDSIZE == 64
		fprintf(f, "\t[%d] {%016lx}\n", port->portnum, port->portguid);
#else
		fprintf(f, "\t[%d] {%016Lx}\n", port->portnum, port->portguid);
#endif

	DEBUG("inserting new port %p (%d) to node %p", port, port->portnum, node);
	port->node = node;
	port->remoteport = remoteport;

	port->next = node->ports;
	node->ports = port;

	return 0;
}

Node *
new_node(Node *node, ib_portid_t *path, int dist)
{
	Node *old;

	/* BFS search start with myself */
	if ((old = insert_node(node)))
		return old;	/* known node */

	DEBUG("insert dist %d node %p", dist, node);

	node->dist = dist;
	node->path = *path;

	if (node->type != SWITCH_NODE)
		dist = MAXHOPS; 	/* special Ca list */

	node->dnext = nodesdist[dist];
	nodesdist[dist] = node;

	return 0;	/* new switch is discovered */
}

static int
handle_port(Node *node, Port *port, ib_portid_t *path, int portnum, int dist)
{
	Node *remotenode, *oldnode;
	Port *remoteport;

	DEBUG("handle node %p port %p:%d dist %d", node, port, portnum, dist);
	if (port->physstate != 5)	/* LinkUp */
		return -1;

	if (extend_dpath(&path->drpath, portnum) < 0)
		return -1;

	if (!(remotenode = calloc(1, sizeof(Node))))
		IBERROR("out of memory");

	if (!(remoteport = calloc(1, sizeof(Port))))
		IBERROR("out of memory");

	if (get_node(remotenode, remoteport, path) < 0) {
		IBWARN("NodeInfo on %s port %d failed, skipping port",
			portid2str(path), portnum);
		free(remotenode);
		free(remoteport);

		path->drpath.cnt--;	/* restore path */
		return -1;
	}

	/* Handle loopbacks */
	if (remotenode->nodeguid == node->nodeguid) {
		free(remotenode);

		/* Handle loopback plug */
		if (port->portguid == remoteport->portguid) {
			free(remoteport);
			remoteport = port;
		}
		link_port(remoteport, node, port);
		link_port(port, node, remoteport);

		path->drpath.cnt--;	/* restore path */
		return 0;
	}

	if (link_port(port, node, remoteport)) {
		free(remoteport);
		free(remotenode);
		path->drpath.cnt--;	/* restore path */
		return 1;
	}

	oldnode = new_node(remotenode, path, dist + 1);
	if (oldnode) {
		free(remotenode);
		remotenode = oldnode;
		dump_endnode(path, "known remote", remotenode, remoteport);
	} else
		dump_endnode(path, "new remote", remotenode, remoteport);

	if (link_port(remoteport, remotenode, port))
		free(remoteport);

	path->drpath.cnt--;	/* restore path */
	return 0;
}

/*
 * Return 1 if found, 0 if not, -1 on errors.
 */
static int
discover(ib_portid_t *from)
{
	Node *node;
	Port *port;
	int i;
	int dist = 0;
	ib_portid_t *path;

	DEBUG("from %s", portid2str(from));

	if (!(node = calloc(1, sizeof(Node))))
		IBERROR("out of memory");

	if (!(port = calloc(1, sizeof(Port))))
		IBERROR("out of memory");

	if (get_node(node, port, from) < 0) {
		IBWARN("can't reach node %s", portid2str(from));
		return -1;
	}

	new_node(node, from, 0);

	mynode = node;
	myport = port;

	if (node->type == SWITCH_NODE)
		link_port(port, node, 0);	/* switch SMA */
	else if (handle_port(node, port, from, node->localport, 0) < 0)
		return 0;

	for (dist = 0; dist < MAXHOPS; dist++) {

		for (node = nodesdist[dist]; node; node = node->dnext) {

			path = &node->path;

			DEBUG("dist %d node %p", dist, node);
			dump_endnode(path, "processing", node, port);

			for (i = 1; i <= node->numports; i++) {
				if (i == node->localport)
					continue;
		
				if (!(port = calloc(1, sizeof(Port))))
					IBERROR("out of memory");
		
				if (get_port(port, i, path) < 0) {
					IBWARN("can't reach node %s port %d", portid2str(path), i);
					return 0;
				}

				/* If switch, set port GUID to node GUID */
				if (node->type == SWITCH_NODE)
					port->portguid = node->portguid;

				if (handle_port(node, port, path, i, dist)) {
					free(port);
					continue;
				}
			}
		}
	}

	return 0;
}

char *
node_name(Node *node)
{
	static char buf[256];

#if __WORDSIZE == 64
	sprintf(buf, "\"%s-%016lx\"",
		node->type == SWITCH_NODE ? "S" : "H", node->nodeguid);
#else
	sprintf(buf, "\"%s-%016Lx\"",
		node->type == SWITCH_NODE ? "S" : "H", node->nodeguid);
#endif

	return buf;
}

void
list_node(Node *node)
{
#if __WORDSIZE == 64
	fprintf(f, "%s\t : 0x%016lx ports %d devid 0x%x vendid 0x%x \"%s\"\n",
		node->type == SWITCH_NODE ? "Switch" : "Ca",
		node->nodeguid, node->numports, node->devid, node->vendid,
		clean_nodedesc(node->nodedesc));
#else
	fprintf(f, "%s\t : 0x%016Lx ports %d devid 0x%x vendid 0x%x \"%s\"\n",
		node->type == SWITCH_NODE ? "Switch" : "Ca",
		node->nodeguid, node->numports, node->devid, node->vendid,
		clean_nodedesc(node->nodedesc));
#endif
}

void
out_ids(Node *node)
{
	fprintf(f, "\nvendid=0x%x\ndevid=0x%x\n", node->vendid, node->devid);
	if (node->sysimgguid)
#if __WORDSIZE == 64
		fprintf(f, "sysimgguid=0x%lx\n", node->sysimgguid);
#else
		fprintf(f, "sysimgguid=0x%Lx\n", node->sysimgguid);
#endif
}

void
out_switch(Node *node)
{
	out_ids(node);
#if __WORDSIZE == 64
	fprintf(f, "%s=0x%lx\n", "switchguid", node->nodeguid);
#else
	fprintf(f, "%s=0x%Lx\n", "switchguid", node->nodeguid);
#endif

	fprintf(f, "\nSwitch\t%d %s\t\t# %s port 0 lid %d\n",
		node->numports, node_name(node),
		clean_nodedesc(node->nodedesc), node->smalid);
}

void
out_ca(Node *node)
{
	out_ids(node);
#if __WORDSIZE == 64
	fprintf(f, "%s=0x%lx\n", "caguid", node->nodeguid);
#else
	fprintf(f, "%s=0x%Lx\n", "caguid", node->nodeguid);
#endif
	fprintf(f, "%s\t%d %s\t\t# %s\n",
		"Ca", node->numports, node_name(node),
		clean_nodedesc(node->nodedesc));
}

void
out_switch_port(Port *port)
{
	DEBUG("port %p:%d remoteport %p", port, port->portnum, port->remoteport);
	fprintf(f, "[%d]\t%s[%d]\t\t\n",
		port->portnum, node_name(port->remoteport->node),
		port->remoteport->portnum);
}

void
out_ca_port(Port *port)
{
	fprintf(f, "[%d]\t%s[%d]\t\t# lid %d lmc %d\n",
		port->portnum, node_name(port->remoteport->node),
		port->remoteport->portnum, port->lid, port->lmc);
}

int
dump_topology(int listtype)
{
	Node *node;
	Port *port;
	int i = 0, dist;
	time_t t = time(0);

	if (!listtype) {
		fprintf(f, "#\n# Topology file: generated on %s#\n", ctime(&t));
		fprintf(f, "# Max of %d hops discovered\n", maxhops_discovered);
#if __WORDSIZE == 64
		fprintf(f, "# Initiated from node %016lx port %016lx\n", mynode->nodeguid, mynode->portguid);
#else
		fprintf(f, "# Initiated from node %016Lx port %016Lx\n", mynode->nodeguid, mynode->portguid);
#endif
	}

	/* Make pass on switches */
	for (dist = 0; dist <= maxhops_discovered; dist++) {

		for (node = nodesdist[dist]; node; node = node->dnext) {

			DEBUG("SWITCH: dist %d node %p", dist, node);
			if (!listtype)
				out_switch(node);
			else {
				if (listtype & SWITCH_NODE)
					list_node(node);
				continue;
			}

			for (port = node->ports; port; port = port->next, i++)
				if (port->remoteport)
					out_switch_port(port);
		}
	}

	/* Make pass on CAs */
	for (node = nodesdist[MAXHOPS]; node; node = node->dnext) {

		DEBUG("CA: dist %d node %p", dist, node);
		if (!listtype)
			out_ca(node);
		else {
			if (listtype & CA_NODE)
				list_node(node);
			continue;
		}

		for (port = node->ports; port; port = port->next, i++)
			if (port->remoteport)
				out_ca_port(port);
	}

	return i;
}

void
usage(void)
{
	fprintf(stderr, "Usage: %s [-d(ebug)] -e(rr_show) -v(erbose) -s(how) -l(ist) -H(ca_list) -S(witch_list) -V(ersion) -C ca_name -P ca_port "
			"-t(imeout) timeout_ms] [<netfile>]\n",
			argv0);
	exit(-1);
}

int
main(int argc, char **argv)
{
	int mgmt_classes[2] = {IB_SMI_CLASS, IB_SMI_DIRECT_CLASS};
	ib_portid_t my_portid = {0};
	extern int ibdebug;
	int udebug = 0, list = 0;
	char *ca = 0;
	int ca_port = 0;

	static char const str_opts[] = "C:P:t:devslHSVhu";
	static const struct option long_opts[] = {
		{ "C", 1, 0, 'C'},
		{ "P", 1, 0, 'P'},
		{ "debug", 0, 0, 'd'},
		{ "err_show", 0, 0, 'e'},
		{ "verbose", 0, 0, 'v'},
		{ "show", 0, 0, 's'},
		{ "list", 0, 0, 'l'},
		{ "Hca_list", 0, 0, 'H'},
		{ "Switch_list", 0, 0, 'S'},
		{ "timeout", 1, 0, 't'},
		{ "Version", 0, 0, 'V'},
		{ "help", 0, 0, 'h'},
		{ "usage", 0, 0, 'u'},
		{ }
	};

	f = stdout;

	argv0 = argv[0];

	while (1) {
		int ch = getopt_long(argc, argv, str_opts, long_opts, NULL);
		if ( ch == -1 )
			break;
		switch(ch) {
		case 'C':
			ca = optarg;
			break;
		case 'P':
			ca_port = strtoul(optarg, 0, 0);
			break;
		case 'd':
			ibdebug++;
			madrpc_show_errors(1);
			umad_debug(udebug);
			udebug++;
			break;
		case 't':
			timeout = strtoul(optarg, 0, 0);
			break;
		case 'v':
			verbose++;
			dumplevel++;
			break;
		case 's':
			dumplevel = 1;
			break;
		case 'e':
			madrpc_show_errors(1);
			break;
		case 'l':
			list = CA_NODE | SWITCH_NODE;
			break;
		case 'S':
			list = SWITCH_NODE;
			break;
		case 'H':
			list = CA_NODE;
			break;
		case 'V':
			fprintf(stderr, "%s %s\n", argv0, get_build_version() );
			exit(-1);
		default:
			usage();
			break;
		}
	}
	argc -= optind;
	argv += optind;

	if (argc)
		if (!(f = fopen(argv[0], "w")))
			IBERROR("can't open file %s for writing", argv[0]);

	madrpc_init(ca, ca_port, mgmt_classes, 2);

	if (discover(&my_portid) < 0)
		IBERROR("discover failed");

	dump_topology(list);

	exit(0);
}
