/*
  This software is available to you under a choice of one of two
  licenses.  You may choose to be licensed under the terms of the GNU
  General Public License (GPL) Version 2, available at
  <http://www.fsf.org/copyleft/gpl.html>, or the OpenIB.org BSD
  license, available in the LICENSE.TXT file accompanying this
  software.  These details are also available at
  <http://openib.org/license.html>.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.

  Copyright (c) 2004 Topspin Communications.  All rights reserved.
  Copyright (c) 2005-2006 Mellanox Technologies Ltd.  All rights reserved.

  $Id: port.c 7871 2006-06-11 08:35:44Z eitan $
*/

/*
 * system includes
 */
#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#define __USE_GNU
#define _GNU_SOURCE				  /* define RTLD_NEXT */
#include <dlfcn.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <stdarg.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <fcntl.h>
/*
 * SDP specific includes
 */
#include "libsdp.h"

/* setsockopt() level and optname declarations */
#define SOL_SDP		1025
#define SDP_UNBIND	259		  /* Unbind socket */

void __attribute__ ( ( constructor ) ) __sdp_init( void );
void __attribute__ ( ( destructor ) ) __sdp_fini( void );

/* --------------------------------------------------------------------- */
/* library type definitions.                                             */
/* --------------------------------------------------------------------- */

typedef int (
	*ioctl_func_t ) (
	int fd,
	int request,
	void *arg0,
	void *arg1,
	void *arg2,
	void *arg3,
	void *arg4,
	void *arg5,
	void *arg6,
	void *arg7 );

typedef int (
	*fcntl_func_t ) (
	int fd,
	int cmd,
	... );

typedef int (
	*socket_func_t ) (
	int domain,
	int type,
	int protocol );

typedef int (
	*setsockopt_func_t ) (
	int s,
	int level,
	int optname,
	const void *optval,
	socklen_t optlen );

typedef int (
	*connect_func_t ) (
	int sockfd,
	const struct sockaddr * serv_addr,
	socklen_t addrlen );

typedef int (
	*listen_func_t ) (
	int s,
	int backlog );

typedef int (
	*bind_func_t ) (
	int sockfd,
	const struct sockaddr * my_addr,
	socklen_t addrlen );

typedef int (
	*close_func_t ) (
	int fd );

typedef int (
	*dup_func_t ) (
	int fd );

typedef int (
	*dup2_func_t ) (
	int oldfd,
	int newfd );

typedef int (
	*getsockname_func_t ) (
	int fd,
	struct sockaddr * name,
	socklen_t * namelen );

typedef int (
	*getpeername_func_t ) (
	int fd,
	struct sockaddr * name,
	socklen_t * namelen );

typedef int (
	*accept_func_t ) (
	int fd,
	struct sockaddr * addr,
	socklen_t * addrlen );

typedef int (
	*select_func_t ) (
	int n,
	fd_set * readfds,
	fd_set * writefds,
	fd_set * exceptfds,
	struct timeval * timeout );

struct socket_lib_funcs
{
	ioctl_func_t ioctl;
	fcntl_func_t fcntl;
	socket_func_t socket;
	setsockopt_func_t setsockopt;
	connect_func_t connect;
	listen_func_t listen;
	bind_func_t bind;
	close_func_t close;
	dup_func_t dup;
	dup2_func_t dup2;
	getpeername_func_t getpeername;
	getsockname_func_t getsockname;
	accept_func_t accept;
	select_func_t select;
};										  /* socket_lib_funcs */

static int simple_sdp_library;
static int max_file_descriptors;
static int dev_null_fd;

/* --------------------------------------------------------------------- */
/* library static and global variables                                   */
/* --------------------------------------------------------------------- */
extern char *program_invocation_name, *program_invocation_short_name;

#ifdef RTLD_NEXT
static void *__libc_dl_handle = RTLD_NEXT;
#else
static void *__libc_dl_handle;
#endif

/* extra fd attributes we need for our algorithms */
struct sdp_extra_fd_attributes
{
	int shadow_fd;					  /* file descriptor of shadow sdp socket    */
	short last_accept_was_tcp;	  /* used by accept to alternate tcp and sdp */
	short is_sdp;					  /* 1 if the fd represents an sdp socket    */
#ifndef SDP_SUPPORTS_IPv6
	short local_is_ipv6;			  /* set to 1 if local address uses IPv6     */
	short remote_is_ipv6;		  /* set to 1 if remote address uses IPv6    */
#endif
};										  /* sdp_extra_fd_attributes */

/* stores the extra attributes struct by fd */
static struct sdp_extra_fd_attributes *libsdp_fd_attributes;

static struct socket_lib_funcs _socket_funcs = {
	.socket = NULL,
	/* Automatically sets all other elements to NULL */
};										  /* _socket_funcs */

/* --------------------------------------------------------------------- */
/* Prototypes                                                            */
/* --------------------------------------------------------------------- */
void __sdp_init(
	void );

/* --------------------------------------------------------------------- */
/*                                                                       */
/* local static functions.                                               */
/*                                                                       */
/* --------------------------------------------------------------------- */

/* ========================================================================= */
/*..init_extra_attribute -- initialize the set of extra attributes for a fd */
static void
init_extra_attribute(
	int fd )
{
	if ( ( 0 <= fd ) && ( max_file_descriptors > fd ) ) {
		libsdp_fd_attributes[fd].shadow_fd = -1;
		libsdp_fd_attributes[fd].is_sdp = 0;
		libsdp_fd_attributes[fd].last_accept_was_tcp = -1;
#ifndef SDP_SUPPORTS_IPv6
		libsdp_fd_attributes[fd].local_is_ipv6 = 0;
		libsdp_fd_attributes[fd].remote_is_ipv6 = 0;
#endif
	}
}

/* ========================================================================= */
/*..get_shadow_fd_by_fd -- given an fd return its shadow fd if exists        */
static inline int
get_shadow_fd_by_fd(
	int fd )
{
	if ( !( 0 > fd ) && ( max_file_descriptors > fd ) )
		return libsdp_fd_attributes[fd].shadow_fd;
	else
		return -1;
}

/* ========================================================================= */
/*..set_shadow_for_fd --                                                     */
static inline void
set_shadow_for_fd(
	int fd,
	int shadow_fd )
{
	if ( !( 0 > fd ) && ( max_file_descriptors > fd ) )
		libsdp_fd_attributes[fd].shadow_fd = shadow_fd;
}

/* ========================================================================= */
/*..set_is_sdp_socket --                                                     */
static inline void
set_is_sdp_socket(
	int fd,
	short is_sdp )
{
	if ( !( 0 > fd ) && ( max_file_descriptors > fd ) )
		libsdp_fd_attributes[fd].is_sdp = is_sdp;
}

/* ========================================================================= */
/*..get_is_sdp_socket -- given an fd return 1 if it is an SDP socket         */
static inline int
get_is_sdp_socket(
	int fd )
{
	if ( !( 0 > fd ) && ( max_file_descriptors > fd ) )
		return libsdp_fd_attributes[fd].is_sdp;
	else
		return 0;
}

/* ========================================================================= */
/*..last_accept_was_tcp -- given an fd return 1 if last accept was tcp       */
static inline int
last_accept_was_tcp(
	int fd )
{
	if ( !( 0 > fd ) && ( max_file_descriptors > fd ) )
		return libsdp_fd_attributes[fd].last_accept_was_tcp;
	else
		return 0;
}

/* ========================================================================= */
/*..set_last_accept -- given an fd set last accept was tcp                   */
static inline void
set_last_accept(
	int fd,
	int was_tcp )
{
	if ( !( 0 > fd ) && ( max_file_descriptors > fd ) )
		libsdp_fd_attributes[fd].last_accept_was_tcp = was_tcp;
}

/* ========================================================================= */
/*..cleanup_shadow -- an error occured on an SDP socket, cleanup             */
static int
cleanup_shadow(
	int fd )
{
	int shadow_fd = get_shadow_fd_by_fd( fd );

	if ( shadow_fd == -1 )
		return 0;
	libsdp_fd_attributes[fd].shadow_fd = -1;
	libsdp_fd_attributes[fd].last_accept_was_tcp = 0;
	init_extra_attribute( shadow_fd );
	return ( _socket_funcs.close( shadow_fd ) );
}										  /* cleanup_shadow */

/* ========================================================================= */
/*..replace_fd_with_its_shadow -- perform all required for such promotion    */
static int
replace_fd_with_its_shadow(
	int fd )
{
	int shadow_fd = libsdp_fd_attributes[fd].shadow_fd;

	if ( shadow_fd == -1 ) {
		__sdp_log( 9, "Error replace_fd_with_its_shadow: no shadow for fd:%d\n",
					  fd );
		return EINVAL;
	}

	/* copy the attributes of the shadow before we clean them up */
	libsdp_fd_attributes[fd] = libsdp_fd_attributes[shadow_fd];
	init_extra_attribute( shadow_fd );
	libsdp_fd_attributes[fd].shadow_fd = -1;
	_socket_funcs.close( fd );
	if ( _socket_funcs.dup2( shadow_fd, fd ) < 0 ) {
		init_extra_attribute( fd );
		_socket_funcs.close( shadow_fd );
		return EINVAL;
	}
	return 0;
}

/* ========================================================================= */
/*..sockaddr_to_sdp -- convert the address to sdp address. return 0 if ok    */
static int
sockaddr_to_sdp(
	const struct sockaddr *addr_in,
	struct sockaddr *addr_out,
	int *was_ipv6 )
{
	const struct sockaddr_in *sin = ( const struct sockaddr_in * )addr_in;
	const struct sockaddr_in6 *sin6 = ( const struct sockaddr_in6 * )addr_in;
	struct sockaddr_in *sout = ( struct sockaddr_in * )addr_out;
	static struct in6_addr ipv4_embedded_addr = {.s6_addr = {0}, };
	char buf[MAX_ADDR_STR_LEN];

	/* currently SDP supports only IPv4 ... */
	if ( !addr_in ) {
		__sdp_log( 9, "Error sockaddr_to_sdp: provided NULL input pointer\n" );
		return -1;
	}
	if ( !addr_out ) {
		__sdp_log( 9, "Error sockaddr_to_sdp: provided NULL output pointer\n" );
		return -1;
	}

	if ( sin->sin_family == AF_INET ) {
		__sdp_log( 1, "sockaddr_to_sdp: Given IPv4\n" );
		memcpy( addr_out, sin, sizeof( struct sockaddr ) );
		/* sout->sin_family = AF_INET_SDP; */
		if ( was_ipv6 )
			*was_ipv6 = 0;
	} else if ( sin6->sin6_family == AF_INET6 ) {
		/* cannot convert IPv6 that is not IPv4 embedding */
		if ( memcmp
			  ( &ipv4_embedded_addr.s6_addr[0], &sin6->sin6_addr.s6_addr[0], 10 )
			  &&
			  ( ( ( sin6->sin6_addr.s6_addr[10] == 0 ) &&
					( sin6->sin6_addr.s6_addr[11] == 0 ) ) ||
				 ( ( sin6->sin6_addr.s6_addr[10] == 0xff ) &&
					( sin6->sin6_addr.s6_addr[11] == 0xff ) ) ) ) {
			__sdp_log( 9,
						  "Error sockaddr_to_sdp: Given IPv6 address that is not an embedded IPv4\n" );
			return -1;
		}
		__sdp_log( 1, "sockaddr_to_sdp: Given IPv4 embedded in IPv6\n" );
		memset( addr_out, 0, sizeof( struct sockaddr ) );
		sout->sin_family = AF_INET;
		sout->sin_port = sin6->sin6_port;
		memcpy( &sout->sin_addr, &( sin6->sin6_addr.s6_addr[12] ), 4 );
		if ( inet_ntop( sout->sin_family, ( void * )&( sout->sin_addr ),
							 buf, MAX_ADDR_STR_LEN ) == NULL ) {
			__sdp_log( 1,
						  "sockaddr_to_sdp: Converted IPv4 address is illegal\n" );
			return 1;
		}
		__sdp_log( 1, "sockaddr_to_sdp: Converted IPv4 is:%s\n", buf );
		/* sout->sin_family = AF_INET_SDP; */
		if ( was_ipv6 )
			*was_ipv6 = 1;
	} else if ( sin->sin_family == 0 ) {
		__sdp_log( 1, "sockaddr_to_sdp: Converted NULL address\n" );
		memcpy( addr_out, addr_in, sizeof( struct sockaddr ) );
	} else {
		__sdp_log( 9, "Error sockaddr_to_sdp: address family <%d> is unknown\n",
					  sin->sin_family );
		return 1;
	}
	return 0;
}

/* ========================================================================= */
/*..sockaddr_from_sdp -- convert address from sdp address to original one    */
static int
sockaddr_from_sdp(
	int to_ipv6,
	const struct sockaddr *addr_in,
	struct sockaddr *addr_out )
{
	const struct sockaddr_in *sin = ( const struct sockaddr_in * )addr_in;
	struct sockaddr_in *sout = ( struct sockaddr_in * )addr_out;
	struct sockaddr_in6 *sout6 = ( struct sockaddr_in6 * )addr_out;

	if ( !addr_out ) {
		__sdp_log( 9,
					  "Error sockaddr_from_sdp: provided NULL output pointer\n" );
		return -1;
	}

	/* TODO: currently SDP supports only IPv4, when IPv6 supported just change family */
	if ( !to_ipv6 ) {
		__sdp_log( 1, "sockaddr_from_sdp: Converting to IPv4\n" );
		memcpy( addr_out, sin, sizeof( struct sockaddr ) );
		sout->sin_family = AF_INET;
	} else {
		__sdp_log( 1,
					  "sockaddr_from_sdp: Converting to IPv6 (embedded IPv4)\n" );
		memset( addr_out, 0, sizeof( struct sockaddr ) );
		sout6->sin6_family = AF_INET6;
		sout6->sin6_port = sin->sin_port;
		memcpy( &( sout6->sin6_addr.s6_addr[12] ), &sin->sin_addr, 4 );
	}
	return 0;
}

/* ========================================================================= */
/*..is_filtered_unsuported_sockopt -- return 1 if to filter sockopt failure  */
static inline int
is_filtered_unsuported_sockopt(
	int level,
	int optname )
{
	/* 
	 * TODO: until we know exactly which unsupported opts are really 
	 * a don't care we always pass the error
	 */
	return 0;
#if 0
	/* these are the SOL_TCP OPTS we should consider filterring */
	TCP_NODELAY 1					  /* Don't delay send to coalesce packets  */
		TCP_MAXSEG 2				  /* Set maximum segment size  */
		TCP_CORK 3					  /* Control sending of partial frames  */
		TCP_KEEPIDLE 4				  /* Start keeplives after this period */
		TCP_KEEPINTVL 5			  /* Interval between keepalives */
		TCP_KEEPCNT 6				  /* Number of keepalives before death */
		TCP_SYNCNT 7				  /* Number of SYN retransmits */
		TCP_LINGER2 8				  /* Life time of orphaned FIN-WAIT-2 state */
		TCP_DEFER_ACCEPT 9		  /* Wake up listener only when data arrive */
		TCP_WINDOW_CLAMP 10		  /* Bound advertised window */
		TCP_INFO 11					  /* Information about this connection. */
		TCP_QUICKACK 12			  /* Bock/reenable quick ACKs.  */
#endif
}

/* ========================================================================= */
/*..is_invalid_addr -- return 1 if given pointer is not valid                */
/* NOTE: invalidation of the size is going to happen during actual call      */
static inline int
is_invalid_addr(
	const void *p )
{
	/* HACK: on some systems we can not write to check for pointer validity */
	ssize_t ret = write( dev_null_fd, p, 1 );

	return ( ret == -1 ? errno == EFAULT : 0 );
}

/* --------------------------------------------------------------------- */
/*                                                                       */
/* Socket library function overrides.                                    */
/*                                                                       */
/* --------------------------------------------------------------------- */

/* ========================================================================= */
/*..ioctl -- replacement ioctl call. */
int
ioctl(
	int fd,
	int request,
	void *arg0,
	void *arg1,
	void *arg2,
	void *arg3,
	void *arg4,
	void *arg5,
	void *arg6,
	void *arg7 )
{
	int shadow_fd = get_shadow_fd_by_fd( fd );
	int sret = 0;
	int ret = 0;

	if ( NULL == _socket_funcs.ioctl ) {
		__sdp_log( 9, "Error ioctl: no implementation for ioctl found\n" );
		return -1;
	}

	__sdp_log( 2, "IOCTL: <%s:%d:%d> request <%d>\n",
				  program_invocation_short_name, fd, shadow_fd, request );

	ret = _socket_funcs.ioctl( fd, request,
										arg0, arg1, arg2, arg3, arg4, arg5, arg6,
										arg7 );

	/* if shadow and no error on tcp */
	if ( !( 0 > ret ) && ( -1 != shadow_fd ) ) {
		sret = _socket_funcs.ioctl( shadow_fd, request,
											 arg0, arg1, arg2, arg3, arg4, arg5, arg6,
											 arg7 );
		if ( sret < 0 ) {
			__sdp_log( 9,
						  "Error ioctl: <%d> calling ioctl for SDP socket, closing it.\n",
						  errno );
			cleanup_shadow( fd );
		}
	}

	__sdp_log( 2, "IOCTL: <%s:%d:%d> result <%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd, ret, sret );

	return ret;
}										  /* ioctl */

/* ========================================================================= */
/*..fcntl -- replacement fcntl call.                                         */
int
fcntl(
	int fd,
	int cmd,
	... )
{
	int shadow_fd = get_shadow_fd_by_fd( fd );
	int sret = 0;
	int ret = 0;

	void *arg;
	va_list ap;

	va_start( ap, cmd );
	arg = va_arg( ap, void * );

	va_end( ap );


	if ( NULL == _socket_funcs.fcntl ) {
		__sdp_log( 9, "Error fcntl: no implementation for fcntl found\n" );
		return -1;
	}

	__sdp_log( 2, "FCNTL: <%s:%d:%d> command <%d> argument <%d>\n",
				  program_invocation_short_name, fd, shadow_fd, cmd, 0 );

	ret = _socket_funcs.fcntl( fd, cmd, arg );
	if ( !( 0 > ret ) && ( -1 != shadow_fd ) ) {
		sret = _socket_funcs.fcntl( shadow_fd, cmd, arg );
		if ( sret < 0 ) {
			__sdp_log( 9,
						  "Error fcntl: <%d> calling fcntl(%d, %d, %x) for SDP socket. Closing it.\n",
						  shadow_fd, cmd, 0, errno );
			cleanup_shadow( fd );
		}
	}

	__sdp_log( 2, "FCNTL: <%s:%d:%d> result <%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd, ret, sret );

	return ret;
}										  /* fcntl */

/* ========================================================================= */
/*..setsockopt -- replacement setsockopt call.                               */
int
setsockopt(
	int fd,
	int level,
	int optname,
	const void *optval,
	socklen_t optlen )
{
	int shadow_fd = get_shadow_fd_by_fd( fd );
	int sret = 0;
	int ret = 0;

	if ( NULL == _socket_funcs.setsockopt ) {
		__sdp_log( 9,
					  "Error setsockopt: no implementation for setsockopt found\n" );
		return -1;
	}

	__sdp_log( 2, "SETSOCKOPT: <%s:%d:%d> level <%d> name <%d>\n",
				  program_invocation_short_name, fd, shadow_fd, level, optname );

	ret = _socket_funcs.setsockopt( fd, level, optname, optval, optlen );
	if ( !( 0 > ret ) && ( shadow_fd != -1 ) ) {
		sret = _socket_funcs.setsockopt( shadow_fd, level,
													optname, optval, optlen );
		if ( sret < 0 ) {
			__sdp_log( 8,
						  "Warning sockopts: ignoring error on shadow SDP socket fd:<%d>\n",
						  fd );
			/* 
			 * HACK: we should allow some errors as some sock opts are unsupported  
			 * __sdp_log(9, "Error %d calling setsockopt for SDP socket, closing\n", errno); 
			 * cleanup_shadow(fd); 
			 */
		}
	}

	/* Due to SDP limited implmentation of sockopts we ignore some errors */
	if ( ( ret < 0 ) && get_is_sdp_socket( fd ) &&
		  is_filtered_unsuported_sockopt( level, optname ) ) {
		__sdp_log( 8,
					  "Warning sockopts: ignoring error on non implemented sockopt on SDP socket fd:<%d> level:<%d> opt:<%d>\n",
					  fd, level, optval );
		ret = 0;
	}

	__sdp_log( 2, "SETSOCKOPT: <%s:%d:%d> result <%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd, ret, sret );

	return ret;
}										  /* setsockopt */

/* ========================================================================= */
/*..socket -- replacement socket call.                                       */
int
socket(
	int domain,
	int type,
	int protocol )
{
	int s = -1;
	int shadow_fd = -1;
	use_family_t family_by_prog;

	if ( NULL == _socket_funcs.socket ) {
		__sdp_log( 9, "Error socket: no implementation for socket found\n" );
		return -1;
	}

	__sdp_log( 2, "SOCKET: <%s> domain <%d> type <%d> protocol <%d>\n",
				  program_invocation_short_name, domain, type, protocol );

	/* check to see if we can skip the shadow */
	if ( ( AF_INET == domain || AF_INET6 == domain ) &&
		  ( SOCK_STREAM == type ) )
		if ( simple_sdp_library )
			family_by_prog = USE_SDP;
		else
			family_by_prog = __sdp_match_by_program(  );
	else
		family_by_prog = USE_TCP;

	if ( family_by_prog == USE_TCP ) {
		__sdp_log( 1, "SOCKET: making TCP only socket (no shadow)\n" );
		s = _socket_funcs.socket( domain, type, protocol );
		set_is_sdp_socket( s, 0 );
		goto done;
	} else if ( family_by_prog == USE_SDP ) {
		/* HACK: convert the protocol if IPPROTO_IP */
		if ( protocol == 0 )
			protocol = IPPROTO_TCP;

		__sdp_log( 1, "SOCKET: making SDP socket type:%d proto:%d\n",
					  type, protocol );
		s = _socket_funcs.socket( AF_INET_SDP, type, protocol );
		set_is_sdp_socket( s, 1 );
		goto done;
	}

	/* HACK: if we fail creating the TCP socket should we abort ? */
	__sdp_log( 1, "SOCKET: making TCP socket\n" );
	s = _socket_funcs.socket( domain, type, protocol );
	set_is_sdp_socket( s, 0 );
	if ( !( 0 > s ) && ( max_file_descriptors > s ) ) {
		if ( ( ( AF_INET == domain ) || ( AF_INET6 == domain ) ) &&
			  ( SOCK_STREAM == type ) ) {

			if ( protocol == 0 )
				protocol = IPPROTO_TCP;
			__sdp_log( 1, "SOCKET: making SDP shadow socket type:%d proto:%d\n",
						  type, protocol );
			shadow_fd = _socket_funcs.socket( AF_INET_SDP, type, protocol );
			if ( shadow_fd < 0 ) {
				__sdp_log( 9,
							  "Error socket: <%d> calling socket for SDP socket\n",
							  errno );
				/* fail if we did not make the SDP socket */
				__sdp_log( 1, "SOCKET: closing TCP socket:<%d>\n", s );
				_socket_funcs.close( s );
				init_extra_attribute( s );
				s = -1;
			} else {
				if ( libsdp_fd_attributes[s].shadow_fd != -1 ) {
					__sdp_log( 8,
								  "Warning socket: overriding existing shadow fd:%d for fd:%d\n",
								  libsdp_fd_attributes[s].shadow_fd, s );
				}
				set_is_sdp_socket( shadow_fd, 1 );
				set_shadow_for_fd( s, shadow_fd );
			}
		}
	} else {
		__sdp_log( 9,
					  "Error socket: ignoring SDP socket since TCP fd:%d out of range\n",
					  s );
	}

 done:
	__sdp_log( 2, "SOCKET: <%s:%d:%d>\n",
				  program_invocation_short_name, s, shadow_fd );

	return s;
}										  /* socket */

/* ========================================================================= */
/*..bind -- replacement bind call.                                           */
/* 
   As we do not know the role of this socket yet so we cannot choose AF. 
   We need to be able to handle shadow too.
   SDP sockets (may be shadow or not) must be using converted address 
   We error on the non shadow bind failure only.
   
   HACK: If given a ANY_PORT (0) bind is assigning a free one. What if that port 
   is free for TCP but busy for SDP? Since there is no simple way to "re-bind" 
   the TCP port without closing it first we currently just fail the bind.
*/
int
bind(
	int fd,
	const struct sockaddr *my_addr,
	socklen_t addrlen )
{
	int shadow_fd = get_shadow_fd_by_fd( fd );
	struct sockaddr_in *my_sin = ( struct sockaddr_in * )my_addr;
	struct sockaddr sdp_addr;
	struct sockaddr_in *sdp_sin = ( struct sockaddr_in * )&sdp_addr;
	int ret, sret = -1;
	char buf[MAX_ADDR_STR_LEN];
	struct sockaddr_in tmp_sin;
	socklen_t tmp_sinlen = sizeof( tmp_sin );
	int was_ipv6;

	if ( NULL == _socket_funcs.bind ) {
		__sdp_log( 9, "Error bind: no implementation for bind found\n" );
		return -1;
	}

	if ( ( my_addr == NULL ) || is_invalid_addr( my_addr ) ) {
		errno = EFAULT;
		__sdp_log( 9, "Error bind: illegal address provided\n" );
		return -1;
	}

	if ( ( my_sin->sin_family == AF_INET ) ||
		  ( my_sin->sin_family == AF_INET6 ) )
		if ( inet_ntop
			  ( my_sin->sin_family, ( void * )&( my_sin->sin_addr ), buf,
				 MAX_ADDR_STR_LEN ) == NULL ) {
			__sdp_log( 9, "Error bind: provided illegal address: %s\n",
						  strerror( errno ) );
			return -1;
		}

	__sdp_log( 2, "BIND: <%s:%d:%d> type <%d> IP <%s> port <%d>\n",
				  program_invocation_short_name, fd, shadow_fd,
				  my_sin->sin_family, buf, ntohs( my_sin->sin_port ) );

	if ( get_is_sdp_socket( fd ) ) {
		if ( sockaddr_to_sdp( my_addr, &sdp_addr, &was_ipv6 ) ) {
			__sdp_log( 9, "Error bind: failed to convert address:%s for SDP\n",
						  buf );
			ret = EADDRNOTAVAIL;
			goto done;
		}
#ifndef SDP_SUPPORTS_IPv6
		if ( was_ipv6 )
			libsdp_fd_attributes[fd].local_is_ipv6 = 1;
#endif
		__sdp_log( 1, "BIND: binding SDP socket:<%d>\n", fd );
		ret = _socket_funcs.bind( fd, &sdp_addr, sizeof sdp_addr );
	} else {
		__sdp_log( 1, "BIND: binding TCP socket:<%d>\n", fd );
		ret = _socket_funcs.bind( fd, my_addr, addrlen );
	}

	/* if we do not have a shadow we are done */
	if ( shadow_fd == -1 )
		goto done;

	/* we have a shadow so go ahead and get the address converted to sdp */
	if ( sockaddr_to_sdp( my_addr, &sdp_addr, &was_ipv6 ) ) {
		__sdp_log( 9,
					  "Error bind: failed to convert address:%s for shadow SDP\n",
					  buf );
		ret = EADDRNOTAVAIL;
		goto done;
	}
#ifndef SDP_SUPPORTS_IPv6
	if ( was_ipv6 )
		libsdp_fd_attributes[fd].local_is_ipv6 = 1;
#endif

	/* if the main fd failed we only later dup the shadow into it and cleanup the shadow */
	if ( ret < 0 ) {
		__sdp_log( 1, "BIND: failed to bind TCP fd:<%d>\n", fd );
	} else {
		/* if the tcp socket was bind to port 0 we should get its address into the sdp_sin */
		if ( ntohs( sdp_sin->sin_port ) == 0 ) {
			/* need to get the port assigned to the tcp socket */
			if ( _socket_funcs.
				  getsockname( fd, ( struct sockaddr * )&tmp_sin,
									&tmp_sinlen ) < 0 ) {
				__sdp_log( 9,
							  "Error bind: getsockname return <%d> for TCP socket, failing the SDP bind\n",
							  errno );
				sret = EADDRNOTAVAIL;
				goto done;
			}
			__sdp_log( 1, "BIND: using TCP assigned port:<%d>\n",
						  ntohs( tmp_sin.sin_port ) );
			sdp_sin->sin_port = tmp_sin.sin_port;
		}
	}

	/* bind the SDP socket */
	__sdp_log( 1, "BIND: binding SDP socket:<%d>\n", shadow_fd );
	sret = _socket_funcs.bind( shadow_fd, &sdp_addr, sizeof sdp_addr );
	if ( sret < 0 ) {
		__sdp_log( 9,
					  "Error bind: return code <%d> for SDP socket, Closing shadow\n",
					  errno );
		cleanup_shadow( fd );
		goto done;
	}

	/* we might need to dup the SDP socket into the original fd if the original fail */
	if ( ret < 0 ) {
		__sdp_log( 1, "BIND: replacing TCP socket:<%d> by its shadow:<%d>\n",
					  fd, shadow_fd );
		ret = replace_fd_with_its_shadow( fd );
		if ( ret < 0 ) {
			__sdp_log( 9,
						  "Error: during bind fail to dup2 shadow into orig fd:%d\n",
						  fd );
			goto done;
		}
	}

 done:
	__sdp_log( 2, "BIND: <%s:%d:%d> result <%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd, ret, sret );

	return ret;
}										  /* bind */

/* ========================================================================= */
/*..connect -- replacement connect call.                                     */
/*
  Given the connect address we can take out AF decision                     
  if target AF == both it means SDP and fall back to TCP                   
  if any connect worked we are fine
*/
int
connect(
	int fd,
	const struct sockaddr *serv_addr,
	socklen_t addrlen )
{
	struct sockaddr_in *serv_sin = ( struct sockaddr_in * )serv_addr;
	struct sockaddr sdp_addr;
	char buf[MAX_ADDR_STR_LEN];
	int shadow_fd = get_shadow_fd_by_fd( fd );
	int ret = -1;
	int was_ipv6;
	use_family_t target_family;

	if ( NULL == _socket_funcs.connect ) {
		__sdp_log( 9, "Error connect: no implementation for connect found\n" );
		return -1;
	}

	if ( ( serv_addr == NULL ) || is_invalid_addr( serv_addr ) ) {
		errno = EFAULT;
		__sdp_log( 9, "Error connect: illegal address provided\n" );
		return -1;
	}

	if ( ( serv_sin->sin_family == AF_INET ) ||
		  ( serv_sin->sin_family == AF_INET6 ) ) {
		if ( inet_ntop
			  ( serv_sin->sin_family, ( void * )&( serv_sin->sin_addr ), buf,
				 MAX_ADDR_STR_LEN ) == NULL ) {
			__sdp_log( 9, "Error connect: provided illegal address: %s\n",
						  strerror( errno ) );
			return EADDRNOTAVAIL;
		}
	}

	__sdp_log( 2, "CONNECT: <%s:%d:%d> type <%d> IP <%s> port <%d>\n",
				  program_invocation_short_name, fd, shadow_fd,
				  serv_sin->sin_family, buf, ntohs( serv_sin->sin_port ) );

	/* if we do not have a shadow - just do the work */
	if ( shadow_fd == -1 ) {
		if ( get_is_sdp_socket( fd ) ) {
			if ( sockaddr_to_sdp( serv_addr, &sdp_addr, &was_ipv6 ) ) {
				__sdp_log( 9,
							  "Error connect: failed to convert address:%s to SDP\n",
							  buf );
				ret = EADDRNOTAVAIL;
				goto done;
			}
#ifndef SDP_SUPPORTS_IPv6
			if ( was_ipv6 )
				libsdp_fd_attributes[fd].remote_is_ipv6 = 1;
#endif
			__sdp_log( 1, "CONNECT: connecting through SDP\n" );
			ret = _socket_funcs.connect( fd, &sdp_addr, sizeof sdp_addr );
		} else {
			__sdp_log( 1, "CONNECT: connecting through TCP\n" );
			ret = _socket_funcs.connect( fd, serv_addr, addrlen );
		}
		goto done;
	}

	/* obtain the target address family */
	target_family = __sdp_match_connect( serv_sin );

	if ( ( target_family == USE_SDP ) || ( target_family == USE_BOTH ) ) {
		/* NOTE: the entire if sequence is negative logic */
		if ( sockaddr_to_sdp( serv_addr, &sdp_addr, &was_ipv6 ) ) {
			__sdp_log( 9,
						  "Error connect: failed to convert to shadow address:%s to SDP\n",
						  buf );
			ret = EADDRNOTAVAIL;
		} else {
			__sdp_log( 1, "CONNECT: connecting SDP fd:%d\n", shadow_fd );
			ret = _socket_funcs.connect( shadow_fd, &sdp_addr, sizeof sdp_addr );
			if ( ret < 0 ) {
				__sdp_log( 9,
							  "Error connect: failed with code <%d> for SDP fd:%d\n",
							  ret, shadow_fd );
			}
#ifndef SDP_SUPPORTS_IPv6
			if ( was_ipv6 )
				libsdp_fd_attributes[fd].remote_is_ipv6 = 1;
#endif
		}

		/* if target is SDP or we succeeded we need to dup into tcp */
		if ( ( target_family == USE_SDP ) || ( ret >= 0 ) ) {
			ret = replace_fd_with_its_shadow( fd );
			if ( ret < 0 ) {
				__sdp_log( 9,
							  "Error connect: failed to dup2 shadow into orig fd:%d\n",
							  fd );
			} else {
				/* we can skip the TCP option if we are done */
				__sdp_log( 1,
							  "CONNECT: connected SDP fd:%d (shadow dup into it)\n",
							  fd );
				goto done;
			}
		}
	}

	if ( ( target_family == USE_TCP ) || ( target_family == USE_BOTH ) ) {
		if ( cleanup_shadow( fd ) < 0 )
			__sdp_log( 9, "Error connect: failed to cleanup shadow for fd:%d\n",
						  fd );

		__sdp_log( 1, "CONNECT: connecting TCP fd:%d\n", fd );
		ret = _socket_funcs.connect( fd, serv_addr, addrlen );
		if ( ret < 0 )
			__sdp_log( 9, "Error connect: failed with code <%d> for TCP fd:%d\n",
						  ret, fd );
		else
			__sdp_log( 1, "CONNECT: connected TCP fd:%d\n", fd );
		goto done;
	}
 done:
	__sdp_log( 2, "CONNECT: <%s:%d:%d> result <%d>\n",
				  program_invocation_short_name, fd, shadow_fd, ret );

	return ret;
}										  /* connect */

/* ========================================================================= */
/*..listen -- replacement listen call.                                       */
/* 
   Now we know our role (passive/server) and our address so we can get AF.
   If both we should try listening on both
*/
int
listen(
	int fd,
	int backlog )
{
	use_family_t target_family;
	int shadow_fd = get_shadow_fd_by_fd( fd );
	int ret = 0, sret = 0;
	struct sockaddr_in tmp_sin;
	socklen_t tmp_sinlen = sizeof( tmp_sin );

	if ( NULL == _socket_funcs.listen ) {
		__sdp_log( 9, "Error listen: no implementation for listen found\n" );
		return -1;
	}

	__sdp_log( 2, "LISTEN: <%s:%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd );

	/* if there is no shadow - simply call listen */
	if ( shadow_fd == -1 ) {
		__sdp_log( 1, "LISTEN: calling listen on fd:%d\n", fd );
		ret = _socket_funcs.listen( fd, backlog );
		goto done;
	}

	/* we need to obtain the address from the fd */
	if ( _socket_funcs.
		  getsockname( fd, ( struct sockaddr * )&tmp_sin, &tmp_sinlen ) < 0 ) {
		__sdp_log( 9, "Error listen: getsockname return <%d> for TCP socket\n",
					  errno );
		sret = EADDRNOTAVAIL;
		goto done;
	}

	target_family = __sdp_match_listen( &tmp_sin );

	if ( ( target_family == USE_TCP ) || ( target_family == USE_BOTH ) ) {
		__sdp_log( 1, "LISTEN: calling listen on TCP fd:%d\n", fd );
		ret = _socket_funcs.listen( fd, backlog );
		if ( ret < 0 ) {
			__sdp_log( 9, "Error listen: failed with code <%d> on TCP fd:<%d>\n",
						  errno, fd );
		} else {
			__sdp_log( 1, "LISTEN: listening on TCP\n" );
		}
	}

	if ( ( target_family == USE_SDP ) || ( target_family == USE_BOTH ) ) {
		__sdp_log( 1, "LISTEN: calling listen on SDP fd:<%d>\n", shadow_fd );
		sret = _socket_funcs.listen( shadow_fd, backlog );
		if ( sret < 0 ) {
			__sdp_log( 9, "Error listen: failed with code <%d> SDP fd:<%d>\n",
						  errno, shadow_fd );
		} else {
			__sdp_log( 1, "LISTEN: listening on SDP\n" );
		}
	}

	/* cleanup the un-needed shadow if TCP and did not fail */
	if ( ( target_family == USE_TCP ) && !( ret < 0 ) ) {
		__sdp_log( 1, "LISTEN: cleaning up shadow SDP\n" );
		if ( cleanup_shadow( fd ) < 0 )
			__sdp_log( 9, "Error listen: failed to cleanup shadow for fd:%d\n",
						  fd );
	}

	/* cleanup the TCP socket and replace with SDP */
	if ( ( target_family == USE_SDP ) && !( sret < 0 ) ) {
		__sdp_log( 1, "LISTEN: cleaning TCP socket and dup2 SDP into it\n" );
		if ( 0 > ( sret = replace_fd_with_its_shadow( fd ) ) )
			__sdp_log( 9,
						  "Error listen: failed to dup2 shadow into orig fd:%d\n",
						  fd );
	}

 done:
	__sdp_log( 2, "LISTEN: <%s:%d:%d> result <%d>\n",
				  program_invocation_short_name, fd, shadow_fd, ret );
	/* its a success only if both are ok */
	if ( ret < 0 )
		return ( ret );
	if ( sret < 0 )
		return ( sret );
	return 0;
}										  /* listen */

/* ========================================================================= */
/*..close -- replacement close call. */
int
close(
	int fd )
{
	int shadow_fd = get_shadow_fd_by_fd( fd );
	int ret;

	if ( NULL == _socket_funcs.close ) {
		__sdp_log( 9, "Error close: no implementation for close found\n" );
		return -1;
	}

	__sdp_log( 2, "CLOSE: <%s:%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd );

	if ( shadow_fd != -1 ) {
		__sdp_log( 1, "CLOSE: closing shadow fd:<%d>\n", shadow_fd );
		if ( cleanup_shadow( fd ) < 0 )
			__sdp_log( 9, "Error close: failed to cleanup shadow for fd:%d\n",
						  fd );
	}

	ret = _socket_funcs.close( fd );
	init_extra_attribute( fd );
	__sdp_log( 2, "CLOSE: <%s:%d:%d> result <%d>\n",
				  program_invocation_short_name, fd, shadow_fd, ret );
	return ret;
}										  /* close */

/* ========================================================================= */
/*..dup -- replacement dup call.                                             */
/* we duplicate the fd and its shadow if exists - ok if the main worked      */
int
dup(
	int fd )
{
	int newfd, new_shadow_fd = -1;
	int shadow_fd = get_shadow_fd_by_fd( fd );

	if ( NULL == _socket_funcs.dup ) {
		__sdp_log( 9, "Error dup: no implementation for dup found\n" );
		return -1;
	}

	__sdp_log( 2, "DUP: <%s:%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd );

	__sdp_log( 1, "DUP: duplication fd:<%d>\n", fd );
	newfd = _socket_funcs.dup( fd );
	if ( ( newfd > max_file_descriptors ) || ( newfd < 0 ) ) {
		__sdp_log( 9, "Error dup: new fd <%d> out of range.\n", newfd );
	} else {
		if ( ( newfd != fd ) && ( fd != 0 ) ) {
			/* copy attributes from old fd */
			libsdp_fd_attributes[newfd] = libsdp_fd_attributes[fd];
			libsdp_fd_attributes[newfd].shadow_fd = -1;

			if ( shadow_fd != -1 ) {
				__sdp_log( 1, "DUP: duplication shadow fd:<%d>\n", shadow_fd );
				new_shadow_fd = _socket_funcs.dup( shadow_fd );
				if ( ( new_shadow_fd > max_file_descriptors ) ||
					  ( new_shadow_fd < 0 ) ) {
					__sdp_log( 9, "Error dup: new shadow fd <%d> out of range.\n",
								  new_shadow_fd );
				} else {
					libsdp_fd_attributes[new_shadow_fd] =
						libsdp_fd_attributes[shadow_fd];
					libsdp_fd_attributes[newfd].shadow_fd = new_shadow_fd;
				}
			}							  /* shadow exists */
		}								  /* newfd is ok */
	}

	__sdp_log( 2, "DUP: <%s:%d:%d> return <%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd, newfd,
				  new_shadow_fd );

	return newfd;
}										  /* dup */

/* ========================================================================= */
/*..dup2 -- replacement dup2 call.                                           */
/* since only the main new fd is given we only move the shadow if exists     */
int
dup2(
	int fd,
	int newfd )
{
	int shadow_fd = get_shadow_fd_by_fd( fd );
	int new_shadow_fd = -1;

	if ( NULL == _socket_funcs.dup2 ) {
		__sdp_log( 9, "Error dup2: no implementation for dup2 found\n" );
		return -1;
	}

	__sdp_log( 2, "DUP2: <%s:%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd );

	__sdp_log( 1, "DUP2: duplicating fd:<%d> into:<%d>\n", fd );
	newfd = _socket_funcs.dup2( fd, newfd );
	if ( ( newfd > max_file_descriptors ) || ( newfd < 0 ) ) {
		__sdp_log( 9, "Error dup2: new fd <%d> out of range.\n", newfd );
	} else {
		if ( ( newfd != fd ) && ( fd != 0 ) ) {
			/* copy attributes from old fd */
			libsdp_fd_attributes[newfd] = libsdp_fd_attributes[fd];
			libsdp_fd_attributes[newfd].shadow_fd = -1;

			/* if it had a shadow create a new shadow */
			if ( shadow_fd != -1 ) {
				__sdp_log( 1, "DUP2: duplication shadow fd:<%d>\n", shadow_fd );
				new_shadow_fd = _socket_funcs.dup( shadow_fd );
				if ( ( new_shadow_fd > max_file_descriptors ) ||
					  ( new_shadow_fd < 0 ) ) {
					__sdp_log( 9, "Error dup2: new shadow fd <%d> out of range.\n",
								  new_shadow_fd );
				} else {
					libsdp_fd_attributes[new_shadow_fd] =
						libsdp_fd_attributes[shadow_fd];
					libsdp_fd_attributes[newfd].shadow_fd = new_shadow_fd;
				}
			}
		}								  /* newfd is ok */
	}

	__sdp_log( 2, "DUP2: <%s:%d:%d> return <%d:%d>\n",
				  program_invocation_short_name, fd, shadow_fd, newfd,
				  new_shadow_fd );

	return newfd;
}										  /* dup */

/* ========================================================================= */
/*..getsockname -- replacement getsocknanme call.                            */
int
getsockname(
	int fd,
	struct sockaddr *name,
	socklen_t * namelen )
{
	int ret = 0;
	int to_ipv6 = 0;
	struct sockaddr tmp_addr;

	/*
	 * ensure the SDP protocol family is not exposed to the user, since
	 * this is meant to be a transparency layer.
	 */
	if ( NULL == _socket_funcs.getsockname ) {
		__sdp_log( 9,
					  "Error getsockname: no implementation for getsockname found\n" );
		return -1;
	}

	__sdp_log( 2, "GETSOCKNAME <%s:%d>\n", program_invocation_short_name, fd );

	ret = _socket_funcs.getsockname( fd, name, namelen );

	/* we need address conversion - only if socket is sdp */
	if ( ( ret > 0 ) && get_is_sdp_socket( fd ) && name && namelen ) {

#ifndef SDP_SUPPORTS_IPv6
		to_ipv6 = libsdp_fd_attributes[fd].local_is_ipv6;
#endif

		memcpy( &tmp_addr, name, sizeof( struct sockaddr ) );
		/* HACK: need to make sure we convert into IPv6 if original was IPv6 */
		sockaddr_from_sdp( to_ipv6, &tmp_addr, name );
	}

	__sdp_log( 2, "GETSOCKNAME <%s:%d> result <%d>\n",
				  program_invocation_short_name, fd, ret );

	return ret;
}										  /* getsockname */

/* ========================================================================= */
/*..getpeername -- replacement getpeername call. */
int
getpeername(
	int fd,
	struct sockaddr *name,
	socklen_t * namelen )
{
	int ret = 0;
	int to_ipv6 = 0;
	struct sockaddr tmp_addr;

	if ( NULL == _socket_funcs.getpeername ) {
		__sdp_log( 9,
					  "Error getpeername: no implementation for getpeername found\n" );
		return -1;
	}

	__sdp_log( 2, "GETPEERNAME <%s:%d>\n", program_invocation_short_name, fd );

	ret = _socket_funcs.getpeername( fd, name, namelen );
	/* we need address conversion - only if socket is sdp */
	if ( ( ret > 0 ) && get_is_sdp_socket( fd ) && name && namelen ) {
#ifndef SDP_SUPPORTS_IPv6
		to_ipv6 = libsdp_fd_attributes[fd].remote_is_ipv6;
#endif

		memcpy( &tmp_addr, name, sizeof( struct sockaddr ) );
		/* HACK: need to make sure we convert into IPv6 if original was IPv6 */
		sockaddr_from_sdp( to_ipv6, &tmp_addr, name );
	}

	__sdp_log( 2, "GETPEERNAME <%s:%d> result <%d:%d> family=%d s_addr=%d\n",
				  program_invocation_short_name, fd, ret,
				  ( !( 0 > ret ) ? 0 : -1 ), name->sa_family,
				  ( ( struct sockaddr_in * )name )->sin_addr.s_addr );

	return ret;
}										  /* getpeername */



/* ========================================================================= */
/*..accept -- replacement accept call.                                       */
/*
  If we have a shadow we need to decide which socket we want to accept on
  so we select first and then give priority based on previous selection
*/
int
accept(
	int fd,
	struct sockaddr *addr,
	socklen_t * addrlen )
{
	int shadow_fd = get_shadow_fd_by_fd( fd );
	int ret = 0;
	fd_set fds;
	struct sockaddr tmp_addr;
	int to_ipv6 = 0;

	/*
	 * ensure the SDP protocol family is not exposed to the user, since
	 * this is meant to be a transparency layer.
	 */
	if ( NULL == _socket_funcs.accept ) {
		__sdp_log( 9, "Error accept: no implementation for accept found\n" );
		return -1;
	}

	__sdp_log( 2, "ACCEPT: <%s:%d>\n", program_invocation_short_name, fd );

	if ( shadow_fd == -1 ) {
		__sdp_log( 1, "ACCEPT: accepting on fd:<%d>\n", fd );
		ret = _socket_funcs.accept( fd, addr, addrlen );
	} else {

#ifndef SDP_SUPPORTS_IPv6
		to_ipv6 = libsdp_fd_attributes[fd].local_is_ipv6;
#endif

		FD_ZERO( &fds );
		FD_SET( fd, &fds );
		FD_SET( shadow_fd, &fds );
		__sdp_log( 1, "ACCEPT: selecting both fd:<%d> and shadow:<%d>\n", fd,
					  shadow_fd );
		ret =
			_socket_funcs.select( 1 + ( ( fd > shadow_fd ) ? fd : shadow_fd ),
										 &fds, NULL, NULL, NULL );
		if ( ret >= 0 ) {
			if ( last_accept_was_tcp( fd ) == 0 ) {
				if ( FD_ISSET( fd, &fds ) ) {
					set_last_accept( fd, 1 );
					__sdp_log( 1, "ACCEPT: accepting on TCP fd:<%d>\n", fd );
					ret = _socket_funcs.accept( fd, addr, addrlen );
				} else {
					__sdp_log( 1, "ACCEPT: accepting on SDP fd:<%d>\n",
								  shadow_fd );
					ret = _socket_funcs.accept( shadow_fd, addr, addrlen );
					if ( ( ret > 0 ) && addr && addrlen ) {
						memcpy( &tmp_addr, addr, sizeof( struct sockaddr ) );
						sockaddr_from_sdp( to_ipv6, &tmp_addr, addr );
					}
				}
			} else {					  /* last was TCP - so start with SDP */
				if ( FD_ISSET( shadow_fd, &fds ) ) {
					set_last_accept( fd, 1 );
					__sdp_log( 1, "ACCEPT: accepting on SDP fd:<%d>\n",
								  shadow_fd );
					ret = _socket_funcs.accept( shadow_fd, addr, addrlen );
					if ( ( ret > 0 ) && addr && addrlen ) {
						memcpy( &tmp_addr, addr, sizeof( struct sockaddr ) );
						sockaddr_from_sdp( to_ipv6, &tmp_addr, addr );
					}
				} else {
					__sdp_log( 1, "ACCEPT: accepting on TCP fd:<%d>\n", fd );
					ret = _socket_funcs.accept( fd, addr, addrlen );
				}
			}
		} else {
			__sdp_log( 9, "Error accept: select returned :<%d> %s\n",
						  ret, strerror( errno ) );
		}
	}

	__sdp_log( 2, "ACCEPT: <%s:%d> return <%d>\n",
				  program_invocation_short_name, fd, ret );

	return ret;
}										  /* accept */

/* ========================================================================= */
/*..select -- replacement socket call.                                       */
/* 
   if we have shadow we must select on it too - which requires a hack back 
   and forth
*/
int
select(
	int n,
	fd_set * readfds,
	fd_set * writefds,
	fd_set * exceptfds,
	struct timeval *timeout )
{
	int shadow_fd;
	int ret;
	int current;
	int maxi = 0;
	fd_set new_fds;

	if ( NULL == _socket_funcs.select ) {
		__sdp_log( 9, "Error select: no implementation for select found\n" );
		return -1;
	}

	__sdp_log( 2, "SELECT: <%s:%d>\n", program_invocation_short_name, n );

	/* if we do not read - nothing to do */
	if ( readfds == NULL ) {
		ret = _socket_funcs.select( n, readfds, writefds, exceptfds, timeout );
		goto done;
	}

	FD_ZERO( &new_fds );
	if ( n > 0 ) {
		maxi = n - 1;
	}

	/* add shadow bits */
	for ( current = 0; current < n; current++ ) {
		if ( FD_ISSET( current, readfds ) ) {
			FD_SET( current, &new_fds );
			if ( current > maxi ) {
				maxi = current;
			}
			shadow_fd = get_shadow_fd_by_fd( current );
			if ( shadow_fd != -1 ) {
				__sdp_log( 1, "SELECT: adding fd:<%d> shadow_fd:<%d> to readfs\n",
							  current, shadow_fd );
				FD_SET( shadow_fd, &new_fds );
				if ( shadow_fd > maxi ) {
					maxi = shadow_fd;
				}
			}
		}
	}

	__sdp_log( 1, "SELECT: invoking select n=<%d>\n", 1 + maxi );
	ret = _socket_funcs.select( 1 + maxi,
										 &new_fds, writefds, exceptfds, timeout );

	/* remove the count and bits of the shadows */
	if ( ret >= 0 ) {
		for ( current = 0; current < n; current++ ) {
			shadow_fd = get_shadow_fd_by_fd( current );
			if ( shadow_fd == -1 ) {
				if ( FD_ISSET( current, readfds ) &&
					  FD_ISSET( current, &new_fds ) == 0 ) {
					FD_CLR( current, readfds );
				}
			} else {
				if ( FD_ISSET( current, readfds ) && FD_ISSET( current, &new_fds )
					  && FD_ISSET( shadow_fd, &new_fds ) ) {
					ret -= 1;
				}
				if ( FD_ISSET( current, readfds ) &&
					  FD_ISSET( current, &new_fds ) == 0 &&
					  FD_ISSET( shadow_fd, &new_fds ) == 0 ) {
					FD_CLR( current, readfds );
				}
			}
		}
	}

 done:

	__sdp_log( 2, "SELECT: <%s:%d> return <%d>\n",
				  program_invocation_short_name, n, ret );
	return ret;
}										  /* select */

/* ========================================================================= */

/* --------------------------------------------------------------------- */
/*                                                                       */
/* Library load/unload initialization/cleanup                            */
/*                                                                       */
/* --------------------------------------------------------------------- */
/* ========================================================================= */
/*..__sdp_init -- intialize the library */
void
__sdp_init(
	void )
{
	char *config_file, *error_str;
	int fd;
	struct rlimit nofiles_limit;

	dev_null_fd = open( "/dev/null", O_WRONLY );

	/* figure out the max number of file descriptors */
	if ( getrlimit( RLIMIT_NOFILE, &nofiles_limit ) )
		max_file_descriptors = 1024;
	else
		max_file_descriptors = nofiles_limit.rlim_cur;

	/* allocate and initialize the shadow sdp sockets array */
	libsdp_fd_attributes =
		( struct sdp_extra_fd_attributes * )calloc( max_file_descriptors,
																  sizeof( struct
																			 sdp_extra_fd_attributes ) );
	for ( fd = 0; fd < max_file_descriptors; fd++ )
		init_extra_attribute( fd );

	/*
	 * FIXME: Avoid using fprintf unless explicitly configured by the user.
	 * stderr (fd 2) may have been closed by someone else already.
	 */

#ifndef RTLD_NEXT
	/*
	 * open libc for original socket call.
	 */
	__libc_dl_handle = dlopen( "/lib64/libc.so.6", RTLD_LAZY );
	if ( NULL == __libc_dl_handle ) {
		__libc_dl_handle = dlopen( "/lib/libc.so.6", RTLD_LAZY );
		if ( NULL == __libc_dl_handle ) {
			fprintf( stderr, "%s\n", dlerror(  ) );
			exit( -1 );
		}
	}
#endif

	/*
	 * Get the original functions
	 */
	_socket_funcs.ioctl = dlsym( __libc_dl_handle, "ioctl" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.fcntl = dlsym( __libc_dl_handle, "fcntl" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.socket = dlsym( __libc_dl_handle, "socket" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.setsockopt = dlsym( __libc_dl_handle, "setsockopt" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.connect = dlsym( __libc_dl_handle, "connect" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.listen = dlsym( __libc_dl_handle, "listen" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.bind = dlsym( __libc_dl_handle, "bind" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.close = dlsym( __libc_dl_handle, "close" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.dup = dlsym( __libc_dl_handle, "dup" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.dup2 = dlsym( __libc_dl_handle, "dup2" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.getpeername = dlsym( __libc_dl_handle, "getpeername" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.getsockname = dlsym( __libc_dl_handle, "getsockname" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.accept = dlsym( __libc_dl_handle, "accept" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	_socket_funcs.select = dlsym( __libc_dl_handle, "select" );
	if ( NULL != ( error_str = dlerror(  ) ) ) {
		fprintf( stderr, "%s\n", error_str );
		exit( -1 );
	}

	if ( getenv( "SIMPLE_LIBSDP" ) != NULL ) {
		simple_sdp_library = 1;
	}
	if ( getenv( "ALWAYS_USE_SDP" ) != NULL ) {
		simple_sdp_library = 1;
	}
	if ( simple_sdp_library )
		return;

	config_file = getenv( "LIBSDP_CONFIG_FILE" );
	if ( config_file ) {
		__sdp_parse_config( config_file );
	} else {
#define LIBSDP_DEFAULT_CONFIG_FILE  SYSCONFDIR "/libsdp.conf"
		__sdp_parse_config( LIBSDP_DEFAULT_CONFIG_FILE );
	}
	__sdp_log( 1, "Max file descriptors:%d\n", max_file_descriptors );
}										  /* __sdp_init */

/* ========================================================================= */
/*..__sdp_fini -- when the library is unloaded this is called */
void
__sdp_fini(
	void )
{
#ifndef RTLD_NEXT
	dlclose( __libc_dl_handle );
#endif
}										  /* _fini */
