/*
  This software is available to you under a choice of one of two
  licenses.  You may choose to be licensed under the terms of the GNU
  General Public License (GPL) Version 2, available at
  <http://www.fsf.org/copyleft/gpl.html>, or the OpenIB.org BSD
  license, available in the LICENSE.TXT file accompanying this
  software.  These details are also available at
  <http://openib.org/license.html>.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.

  Copyright (c) 2004 Topspin Communications.  All rights reserved.
  Copyright (c) 2006 Mellanox Technologies Ltd. All rights reserved.

  $Id: log.c 7596 2006-05-31 09:54:36Z eitan $
*/

/*
 * system includes
 */
#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <syslog.h>

/*
 * SDP specific includes
 */
#include "libsdp.h"

extern char *program_invocation_short_name;

typedef enum
{
	SDP_LOG_FILE,
	SDP_LOG_SYSLOG,
} __sdp_log_type_t;

/* --------------------------------------------------------------------- */
/* library static and global variables                                   */
/* --------------------------------------------------------------------- */
int __sdp_min_level = 9;
static __sdp_log_type_t __sdp_log_type = SDP_LOG_FILE;
static FILE *__sdp_log_file = NULL;

void
__sdp_log(
	int level,
	char *format,
	... )
{
	va_list ap;
	char syslog_format[512];

	if ( level < __sdp_min_level ) {
		return;
	}

	va_start( ap, format );
	switch ( __sdp_log_type ) {
	case SDP_LOG_SYSLOG:
		sprintf( syslog_format, "%s[%d] libsdp %s ",
					program_invocation_short_name, getpid(  ), format );
		vsyslog( LOG_USER | LOG_NOTICE, syslog_format, ap );
		break;
	case SDP_LOG_FILE:
		if ( __sdp_log_file == NULL ) {
			vfprintf( stderr, format, ap );
#if 0									  /* might slow everything too much? */
			( void )fflush( stderr );
#endif
		} else {
			vfprintf( __sdp_log_file, format, ap );
#if 0									  /* might slow everything too much? */
			( void )fflush( __sdp_log_file );
#endif
		}
		break;
	}
	va_end( ap );
}

void
__sdp_log_set_min_level(
	int level )
{
	__sdp_min_level = level;
}

static void
__sdp_log_set_log_type(
	__sdp_log_type_t type )
{
	if ( __sdp_log_file != NULL ) {
		fclose( __sdp_log_file );
		__sdp_log_file = NULL;
	}

	__sdp_log_type = type;
}

int
__sdp_log_set_log_stderr(
	void )
{
	__sdp_log_set_log_type( SDP_LOG_FILE );
	/* NULL means stderr */

	return 1;
}

int
__sdp_log_set_log_syslog(
	void )
{
	__sdp_log_set_log_type( SDP_LOG_SYSLOG );

	return 1;
}

int
__sdp_log_set_log_file(
	char *filename )
{
	FILE *f;
	char *p, tfilename[PATH_MAX + 1];

	/* Strip off any paths from the filename */
	p = strrchr( filename, '/' );
	if ( p ) {
		snprintf( tfilename, sizeof( tfilename ), "/tmp/%s", p + 1 );
	} else {
		snprintf( tfilename, sizeof( tfilename ), "/tmp/%s", filename );
	}

	f = fopen( tfilename, "a" );
	if ( !f ) {
		__sdp_log( 9, "Couldn't open filename '%s' for logging\n", tfilename );
		return 0;
	}

	__sdp_log_set_log_type( SDP_LOG_FILE );
	__sdp_log_file = f;

	return 1;
}
